// Kinetophone_narrator_app.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_narrator_app.hpp"

// include *mm first to avoid conflicts
#include <gtkmm/main.h>

#include "Kinetophone_narrator_config.hpp"
#include "Kinetophone_narrator_model.hpp"
#include "Kinetophone_narrator_view.hpp"
#include "Kinetophone_narrator_window.hpp"
#include "Kinetophone_narrator_controller.hpp"
#include "../base/Narration_recorder.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <exception>
#include <iostream>
#include <string>
#include <sstream>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION "(unknown)"
#endif

using std::cerr;
using std::endl;
using std::exception;
using std::string;
using std::stringstream;
using sigc::mem_fun;
using Gtk::Main;
using Roan_trail::Recorder::Narration_recorder;
using Roan_trail::Recorder::Sound_recorder_config;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//

namespace
{
  void ih_glib_exception_handler()
  {
    try
    {
      throw;
    }
    catch (const Glib::Exception& e)
    {
      cerr << "Unhandled GUI error: " << e.what() << endl;
      cerr << "Terminating." << endl;
    }
    catch (const exception& e)
    {
      cerr << "Unhandled exception: " << e.what() << endl;
      cerr << "Terminating." << endl;
    }
  }
}

//
// Constructor/destructor
//

Kinetophone_narrator_app::Kinetophone_narrator_app(int argc, const char** argv)
  : Application(argc, argv),
    m_config(new Kinetophone_narrator_config),
    m_model(0)
{
  precondition(argv);

  postcondition(mf_invariant(false));
}

Kinetophone_narrator_app::~Kinetophone_narrator_app()
{
  precondition(mf_invariant(false));

  delete m_model;
  delete m_config;
}

int Kinetophone_narrator_app::run()
{
  precondition(mf_invariant());

  int return_value = 1;
  Error_param error;
  int user_error_code = 0;

  start_error_block();

  // see what the user wants to do
  int narrator_command = m_config->parse_program_options(argc(), argv(), error);
  on_error(Kinetophone_narrator_config::command_error == narrator_command,
           new Kinetophone_error(error_location(),
                                 Kinetophone_error::command_line,
                                 error()));

  // update the installation directory if the program options override the default one
  if ("" != m_config->custom_installation_dir)
  {
    Application::application()->set_installation_dir(m_config->custom_installation_dir);
  }

  switch (narrator_command)
  {
  case Kinetophone_narrator_config::command_help:
    // HELP
    Kinetophone_narrator_view::output_console(m_config->help_message);
    break;
  case Kinetophone_narrator_config::command_list_devices:
    // LIST DEVICES
    {
      bool listed_devices = Kinetophone_narrator_view::list_devices(error);
      on_error(!listed_devices, new Kinetophone_error(error_location(),
                                                      Kinetophone_error::general,
                                                      error()));
    }
    break;
  case Kinetophone_narrator_config::command_output_version:
    // VERSION
    {
      stringstream version;
      version << "Kinetophone Narrator Version " << string(PACKAGE_VERSION) << endl;
      Kinetophone_narrator_view::output_console(version.str());
    }
    break;
  case Kinetophone_narrator_config::command_narrate:
    // NARRATE
    {
      Error_param record_error;
      bool narrate_success = mf_narrate(record_error);
      on_error(!narrate_success, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::general,
                                                       record_error()));
    }
    break;
  default:
    assert(0 && "Invalid command");
    break;
  }

  return_value =  0;
  goto exit_point;

  end_error_block();

 error_handler:
  {
    string error_message;
    user_error_code = handler_error->format_message_for_chain(m_config->detailed_error, error_message);
    Kinetophone_narrator_view::output_error(error_message);
  }
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_value = user_error_code;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return return_value;
}

void Kinetophone_narrator_app::terminate(int code, const string& message)
{
  precondition(mf_invariant());

  Narration_recorder *recorder = m_model->narration_recorder();
  if (Roan_trail::Recorder::Recorder::state_stopped != recorder->state())
  {
    Error_param e(false);
    recorder->stop(e); // ignore error
  }

  Application::terminate(code, message);

  // no postcondition, above terminate() call does not return
}

//
// Protected member functions
//

bool Kinetophone_narrator_app::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  // object allocation checking
  if (!m_config)
  {
    goto exit_point;
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Private member functions
//

void Kinetophone_narrator_app::mf_connect_signals(Kinetophone_narrator_controller& controller,
                                                  Kinetophone_narrator_view& view)
{
  // connect signals
  //   view to controller
  view.signal_retake().connect(mem_fun(controller, &Kinetophone_narrator_controller::retake));
  view.signal_slide().connect(mem_fun(controller, &Kinetophone_narrator_controller::slide));
  view.signal_previous().connect(mem_fun(controller, &Kinetophone_narrator_controller::previous));
  view.signal_next().connect(mem_fun(controller, &Kinetophone_narrator_controller::next));
  view.signal_thumbnail_choose().
    connect(mem_fun(controller, &Kinetophone_narrator_controller::thumbnail_choose));
  view.signal_mute().connect(mem_fun(controller, &Kinetophone_narrator_controller::mute));
  view.signal_record().connect(mem_fun(controller, &Kinetophone_narrator_controller::record));
  view.signal_pause().connect(mem_fun(controller, &Kinetophone_narrator_controller::pause));
  view.signal_toggle_metering().
    connect(mem_fun(controller, &Kinetophone_narrator_controller::toggle_metering));
  view.signal_reset_overflow_counts().
    connect(mem_fun(controller, &Kinetophone_narrator_controller::reset_overflow_counts));
  view.signal_reset_clipped().
    connect(mem_fun(controller, &Kinetophone_narrator_controller::reset_clipped));
  view.signal_notes_changed().
    connect(mem_fun(controller, &Kinetophone_narrator_controller::notes_changed));
  view.signal_notes_editing_done().
    connect(mem_fun(controller, &Kinetophone_narrator_controller::notes_editing_done));
  view.signal_export_session().
    connect(mem_fun(controller, &Kinetophone_narrator_controller::export_session));
  view.signal_quit().
    connect(mem_fun(controller, &Kinetophone_narrator_controller::quit));
  //   controller to view
  controller.signal_model_slides_will_update().
    connect(mem_fun(view, &Kinetophone_narrator_view::model_slides_will_update));
  controller.signal_model_recorder_will_update().
    connect(mem_fun(view, &Kinetophone_narrator_view::model_recorder_will_update));
  controller.signal_model_slides_updated().
    connect(mem_fun(view, &Kinetophone_narrator_view::model_slides_updated));
  controller.signal_model_recorder_updated().
    connect(mem_fun(view, &Kinetophone_narrator_view::model_recorder_updated));
  controller.signal_update_time().
    connect(mem_fun(view, &Kinetophone_narrator_view::update_time));
  controller.signal_update_levels().
    connect(mem_fun(view, &Kinetophone_narrator_view::update_levels));
  controller.signal_update_available_space().
    connect(mem_fun(view, &Kinetophone_narrator_view::update_available_space));
  controller.signal_update_statusbar().
    connect(mem_fun(view, &Kinetophone_narrator_view::update_statusbar));
  controller.signal_toggle_metering().
    connect(mem_fun(view, &Kinetophone_narrator_view::toggle_metering));
  controller.signal_reset_clipped().
    connect(mem_fun(view, &Kinetophone_narrator_view::reset_clipped));
  controller.signal_overwrite_recording().
    connect(mem_fun(view, &Kinetophone_narrator_view::ask_overwrite_recording));
  controller.signal_session_info().
    connect(mem_fun(view, &Kinetophone_narrator_view::show_session_info));
  controller.signal_error().
    connect(mem_fun(view, &Kinetophone_narrator_view::show_error));
}

bool Kinetophone_narrator_app::mf_narrate(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  // create the model
  delete m_model;
  m_model = new Kinetophone_narrator_model(*m_config);

  bool narration_recorder_started_up = false;
  Error_param error;
  Narration_recorder* narration_recorder = m_model->narration_recorder();

  start_error_block();

  // start the sound recorder subsystem
  narration_recorder_started_up = narration_recorder->startup(*(m_config->sound_recorder_config), error);
  on_error(!narration_recorder_started_up, new Kinetophone_error(error_location(),
                                                                 Kinetophone_error::startup,
                                                                 error()));
  // update the Kinetophone narrator config from the recorder
  // before running the GUI
  *(m_config->sound_recorder_config) = *narration_recorder->config();
  string output_file = narration_recorder->output_file();
  m_config->sound_recorder_config->sound_file->file_name = output_file;

  // start and run GUI...
  bool GUI_success = mf_run_GUI(error);
  on_error(!GUI_success, new Kinetophone_error(error_location(),
                                               Kinetophone_error::general,
                                               error()));

  // after GUI exits...

  if (!narration_recorder->have_recording())
  {
    narration_recorder->cleanup_temporary_file(); // clean up droppings
  }

  // shut down the sound recorder subsystem
  bool narration_recorder_shutdown = narration_recorder->shutdown(error);
  narration_recorder_started_up = false;
  on_error(!narration_recorder_shutdown,new Kinetophone_error(error_location(),
                                                              Kinetophone_error::shutdown,
                                                              error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (narration_recorder_started_up)
  {
    Error_param e(false);
    narration_recorder->shutdown(e); // ignore error
  }
  return_value = false;
  goto exit_point;

 exit_point:
  return return_value;
}

bool Kinetophone_narrator_app::mf_run_GUI(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;
  Error_param error;

  start_error_block();

  int argc_arg = argc();
  char** argv_arg = const_cast<char**>(argv());
  Main gtk_application(argc_arg, argv_arg);
  Glib::add_exception_handler(&ih_glib_exception_handler);

  // load the images
  const string image_source_path = m_config->image_source_path;
  const bool slides_loaded = m_model->slides()->load_from_source(image_source_path, error);
  on_error(!slides_loaded, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::source_load,
                                                 error()));

  // import the session file and update the model, if specified
  if ("" != m_config->session_file_path)
  {
    const bool session_imported = m_model->import_from_XML(m_config->session_file_path, error);
    on_error(!session_imported, new Kinetophone_error(error_location(),
                                                      Kinetophone_error::general,
                                                      error()));
  }

  Kinetophone_narrator_view view(*m_model, *m_config);
  Kinetophone_narrator_controller controller(*m_model, *m_config);

  mf_connect_signals(controller, view); // connect these before setting up view/controller

  controller.setup();
  view.setup();

  // TODO: check this:
  // Note: minimize heap allocation after this point (should be limited to error handling).
  // Ref. Standards Rule #206

  // hand event processing off to Gtk
  gtk_application.run(view.narrator_window());

  // one last check for record error
  Narration_recorder* recorder = m_model->narration_recorder();
  const Error* record_error = recorder->record_error();
  on_error(record_error, new Kinetophone_error(error_location(),
                                               Kinetophone_error::record,
                                               record_error));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}

//////////
// main //
//////////

int main (int argc, const char **argv)
{
  Kinetophone_narrator_app application(argc, argv);

  int return_code = application.run();

  return return_code;
}
