// Kinetophone_PDF_slide_collection.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.
//
// The missing image icon:
//   kinetophone_missing_image.svn
// was taken from the Gnome icon theme Dropline Neu! 0.2:
//   ./scalable/stock/gtk-missing-image.svg
//   (http://art.gnome.org/themes/icon/1100)
// which is copyright Silvestre Herrera
// (silvestre.herrera -at- gmail.com)
// licensed under the GPL version 2 or later.

#include "Kinetophone_PDF_slide_collection.hpp"

// include *mm first to avoid conflicts
#include <gdk/gdk.h>
#include <gdkmm/pixbuf.h>
#include <gtkmm/invisible.h>
#include <gtkmm/stock.h>

#include "../base/Application.hpp"
#include "../base/common.hpp"
#include "../base/Image_types.hpp"
#include "../base/PDF_slide_collection.hpp"
#include "../base/Poppler_compat.hpp"
#include "../base/Slide.hpp"
#include "../base/File_manager.hpp"
#include "../base/error/Poppler_error.hpp"
#include <poppler.h>
#include <string>
#include <vector>

using Glib::RefPtr;
using Glib::wrap;
using Gdk::Pixbuf;
using std::string;
using std::vector;
using Roan_trail::min;
using Roan_trail::max;
using Roan_trail::Error_param;
using Roan_trail::Rect_size;
using Roan_trail::Source::PDF_slide_collection;
using Roan_trail::Source::Poppler_compat;
using Roan_trail::Source::Poppler_error;
using Roan_trail::Source::Slide;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//

namespace
{
  const int ic_thumbnail_size = 64;

  RefPtr<Pixbuf> ih_broken_image()
  {
    Gtk::Invisible invisible_widget;
    return invisible_widget.render_icon(Gtk::Stock::MISSING_IMAGE, Gtk::ICON_SIZE_BUTTON);
  }

  const string ic_missing_image_file = "/kinetophone_missing_image.svg";
}

//
// Constructor/destructor
//
Kinetophone_PDF_slide_collection::Kinetophone_PDF_slide_collection()
  : PDF_slide_collection<Glib::RefPtr<Gdk::Pixbuf> >(),
    m_document(0)
{
}

Kinetophone_PDF_slide_collection::~Kinetophone_PDF_slide_collection()
{
  if (m_document)
  {
    g_object_unref(m_document);
  }
}

//
// Accessors
//

bool Kinetophone_PDF_slide_collection::image_for_slide_at(int slide_index,
                                                          const Rect_size& suggested_size,
                                                          RefPtr<Pixbuf>& return_image) const
{
  precondition(((slide_index >= 0) && (slide_index < slide_count()))
               && !return_image
               && mf_invariant());

  bool return_value = false;

  PopplerPage* page = poppler_document_get_page(m_document, slide_index);

  if (!page)
  {
    const string& installation_dir = Application::application()->installation_dir();
    try
    {
      if ((suggested_size.width <= 0)
          || (suggested_size.height <= 0))
      {
        return_image = Pixbuf::create_from_file(installation_dir + ic_missing_image_file);
      }
      else
      {
        const int max_dimension = max(suggested_size.width, suggested_size.height);
        return_image = Pixbuf::create_from_file(installation_dir + ic_missing_image_file,
                                                max_dimension,
                                                max_dimension);
      }
    }
    catch (...)
    {
      return_image = ih_broken_image();
    }
  }
  else
  {
    double width;
    double height;
    // Note: ignore suggested width and height, use the PDF page size
    // get width/height in pixels
    poppler_page_get_size(page,
                          &width,
                          &height);
    const int image_width = width;
    const int image_height = height;
    GdkPixbuf* pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB,
                                       FALSE,
                                       8,
                                       image_width,
                                       image_height);
    const Roan_trail::Color& background_color = fill_color();
    uint32_t scalar_color = background_color.scalar_value();
    // the fill requires a 32 bit value, adjust if necessary
    if (!background_color.has_alpha)
    {
      scalar_color <<= 8;
    }
    gdk_pixbuf_fill(pixbuf, scalar_color);
    Poppler_compat::poppler_page_render_to_pixbuf(page,
                                                  0,
                                                  0,
                                                  width,
                                                  height,
                                                  1.0,
                                                  0,
                                                  pixbuf);
    return_image = wrap(pixbuf, TRUE);

    g_object_unref(page);
  }

  return_value = true;

  postcondition(return_image
                && mf_invariant());

  return return_value;
}

void Kinetophone_PDF_slide_collection::thumbnail_for_slide_at(int slide_index,
                                                              RefPtr<Pixbuf>& return_image) const
{
  precondition(((slide_index >= 0) && (slide_index < slide_count()))
               && !return_image
               && mf_invariant());

  PopplerPage* page = poppler_document_get_page(m_document, slide_index);
  if (!page)
  {
    try
    {
      const string& installation_dir = Application::application()->installation_dir();
      {
        return_image = Pixbuf::create_from_file(installation_dir + ic_missing_image_file,
                                                ic_thumbnail_size,
                                                ic_thumbnail_size);
      }
    }
    catch (...)
    {
      return_image = ih_broken_image();
    }
  }
  else
  {
    GdkPixbuf* pixbuf = Poppler_compat::poppler_page_get_thumbnail_pixbuf(page, fill_color());
    if (!pixbuf)
    {
      double width;
      double height;
      poppler_page_get_size(page,
                            &width,
                            &height);
      const double width_scale = double(ic_thumbnail_size) / width;
      const double height_scale = double(ic_thumbnail_size) / height;
      const double scale = min(width_scale, height_scale);
      const int image_width = width * scale;
      const int image_height = height * scale;
      pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB,
                              FALSE,
                              8,
                              image_width,
                              image_height);
      const Roan_trail::Color& background_color = fill_color();
      uint32_t scalar_color = background_color.scalar_value();
      // the fill requires a 32 bit value, adjust if necessary
      if (!background_color.has_alpha)
      {
        scalar_color <<= 8;
      }
      gdk_pixbuf_fill(pixbuf, scalar_color);
      Poppler_compat::poppler_page_render_to_pixbuf(page,
                                                    0,
                                                    0,
                                                    image_width,
                                                    image_height,
                                                    scale,
                                                    0,
                                                    pixbuf);
    }

    g_object_unref(page);

    return_image = wrap(pixbuf, TRUE);
  }

  postcondition(return_image
                && mf_invariant());
}

Rect_size Kinetophone_PDF_slide_collection::max_bounds() const
{
  precondition(mf_invariant());

  Rect_size bounds(1, 1);
  for (int slide_index = 0; slide_index < slide_count(); ++slide_index)
  {
    PopplerPage* page = poppler_document_get_page(m_document, slide_index);
    if (page)
    {
      double width;
      double height;
      poppler_page_get_size(page,
                            &width,
                            &height);
      const int image_width = width * resolution();
      if (image_width > bounds.width)
      {
        bounds.width = image_width;
      }
      const int image_height = height * resolution();
      if (image_height > bounds.height)
      {
        bounds.height = image_height;
      }

      g_object_unref(page);

    }
  }

  return bounds;
}

//
// Protected member functions
//

bool Kinetophone_PDF_slide_collection::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  return_value = !check_base_class
    || PDF_slide_collection<RefPtr<Pixbuf> >::mf_invariant(check_base_class);

  goto exit_point;

 exit_point:
  return return_value;
}

// implement virtual function for specific image (Gdk::Pixbuf in this case)
bool Kinetophone_PDF_slide_collection::mf_process_source_with_notes(const string& source_path,
                                                                    Error_param& return_error)
{
  precondition(!m_document
               && !return_error());

  bool return_value = false;

  start_error_block();

  Error_param error;
  const bool document_loaded = mf_load_document(source_path, error);
  on_error(!document_loaded, new Source_error(error_location(),
                                              Source_error::general,
                                              error()));

  const bool import_notes = import_PDF_notes();
  const int page_count = poppler_document_get_n_pages(m_document);
  for (int page_index = 0; page_index < page_count; ++page_index)
  {
    string note;
    PopplerPage* page = poppler_document_get_page(m_document, page_index);
    if (page)
    {
      if (import_notes)
      {
        GList* annotations = g_list_first(poppler_page_get_annot_mapping(page));
        while (annotations)
        {
          const PopplerAnnotMapping* annotation_mapping = static_cast<PopplerAnnotMapping*>(annotations->data);
          PopplerAnnot* annotation = annotation_mapping->annot;

          const PopplerAnnotType annotation_type = poppler_annot_get_annot_type(annotation);
          if (POPPLER_ANNOT_TEXT == annotation_type)
          {
            gchar* text = poppler_annot_get_contents(annotation);
            if (text)
            {
              note += text;
            }
          }
          annotations = annotations->next;
        }
        poppler_page_free_annot_mapping(annotations);
      }
      m_slides.push_back(new Slide<RefPtr<Pixbuf> >(page_index,
                                                    note,
                                                    ""));
      g_object_unref(page);
    }
  }

  // make sure we have at least one page
  const bool no_pages = (m_slides.size() == 0);
  on_error(no_pages, new Source_error(error_location(),
                                      Source_error::invalid_source,
                                      string("Source PDF contains no pages: ")
                                      + source_path));

  return_value = true;

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (m_document)
  {
    g_object_unref(m_document);
    m_document = 0;
  }

  return_value = false;

  goto exit_point;

 exit_point:
  postcondition(((return_value && m_document)
                 || (!return_value && !m_document))
                && return_error.is_valid_at_return(return_value));
  return return_value;
}


// implement virtual function for specific PDF document class
bool Kinetophone_PDF_slide_collection::mf_load_document(const string& source_path,
                                                        Error_param& return_error)
{
  precondition(!m_document);

  GFile* file = 0;
  gchar* URI = 0;
  bool return_value = false;

  start_error_block();

  file = g_file_new_for_commandline_arg(source_path.c_str());
  if (file)
  {
    URI = g_file_get_uri(file);
  }
  on_error((!file || !URI), new Source_error(error_location(),
                                             Source_error::invalid_source,
                                             "Could not open PDF source.",
                                             source_path));

  GError* poppler_error = 0;

  PopplerDocument* document = poppler_document_new_from_file(URI,
                                                             0, // passwords not supported
                                                             &poppler_error);
  on_error(!document, new Source_error(error_location(),
                                       Source_error::invalid_source,
                                       new Poppler_error(error_location(),
                                                         source_path,
                                                         poppler_error)));

  // hide markup annotations for all pages
  if (Poppler_compat::markup_annotations_are_rendered())
  {
    const int page_count = poppler_document_get_n_pages(document);
    for (int page_index = 0; page_index < page_count; ++page_index)
    {
      PopplerPage* page = poppler_document_get_page(document, page_index);
      if (page)
      {
        GList* annotations = g_list_first(poppler_page_get_annot_mapping(page));
        while (annotations)
        {
          const PopplerAnnotMapping* annotation_mapping = static_cast<PopplerAnnotMapping*>(annotations->data);
          PopplerAnnot* annotation = annotation_mapping->annot;

          // hide if a markup annotation
          if (POPPLER_IS_ANNOT_MARKUP(annotation))
          {
            Poppler_compat::hide_markup_annotation(POPPLER_ANNOT_MARKUP(annotation));
          }

          annotations = annotations->next;
        }
        poppler_page_free_annot_mapping(annotations);
      }
      g_object_unref(page);
    }
  }

  m_document = document;

  return_value = true;

  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  if (URI)
  {
    g_free(URI);
  }
  if (file)
  {
    g_object_unref(file);
  }
  postcondition(((return_value && m_document)
                 || (!return_value && !m_document))
                && return_error.is_valid_at_return(return_value));
  return return_value;
}
