// Kinetophone_dbus_recorder.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_dbus_recorder.hpp"
#include "Kinetophone_dbus_server_glue.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/Device_enum.hpp"
#include "../base/Segment.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include "../base/error/Record_error.hpp"
#include <libxml++/document.h>
#include <libxml++/exceptions/exception.h>
#include <libxml++/nodes/node.h>
#include <libxml++/parsers/domparser.h>
#include <boost/lexical_cast.hpp>
#include <dbus-c++/dbus.h>
#include <map>
#include <string>
#include <vector>

using DBus::Connection;
using xmlpp::Document;
using xmlpp::Node;
using std::endl;
using std::string;
using std::map;
using std::vector;
using Roan_trail::Long_int;
using Roan_trail::Error;
using Roan_trail::Recorder::Segment;
using Roan_trail::Recorder::Record_error;
using Roan_trail::Recorder::Sound_recorder_config;
using Roan_trail::Recorder::Device_enum;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//

namespace
{
  const string ic_kinetophone_dbus_recorder_name = "com.roantrail.dbus.kinetophone.recorder";
  const string ic_kinetophone_dbus_recorder_path = "/com/roantrail/dbus/kinetophone/recorder";
}

//
// Constructor, destructor
//

Kinetophone_dbus_recorder::Kinetophone_dbus_recorder(Connection& connection, const string& path)
  : ObjectAdaptor(connection, path.c_str()),
    m_channels(0), m_sound_levels()
{
  postcondition(mf_invariant(false));
}

//
// Control
//

void Kinetophone_dbus_recorder::remove(bool& return_have_error, Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;

  const bool have_shutdown = Narration_recorder::shutdown(error);
  on_error(!have_shutdown, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::shutdown,
                                                 error()));

  m_channels = 0;

  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

//
// Helpers
//

const string& Kinetophone_dbus_recorder::dbus_name()
{
  return ic_kinetophone_dbus_recorder_name;
}

string Kinetophone_dbus_recorder::dbus_path(Long_int ID)
{
  return ic_kinetophone_dbus_recorder_path + "/" + boost::lexical_cast<string>(ID);
}

//
// DBUS interface
//

//
//   control (messages)
//

//
//     ...startup/shutdown functions
//

void Kinetophone_dbus_recorder::startup(const map<string, string>& settings,
                                        bool& return_have_error,
                                        Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Sound_recorder_config config;
  config.from_map(settings);
  Error_param error;
  const bool have_startup = Narration_recorder::startup(config, error);
  on_error(!have_startup, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::startup,
                                                error()));

  m_channels = Narration_recorder::config()->sound_file->channels;
  m_sound_levels.resize(m_channels);

  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition((return_have_error || Narration_recorder::is_started())
                && mf_invariant());
}

//
//     ...basic Recorder functions
//

void Kinetophone_dbus_recorder::record(bool& return_have_error,
                                       Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;
  const bool recording = Narration_recorder::record(error);
  on_error(!recording, new Kinetophone_error(error_location(),
                                             Kinetophone_error::record,
                                             error()));
  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::pause(bool& return_have_error, Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;
  const bool paused = Narration_recorder::pause(error);
  on_error(!paused, new Kinetophone_error(error_location(),
                                          Kinetophone_error::record,
                                          error()));
  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::stop(bool& return_have_error, Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;
  const bool stopped = Narration_recorder::stop(error);
  on_error(!stopped, new Kinetophone_error(error_location(),
                                           Kinetophone_error::record,
                                           error()));
  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

//
//   ...narration/sound recorder functions
//

void Kinetophone_dbus_recorder::record_with_index(const int64_t& index,
                                                  bool& return_have_error,
                                                  Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;
  const bool recording = Narration_recorder::record(static_cast<Long_int>(index), error);
  on_error(!recording, new Kinetophone_error(error_location(),
                                             Kinetophone_error::record,
                                             error()));
  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::new_index(const int64_t& index,
                                          bool& return_have_error,
                                          Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;
  const bool new_index_set = Narration_recorder::new_index(index, error);
  on_error(!new_index_set, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::record,
                                                 error()));
  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::retake(bool& return_have_error, Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;
  const bool have_retake = Narration_recorder::retake(error);
  on_error(!have_retake, new Kinetophone_error(error_location(),
                                               Kinetophone_error::record,
                                               error()));
  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::mute(bool& return_have_error, Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;
  const bool muted = Narration_recorder::mute(error);
  on_error(!muted, new Kinetophone_error(error_location(),
                                         Kinetophone_error::record,
                                         error()));
  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
}

//
//   accessors
//

bool Kinetophone_dbus_recorder::is_muted()
{
  precondition(mf_invariant());

  return Narration_recorder::is_muted();
}

bool Kinetophone_dbus_recorder::is_started()
{
  precondition(mf_invariant());

  return Narration_recorder::is_started();
}

bool Kinetophone_dbus_recorder::can_record()
{
  precondition(mf_invariant());

  return Narration_recorder::can_record();
}

string Kinetophone_dbus_recorder::recorded_time(const uint8_t& separator)
{
  precondition(mf_invariant());

  string rec_time;
  Narration_recorder::recorded_time(rec_time, separator);

  return rec_time;
}

string Kinetophone_dbus_recorder::recorded_time_for_current_index(const uint8_t& separator)
{
  precondition(mf_invariant());

  string rec_time;
  Narration_recorder::recorded_time_for_current_index(rec_time, separator);

  return rec_time;
}

vector<double> Kinetophone_dbus_recorder::sound_levels_RMS()
{
  precondition(mf_invariant());

  Narration_recorder::sound_levels_RMS(m_sound_levels);

  return m_sound_levels;
}

vector<double> Kinetophone_dbus_recorder::sound_levels_peak()
{
  precondition(mf_invariant());

  Narration_recorder::sound_levels_peak(m_sound_levels);

  return m_sound_levels;
}

void Kinetophone_dbus_recorder::frames(int64_t& return_total_frames,
                                       int64_t& return_index_frames,
                                       bool& return_have_error,
                                       Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;

  Long_int total_frames = 0;
  Long_int index_frames = 0;

  const bool got_frames = Narration_recorder::frames(total_frames,
                                                     index_frames,
                                                     error);
  on_error(!got_frames, new Kinetophone_error(error_location(),
                                              Kinetophone_error::general,
                                              "Could not get frames.",
                                              error()));

  return_total_frames = static_cast<int64_t>(total_frames);
  return_index_frames = static_cast<int64_t>(index_frames);

  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  return;
}

void Kinetophone_dbus_recorder::segments(string& return_segments_XML,
                                         bool& return_have_error,
                                         Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  string recorder_XML_representation;
  Error_param error;

  Document kinetophone_document;
  kinetophone_document.create_root_node("kinetophone_dbus_model");
  Node* XML_root_node = kinetophone_document.get_root_node();
  assert(XML_root_node && "error getting root node from kinetophone DBus XML document");

  const bool got_XML_representation = export_as_XML("<internal>",
                                                    XML_root_node,
                                                    error);
  on_error(!got_XML_representation, new Kinetophone_error(error_location(),
                                                          Kinetophone_error::general,
                                                          error()));

  return_segments_XML = kinetophone_document.write_to_string_formatted(); // UTF-8 encoding by default

  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  return;
}

void Kinetophone_dbus_recorder::settings(map<string, string>& return_settings,
                                         bool& return_have_error,
                                         Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  const bool have_map = Narration_recorder::config()->to_map(return_settings);
  on_error(!have_map, new Kinetophone_error(error_location(),
                                            Kinetophone_error::server,
                                            "Could not get recorder settings."));

  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  return;
}

string Kinetophone_dbus_recorder::output_file()
{
  precondition(mf_invariant());

  return Narration_recorder::output_file();
}

bool Kinetophone_dbus_recorder::have_recording()
{
  precondition(mf_invariant());

  return Narration_recorder::have_recording();
}

void Kinetophone_dbus_recorder::record_error(bool& return_have_record_error, Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  const Error* current_record_error = Narration_recorder::record_error();
  on_error(current_record_error, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::record,
                                                       current_record_error));

  return_have_record_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_record_error = true;
  goto exit_point;

 exit_point:
  return;
}

int64_t Kinetophone_dbus_recorder::overflow_count()
{
  precondition(mf_invariant());

  return Narration_recorder::overflow_count();
}

int64_t Kinetophone_dbus_recorder::input_overflow_count()
{
  precondition(mf_invariant());

  return Narration_recorder::input_overflow_count();
}

int64_t Kinetophone_dbus_recorder::output_overflow_count()
{
  precondition(mf_invariant());

  return Narration_recorder::output_overflow_count();
}

void Kinetophone_dbus_recorder::space_available(int64_t& return_bytes_available,
                                                double& return_fraction_available,
                                                double& return_seconds_remaining,
                                                bool& return_have_error)
{
  precondition(mf_invariant());

  Long_int bytes_available;
  double fraction_available;
  double seconds_remaining;
  return_have_error = Narration_recorder::space_available(bytes_available,
                                                          fraction_available,
                                                          seconds_remaining);
  if (!return_have_error)
  {
    return_bytes_available = bytes_available;
    return_fraction_available = fraction_available;
    return_seconds_remaining = seconds_remaining;
  }
}

void Kinetophone_dbus_recorder::output_file_size(int64_t& return_file_size,
                                                 double& return_data_rate,
                                                 bool& return_have_error)
{
  precondition(mf_invariant());

  Long_int file_size;
  double data_rate;
  return_have_error = Narration_recorder::output_file_size(file_size, data_rate);
  if (!return_have_error)
  {
    return_file_size = file_size;
    return_data_rate = data_rate;
  }
}

bool Kinetophone_dbus_recorder::is_RMS_metering_enabled()
{
  precondition(mf_invariant());

  return Narration_recorder::is_RMS_metering_enabled();
}

bool Kinetophone_dbus_recorder::is_peak_metering_enabled()
{
  precondition(mf_invariant());

  return Narration_recorder::is_peak_metering_enabled();
}

int Kinetophone_dbus_recorder::metering_channels()
{
  precondition(mf_invariant());

  return Narration_recorder::metering_channels();
}

// convenience method for display updates, gets commonly updated values
// in one DBus message, rather than separate ones for each parameter
void Kinetophone_dbus_recorder::values_for_update(string& return_recorded_time,
                                                  string& return_recorded_time_for_current_index,
                                                  vector<double>& return_sound_levels,
                                                  int64_t& return_overflow_count,
                                                  bool& return_have_record_error,
                                                  Error_dictionary& return_record_error)
{
  precondition(mf_invariant());

  start_error_block();

  Narration_recorder::recorded_time(return_recorded_time);
  Narration_recorder::recorded_time(return_recorded_time_for_current_index);
  return_sound_levels.resize(m_channels);
  // default to RMS metering
  if (Narration_recorder::is_RMS_metering_enabled())
  {
    Narration_recorder::sound_levels_RMS(return_sound_levels);
  } else if (Narration_recorder::is_peak_metering_enabled())
  {
    Narration_recorder::sound_levels_peak(return_sound_levels);
  }
  // no final else needed, if neither RMS nor peak is enabled do nothing

  return_overflow_count = static_cast<int64_t>(Narration_recorder::overflow_count());

  const Error *record_error = Narration_recorder::record_error();
  on_error(record_error, new Kinetophone_error(error_location(),
                                               Kinetophone_error::record,
                                               record_error));

  return_have_record_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_record_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_record_error = true;
  goto exit_point;

 exit_point:
  return;
}

//
// Mutators
//

void Kinetophone_dbus_recorder::reset_overflow_counters()
{
  precondition(mf_invariant());

  Narration_recorder::reset_overflow_count();

  postcondition(mf_invariant());
}


void Kinetophone_dbus_recorder::enable_RMS_metering(const bool& enable)
{
  precondition(mf_invariant());

  Narration_recorder::enable_RMS_metering(enable);

  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::enable_peak_metering(const bool& enable)
{
  precondition(mf_invariant());

  Narration_recorder::enable_peak_metering(enable);

  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::set_metering_channels(const int& channel_count)
{
  precondition(mf_invariant());

  Narration_recorder::set_metering_channels(channel_count);

  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::cleanup_temporary_file()
{
  precondition(mf_invariant());

  Narration_recorder::cleanup_temporary_file();

  postcondition(mf_invariant());
}

void Kinetophone_dbus_recorder::clear_record_error()
{
  precondition(mf_invariant());

  Narration_recorder::clear_record_error();

  postcondition(mf_invariant());
}

//
//   commands
//

void Kinetophone_dbus_recorder::list_devices(string& return_devices,
                                             bool& return_have_error,
                                             Error_dictionary& return_error)
{
  precondition(mf_invariant());

  start_error_block();

  Error_param error;
  // Note: paFloat32 declaration uses an old-style cast
  const bool got_info = Device_enum::list_devices(paFloat32,
                                                  return_devices,
                                                  error);
  on_error(!got_info, new Kinetophone_error(error_location(),
                                            Kinetophone_error::list_devices,
                                            error()));

  return_have_error = false;
  goto exit_point;

  end_error_block();

 error_handler:
  handler_error->format_chain_as_dictionary(return_error);
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_have_error = true;
  goto exit_point;

 exit_point:
  return;
}

//
// Protected member functions
//

bool Kinetophone_dbus_recorder::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (m_channels < 0)
  {
    goto exit_point;
  }

  return_value = (!check_base_class || Narration_recorder::mf_invariant(check_base_class));
  goto exit_point;

 exit_point:
  return return_value;
}
