// Console_app.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Console_app.hpp"
#include "../base/common.hpp"
#include "../base/error/Curses_error.hpp"
#include <string>
#include <cstdlib>
#include <ncurses.h>

using std::string;
using namespace Roan_trail;

//
// Constructor/destructor
//

Console_app::Console_app(int argc, const char** argv)
: Application(argc, argv)
{
  precondition(argv);

  postcondition(mf_invariant(false));
}

Console_app::~Console_app()
{
  precondition(mf_invariant(false));

  Error_param e(false);
  set_full_screen(false, e); // ignore error
}

//
// Mutators
//

bool Console_app::set_full_screen(bool use_full_screen, Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  if (use_full_screen && !m_full_screen)
  {
    // ncurses initialization
    const char *term_name = getenv("TERM"); // Standards Rule 24 deviation (check_code_ignore)
    const char *name = ((term_name == 0) || (*term_name == '\0'))
      ? "unknown"
      : term_name;
    m_screen = newterm(const_cast<char*>(name), stdout, stdin);
    on_error(!m_screen, new Curses_error("newterm"));
    int curses_err;
    curses_err = def_prog_mode();
    on_error(ERR == curses_err, new Curses_error("def_prog_mode"));
    curses_err = raw(); // no line buffering
    on_error(ERR == curses_err, new Curses_error("raw"));
    curses_err = noecho(); // don't output key presses
    on_error(ERR == curses_err, new Curses_error("noecho"));
    timeout(0);  // non-blocking input
    curses_err = curs_set(0); // turn cursor off
    on_error(ERR == curses_err, new Curses_error("curs_set"));
  }
  else if (!use_full_screen && m_full_screen)
  {
    endwin(); // end
    if (m_screen)
    {
      delscreen(m_screen);
      m_screen = 0;
    }
  }
  // no final else needed, we don't need to change modes

  m_full_screen = use_full_screen;

  return_value = true;
  goto exit_point;

  end_error_block();

 error_handler:
  if (return_error.need_error())
  {
    return_error = handler_error;
  }
  endwin(); // clean up
  if (m_screen)
  {
    delscreen(m_screen);
    m_screen = 0;
  }
  m_full_screen = false;
  goto error_cleanup;

  default_error_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Other
//

void Console_app::terminate(int code, const string& message)
{
  precondition(mf_invariant());

  Error_param e(false);
  Console_app::set_full_screen(false, e); // ignore error

  Application::terminate(code, message);

  // no postcondition, above terminate() call does not return
}


//
// Protected member functions
//

bool Console_app::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  return true;
}

//
// Class members
//

bool Console_app::m_full_screen = false;
SCREEN* Console_app::m_screen = 0;
