// Sound_builder.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef SOUND_BUILDER_HPP_
#define SOUND_BUILDER_HPP_

#include "common.hpp"
#include "Movie_types.hpp"
#include "Sound_source.hpp"
#include <string>

typedef struct SNDFILE_tag SNDFILE;

namespace Roan_trail
{
  class Error_param;

  namespace Recorder
  {
    struct Sound_file_config;
  }

  namespace Builder
  {
    class Sound_builder : public Sound_source
    {
    public:
      // constructor/destructor/initialization
      Sound_builder();
      virtual ~Sound_builder();
      // build functions
      bool start(const std::string& sound_file_path, Error_param& return_error);
      bool end(Error_param& return_error);
      // sound source functions
      //   accessors
      virtual const Roan_trail::Recorder::Sound_file_config& config() const { return *m_config; }
      virtual int frame_size() const { return m_frame_size; }
      //   sound source
      virtual const uint8_t* sound(Long_int start_frame,
                                   Long_int request_frame_count,
                                   Long_int& return_frame_count,
                                   Error_param& error_param);
      virtual const uint8_t* silence(Long_int request_frame_count,
                                     Long_int& return_frame_count,
                                     Error_param& error_param);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      Roan_trail::Recorder::Sound_file_config* m_config;
      bool m_started;
      SNDFILE* m_src_sound_file;
      int m_frame_size;
      uint8_t* m_buffer;
      Long_int m_buffer_size_in_frames;
      uint8_t* m_silence_buffer;
      Long_int m_silence_buffer_size_in_frames;
      typedef Long_int (*Read_function)(SNDFILE* sound_file,
                                        uint8_t* buffer,
                                        int frame_size,
                                        Long_int frames_to_read);
      Read_function m_read_function;
      // read functions
      static Long_int mf_read_raw(SNDFILE* sound_file,
                                  uint8_t* buffer,
                                  int frame_size,
                                  Long_int frames_to_read);
      static Long_int mf_read_short(SNDFILE* sound_file,
                                    uint8_t* buffer,
                                    int frame_size,
                                    Long_int frames_to_read);
      static Long_int mf_read_int(SNDFILE* sound_file,
                                  uint8_t* buffer,
                                  int frame_size,
                                  Long_int frames_to_read);
      static Long_int mf_read_float(SNDFILE* sound_file,
                                    uint8_t* buffer,
                                    int frame_size,
                                    Long_int frames_to_read);
      static Long_int mf_read_double(SNDFILE* sound_file,
                                     uint8_t* buffer,
                                     int frame_size,
                                     Long_int frames_to_read);
      void mf_set_read_function(Read_function& return_function,
                                Roan_trail::Recorder::Sound_file_config& return_config);
      // prevent compiler from generating
      Sound_builder(const Sound_builder& b);
      Sound_builder& operator=(const Sound_builder& b);
    };
  }
}

#endif // SOUND_BUILDER_HPP_
