// Slide.hpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef SLIDE_HPP_
#define SLIDE_HPP_

#include "common.hpp"
#include "error/Error.hpp"
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <string>

namespace Roan_trail
{
  namespace Source
  {
    template <class Image> class Slide
    {
      BOOST_CONCEPT_ASSERT((boost::Assignable<Image>));
      BOOST_CONCEPT_ASSERT((boost::CopyConstructible<Image>));
    public:
      // constructor/destructor
      Slide(Long_int index,
            const std::string& notes,
            const std::string& path);
      virtual ~Slide() {}
      // accessors
      virtual const std::string& notes() const { return m_notes; }
      virtual const std::string& path() const { return m_path; } // can be a file or structure path
      virtual int index() const { return m_index; }
      // mutators
      virtual void set_notes(const std::string& notes) { m_notes = notes; }
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      Long_int m_index;
      std::string m_notes;
      std::string m_path;
      // prevent compiler from generating
      Slide(const Slide& s);
      Slide& operator=(const Slide& s);
    };

    //
    // Template implementation
    //

    //
    // Constructor/destructor
    //

    template <class Image> Slide<Image>::Slide(Long_int index,
                                               const std::string& notes,
                                               const std::string& path)
      : m_index(index),
        m_notes(notes),
        m_path(path)
    {
      precondition((index >= 0));

      postcondition((m_index == index)
                    && (m_notes == notes)
                    && (m_path == path)
                    && mf_invariant(false));
    }

    //
    // Protected member functions
    //

    template <class Image> bool Slide<Image>::mf_invariant(bool check_base_class) const
    {
      static_cast<void>(check_base_class); // avoid unused warning

      bool return_value = false;

      if ((m_index < 0))
      {
        goto exit_point;
      }

      return_value = true;

      goto exit_point;

    exit_point:
      return return_value;
    }
  }
}

#endif // SLIDE_HPP_
