// Poppler_compat.cpp
//
// Copyright 2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.
//
// The replacement functions for poppler_page_render_to_pixbuf,
// poppler_page_get_thumbnail_pixbuf, and copy_cairo_surface_to_pixbuf
// (removed in Poppler 0.17.0) are derived from the ones in
// ./glib/poppler-page.cc (Poppler 0.16.7) under the following
// license:
//
//   poppler-page.cc: glib wrapper for poppler
//   Copyright (C) 2005, Red Hat, Inc.
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2, or (at your option)
//   any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 51 Franklin Street - Fifth Floor, Boston, MA 02110-1301, USA.

#include "Poppler_compat.hpp"

#if POPPLER_CHECK_VERSION(0, 17, 0)
//
#include "Image_types.hpp"
#include <poppler.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

using Roan_trail::Color;
using namespace Roan_trail::Source;
//
#endif

#if POPPLER_CHECK_VERSION(0, 17, 0)
//
namespace
{
  // adapted from Poppler 0.16.7:
  void ih_copy_cairo_surface_to_pixbuf(cairo_surface_t *surface, GdkPixbuf *pixbuf)
  {
    int cairo_width = cairo_image_surface_get_width(surface);
    int cairo_height = cairo_image_surface_get_height(surface);
    const int cairo_rowstride = cairo_image_surface_get_stride(surface);
    unsigned char* cairo_data = cairo_image_surface_get_data(surface);

    const unsigned char* pixbuf_data = gdk_pixbuf_get_pixels(pixbuf);
    const int pixbuf_rowstride = gdk_pixbuf_get_rowstride(pixbuf);
    const int pixbuf_n_channels = gdk_pixbuf_get_n_channels(pixbuf);

    if (cairo_width > gdk_pixbuf_get_width(pixbuf))
    {
      cairo_width = gdk_pixbuf_get_width(pixbuf);
    }
    if (cairo_height > gdk_pixbuf_get_height(pixbuf))
    {
      cairo_height = gdk_pixbuf_get_height(pixbuf);
    }

    for (int y = 0; y < cairo_height; y++)
    {
      unsigned int* src = reinterpret_cast<unsigned int *>(cairo_data + y * cairo_rowstride);
      unsigned char* dst = const_cast<unsigned char*>(pixbuf_data) + y * pixbuf_rowstride;
      for (int x = 0; x < cairo_width; x++)
      {
        dst[0] = (*src >> 16) & 0xff;
        dst[1] = (*src >> 8) & 0xff;
        dst[2] = (*src >> 0) & 0xff;
        if (pixbuf_n_channels == 4)
        {
          dst[3] = (*src >> 24) & 0xff;
        }
        dst += pixbuf_n_channels;
        src++;
      }
    }
  }
}
#endif

#if POPPLER_CHECK_VERSION(0, 17, 0)
// replaces the version from Poppler 0.16.7:
GdkPixbuf* Poppler_compat::poppler_page_get_thumbnail_pixbuf(PopplerPage *page,
                                                             const Roan_trail::Color& background_color)
{
  GdkPixbuf* return_value = 0;
  g_return_val_if_fail(POPPLER_IS_PAGE(page), 0);

  cairo_surface_t* surface = poppler_page_get_thumbnail(page);
  if (!surface)
  {
    goto exit_point;
  }

  int width;
  int height;
  if (!poppler_page_get_thumbnail_size(page,
                                       &width,
                                       &height))
  {
    goto exit_point;
  }
  else
  {
    GdkPixbuf* pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB,
                                       FALSE,
                                       8,
                                       width,
                                       height);
    uint32_t scalar_color = background_color.scalar_value();
    // the fill requires a 32 bit value, adjust if necessary
    if (!background_color.has_alpha)
    {
      scalar_color <<= 8;
    }
    gdk_pixbuf_fill(pixbuf, scalar_color);

    ih_copy_cairo_surface_to_pixbuf(surface, pixbuf);

    return_value = pixbuf;
  }

 exit_point:
  return return_value;
}

// adapted from Poppler 0.16.7:
void Poppler_compat::poppler_page_render_to_pixbuf(PopplerPage *page,
                                                   int src_x,
                                                   int src_y,
                                                   int src_width,
                                                   int src_height,
                                                   double scale,
                                                   int rotation,
                                                   GdkPixbuf *pixbuf)
{
  cairo_surface_t *surface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32,
                                                        src_width,
                                                        src_height);
  cairo_t* cr = cairo_create(surface);
  cairo_save(cr);
  switch (rotation)
  {
  case 90:
    cairo_translate(cr, src_x + src_width, -src_y);
    break;
  case 180:
    cairo_translate(cr, src_x + src_width, src_y + src_height);
    break;
  case 270:
    cairo_translate(cr, -src_x, src_y + src_height);
    break;
  default:
    cairo_translate(cr, -src_x, -src_y);
    break;
  }

  if (scale != 1.0)
  {
    cairo_scale(cr, scale, scale);
  }
  if (rotation != 0)
  {
    cairo_rotate(cr, rotation * G_PI / 180.0);
  }
  poppler_page_render(page, cr);
  cairo_restore(cr);

  cairo_set_operator(cr, CAIRO_OPERATOR_DEST_OVER);
  cairo_set_source_rgb(cr,
                       1.0,
                       1.0,
                       1.0);
  cairo_paint(cr);

  cairo_destroy(cr);

  ih_copy_cairo_surface_to_pixbuf(surface, pixbuf);
  cairo_surface_destroy(surface);
}
//
#endif
