// PDF_slide_collection.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// Abstract base, PDF slide collection class

#ifndef PDF_SLIDE_COLLECTION_HPP_
#define PDF_SLIDE_COLLECTION_HPP_

#include "Slide_collection.hpp"
#include "File_manager.hpp"
#include "error/Source_error.hpp"
#include <libxml++/nodes/element.h>
#include <libxml++/nodes/node.h>
#include <sstream>
#include <string>
#include <vector>
#include <algorithm>
#include <cstddef>

namespace Roan_trail
{
  namespace Source
  {
    template <class Image> class PDF_slide_collection : public Slide_collection<Image>
    {
    public:
      // accessors/mutators
      std::string source_type() const { return "PDF"; }
      double resolution() const { return m_resolution; }
      void set_resolution(double resolution) { m_resolution = resolution; }
      bool import_PDF_notes() const { return m_import_PDF_notes; }
      void set_import_PDF_notes(bool import_notes) { m_import_PDF_notes = import_notes; }
      const Roan_trail::Color& fill_color() const { return *m_fill_color; }
      void set_fill_color(const Roan_trail::Color& color) { *m_fill_color = color; }
      // load/save
      //   using source
      virtual bool load_from_source(const std::string& path, Error_param& return_error);
      //   load whole collection from XML
      virtual bool load_from_XML(const std::string& file_path,
                                 const xmlpp::Node* XML_node,
                                 const std::string& custom_source_path,
                                 Error_param& return_error);
      //   save whole collection to XML
      virtual bool save_as_XML(const std::string& file_path,
                               xmlpp::Node* XML_node,
                               Error_param& return_error);
    protected:
      PDF_slide_collection();
      virtual ~PDF_slide_collection();
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
      //
      virtual bool mf_process_source_with_notes(const std::string& source_path, Error_param& return_error) = 0;
      virtual bool mf_load_document(const std::string& source_path, Error_param& return_error) = 0;
    private:
      Roan_trail::Color* m_fill_color;
      double m_resolution; // DPI
      bool m_import_PDF_notes;
      // prevent compiler from generating
      PDF_slide_collection(const PDF_slide_collection& slide_collection);
      PDF_slide_collection& operator= (const PDF_slide_collection& slide_collection);
    };

    //
    // Template implementation
    //

    //
    // Constructor/destructor
    //

    template <class Image> PDF_slide_collection<Image>::PDF_slide_collection()
      : Slide_collection<Image>(),
        m_fill_color(new Roan_trail::Color(0xFFFFFF)), // white
        m_resolution(300.0),
        m_import_PDF_notes(true)
    {
    }

    template <class Image> PDF_slide_collection<Image>::~PDF_slide_collection()
    {
      delete m_fill_color;
    }

    //
    //   Load/save
    //

    template <class Image> bool PDF_slide_collection<Image>::load_from_source(const std::string& path,
                                                                              Error_param& return_error)
    {
      precondition(!return_error()
                   && mf_invariant());

      bool return_value = true;

      start_error_block();

      const std::string& source_path = path;

      const bool source_exists = File_manager::path_exists(source_path);
      on_error(!source_exists, new Source_error(error_location(),
                                                Source_error::invalid_source,
                                                std::string("Source path not found: ") + source_path));
      const bool is_directory = File_manager::path_is_directory(source_path);
      on_error(is_directory, new Source_error(error_location(),
                                               Source_error::invalid_source,
                                               std::string("Source path is a directory: ")
                                               + source_path));

      Error_param error;
      const bool images_processed = mf_process_source_with_notes(source_path, error);
      on_error(!images_processed, new Source_error(error_location(),
                                                   Source_error::general,
                                                   error()));

      Slide_collection<Image>::m_source = source_path;
      Slide_collection<Image>::m_source_loaded = true;

      goto exit_point;

      end_error_block();

      default_error_handler_and_cleanup(return_error,
                                        return_value,
                                        false);
      goto exit_point;

    exit_point:
      postcondition(mf_invariant());
      return return_value;
    }

    template <class Image> bool PDF_slide_collection<Image>::load_from_XML(const std::string& file_path,
                                                                           const xmlpp::Node* XML_node,
                                                                           const std::string& custom_source_path,
                                                                           Error_param& return_error)
    {
      precondition(!Slide_collection<Image>::m_source_loaded
                   && XML_node
                   && !return_error()
                   && mf_invariant());

      bool return_value = false;

      start_error_block();

      Error_param error;
      const std::string diagnostic_prefix = "error loading XML file";

      try
      {
        const char *source_path_text = ih_text_for_field("//source_path",
                                                         XML_node,
                                                         diagnostic_prefix,
                                                         file_path,
                                                         error);
        on_error(!source_path_text, new Source_error(error_location(),
                                                     Source_error::general,
                                                     error()));
        std::string source_path = (("" == custom_source_path) ? source_path_text : custom_source_path);
        const bool source_exists = File_manager::path_exists(source_path);
        on_error(!source_exists, new Source_error(error_location(),
                                                  Source_error::invalid_source,
                                                  std::string("Source path not found: ") + source_path));
        const bool is_directory = File_manager::path_is_directory(source_path);
        on_error(is_directory, new Source_error(error_location(),
                                                Source_error::invalid_source,
                                                std::string("Source path is a directory: ")
                                                + source_path));
        const std::vector<xmlpp::Node*> slide_nodes = XML_node->find("//slide");
        on_error(!slide_nodes.size(), new Source_error(error_location(),
                                                       Source_error::load,
                                                       diagnostic_prefix + std::string(", node: ")
                                                       + XML_node->get_path()
                                                       + std::string(" no slides found"),
                                                       file_path
                                                       + ih_format_XML_file_line(XML_node->get_line())));

        std::vector<Slide<Image>*> slides;
        for (std::vector<xmlpp::Node*>::const_iterator current_slide_node = slide_nodes.begin();
             current_slide_node != slide_nodes.end();
             ++current_slide_node)
        {
          // index
          const char *slide_index_text = ih_text_for_field("index",
                                                           (*current_slide_node),
                                                           diagnostic_prefix,
                                                           file_path,
                                                           error);
          on_error(!slide_index_text, new Source_error(error_location(),
                                                       Source_error::general,
                                                       error()));
          size_t current_slide_index;
          std::stringstream current_slide_index_stream(slide_index_text);
          current_slide_index_stream >> current_slide_index;
          // notes
          Error_param notes_error(false);
          const char *notes_text = ih_text_for_field("notes",
                                                     (*current_slide_node),
                                                     diagnostic_prefix,
                                                     source_path,
                                                     notes_error); // ignore error
          std::string slide_notes;
          if (notes_text)
          {
            slide_notes = notes_text;
          }
          // else if notes_text is empty, there might be no notes text node, so ignore any error and go on

          slides.push_back(new Slide<Image>(current_slide_index,
                                            slide_notes,
                                            ""));
          if (slides.size() != (current_slide_index + 1))
          {
            std::stringstream diagnostic_stream;
            diagnostic_stream << diagnostic_prefix;
            diagnostic_stream << ", invalid slide index " << current_slide_index;
            diagnostic_stream << " found at actual index " << slides.size() - 1;
            on_error(true, new Source_error(error_location(),
                                            Source_error::load,
                                            file_path));
          }
        }

        const bool document_loaded = mf_load_document(source_path, error);
        on_error(!document_loaded, new Source_error(error_location(),
                                                    Source_error::general,
                                                    error()));
        Slide_collection<Image>::m_slides = slides;
        Slide_collection<Image>::m_source = source_path;
        Slide_collection<Image>::m_source_loaded = true;

        return_value = true;

      }
      catch (const xmlpp::exception& e)
      {
        const std::string diagnostic = diagnostic_prefix + std::string(": ") + e.what();
        on_error(true, new Source_error(error_location(),
                                        Source_error::load,
                                        diagnostic,
                                        file_path));
      }

      goto exit_point;
      end_error_block();

      default_error_handler_and_cleanup(return_error,
                                        return_value,
                                        false);
      goto exit_point;

    exit_point:
      postcondition(return_error.is_valid_at_return(return_value)
                    && mf_invariant());
      return return_value;
    }

    template <class Image> bool PDF_slide_collection<Image>::save_as_XML(const std::string& file_path,
                                                                         xmlpp::Node* XML_node,
                                                                         Error_param& return_error)
    {
      precondition(XML_node
                   && !return_error()
                   && mf_invariant());

      bool return_value = false;
      start_error_block();

      Error_param error;

      try
      {
        xmlpp::Node* slides_node = XML_node->add_child("slides");
        xmlpp::Element* source_type_element = slides_node->add_child("source_type");
        source_type_element->add_child_text(source_type());
        xmlpp::Element* source_path_element = slides_node->add_child("source_path");
        source_path_element->add_child_text(Slide_collection<Image>::source());
        for (size_t slide_index = 0; slide_index < Slide_collection<Image>::m_slides.size(); ++slide_index)
        {
          const Slide<Image> &current_slide = *(Slide_collection<Image>::m_slides[slide_index]);
          assert((static_cast<int>(slide_index) == current_slide.index())
                 && "error, slide numbers do not match when saving to XML");
          xmlpp::Node* current_slide_node = slides_node->add_child("slide");
          std::stringstream slide_index_stream;
          slide_index_stream << slide_index;
          xmlpp::Element* index_element = current_slide_node->add_child("index");
          index_element->add_child_text(slide_index_stream.str());
          xmlpp::Element* notes_element = current_slide_node->add_child("notes");
          notes_element->add_child_text(current_slide.notes());
        }
      }
      catch (const xmlpp::exception& e)
      {
        std::string diagnostic = std::string("error saving XML to file: ") + e.what();
        on_error(true, new Source_error(error_location(),
                                        Source_error::save,
                                        diagnostic,
                                        file_path));
      }

      return_value = true;

      goto exit_point;
      end_error_block();

      default_error_handler_and_cleanup(return_error,
                                        return_value,
                                        false);
      goto exit_point;

    exit_point:
      postcondition(return_error.is_valid_at_return(return_value)
                    && mf_invariant());
      return return_value;
    }

    //
    //   Protected member functions
    //

    template <class Image> bool PDF_slide_collection<Image>::mf_invariant(bool check_base_class) const
    {
      bool return_value = false;

      if (!m_fill_color)
      {
        goto exit_point;
      }

      return_value = !check_base_class || Slide_collection<Image>::mf_invariant(check_base_class);

      goto exit_point;

    exit_point:
      return return_value;
    }
  }
}

#endif // PDF_SLIDE_COLLECTION_HPP_
