// Narration_recorder_importer.cpp
//
// Copyright 2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Narration_recorder_importer.hpp"
#include "Segment.hpp"
#include "error/Record_error.hpp"
#include <libxml++/exceptions/exception.h>
#include <libxml++/nodes/node.h>
#include <libxml2/libxml/tree.h>
#include <sstream>
#include <string>
#include <vector>

using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using xmlpp::Node;
using Roan_trail::Long_int;
using Roan_trail::Error;
using Roan_trail::Error_param;
using Roan_trail::Recorder::Record_error;
using Roan_trail::Recorder::Segment;
using namespace Roan_trail::Recorder;

//
// Internal helpers
//

namespace
{
  string ih_format_XML_file_line(int line)
  {
    stringstream line_stream;
    line_stream << ":" << line;

    return line_stream.str();
  }

  const char* ih_text_for_field(const char *node_name,
                                const Node* segment_node,
                                const string& file_path,
                                Error_param& return_error)
  {
    precondition(node_name
                 && segment_node
                 && !return_error());

    const char* return_value = 0;

    start_error_block();

    const string diagnostic_prefix = "error loading from XML file, node:";

    const vector<Node*> field_nodes = segment_node->find(node_name);
    on_error(!field_nodes.size(),
             new Record_error(error_location(),
                              Record_error::load,
                              diagnostic_prefix
                              + segment_node->get_path()
                              + string(" ")
                              + string(node_name)
                              + string(" node not found"),
                              file_path + ih_format_XML_file_line(segment_node->get_line())));
    const vector<Node*> field_text_nodes = field_nodes[0]->find("text()");
    on_error(!field_text_nodes.size(),
             new Record_error(error_location(),
                              Record_error::load,
                              diagnostic_prefix
                              + field_nodes[0]->get_path()
                              + string(" text node not found"),
                              file_path + ih_format_XML_file_line(field_nodes[0]->get_line())));
    // the reinterpret cast is used to convert unsigned char to char
    // unfortunately, there seems to be no obvious way around this
    const char *field_text_content =
      reinterpret_cast<const char *>(field_text_nodes[0]->cobj()->content);
    on_error(!field_text_content,
             new Record_error(error_location(),
                              Record_error::load,
                              diagnostic_prefix
                              + field_text_nodes[0]->get_path()
                              + string(" text content not found"),
                              file_path
                              + ih_format_XML_file_line(field_text_nodes[0]->get_line())));

    return_value = field_text_content;
    goto exit_point;

    end_error_block();

    default_error_handler_and_cleanup(return_error,
                                      return_value,
                                      0);
    goto exit_point;

  exit_point:
    postcondition(return_error.is_valid_at_return(return_value));
    return return_value;
  }

  Long_int ih_long_int_from_text(const char *node_text)
  {
    precondition(node_text);

    Long_int return_value;

    stringstream field_stream(node_text);

    field_stream >> return_value;

    return return_value;
  }
}


//
// load/save
//

bool Narration_recorder_importer::import_segments_from_XML(const string& file_path,
                                                           const Node* XML_node,
                                                           vector<Segment>& return_segments,
                                                           Error_param& return_error)
{
  precondition(XML_node
               && !return_error());

  vector<Segment> segments;
  bool return_value = false;

  start_error_block();

  Error_param error;
  const string diagnostic_prefix = "error importing from XML file";

  try
  {
    // get segments
    const vector<Node*> recorder_nodes = XML_node->find("//recorder");
    on_error(!recorder_nodes.size(),
             new Record_error(error_location(),
                              Record_error::load,
                              diagnostic_prefix + string(", node: recorder not found"),
                              file_path));
    const vector<Node*> segment_nodes = recorder_nodes[0]->find("//segment");
    on_error(!segment_nodes.size(),
             new Record_error(error_location(),
                              Record_error::load,
                              diagnostic_prefix + string(", node: ")
                              + recorder_nodes[0]->get_path()
                              + string(" segments node not found"),
                              file_path + ih_format_XML_file_line(recorder_nodes[0]->get_line())));

    for (vector<Node*>::const_iterator current_segment_node = segment_nodes.begin();
         current_segment_node != segment_nodes.end();
         ++current_segment_node)
    {
      const char* index_text = ih_text_for_field("index",
                                                 (*current_segment_node),
                                                 file_path,
                                                 error);
      on_error(!index_text, new Record_error(error_location(),
                                             Record_error::general,
                                             error()));
      const Long_int index = ih_long_int_from_text(index_text);

      const char* start_frame_text = ih_text_for_field("start_frame",
                                                       (*current_segment_node),
                                                       file_path,
                                                       error);
      on_error(!start_frame_text, new Record_error(error_location(),
                                                   Record_error::general,
                                                   error()));
      const Long_int start_frame = ih_long_int_from_text(start_frame_text);

      const char* end_frame_text = ih_text_for_field("end_frame",
                                                     (*current_segment_node),
                                                     file_path,
                                                     error);
      on_error(!end_frame_text, new Record_error(error_location(),
                                                 Record_error::general,
                                                 error()));
      const Long_int end_frame = ih_long_int_from_text(end_frame_text);

      const char* ordinality_text = ih_text_for_field("ordinality",
                                                      (*current_segment_node),
                                                      file_path,
                                                      error);
      on_error(!ordinality_text, new Record_error(error_location(),
                                                  Record_error::general,
                                                  error()));
      const Long_int ordinality = ih_long_int_from_text(ordinality_text);

      const char* retake_text = ih_text_for_field("retake",
                                                  (*current_segment_node),
                                                  file_path,
                                                  error);
      on_error(!retake_text, new Record_error(error_location(),
                                              Record_error::general,
                                              error()));
      const Long_int retake_int = ih_long_int_from_text(retake_text);

      Segment current_segment(index, start_frame, end_frame, ordinality, (retake_int == 1));
      segments.push_back(current_segment);
    }
  }
  catch (const xmlpp::exception& e)
  {
    const string diagnostic = diagnostic_prefix + string(": ") + e.what();
    on_error(true, new Record_error(error_location(),
                                    Record_error::load,
                                    diagnostic,
                                    file_path));
  }

  return_segments = segments;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}

bool Narration_recorder_importer::import_audio_recording_from_XML(const string& file_path,
                                                                  const Node* XML_node,
                                                                  string& return_audio_recording_path,
                                                                  Long_int& return_audio_recording_frame_rate,
                                                                  Error_param& return_error)
{
  precondition(XML_node
               && !return_error());

  const char *audio_recording_path_text = 0;
  Long_int audio_recording_frame_rate = 0;
  bool return_value = false;

  start_error_block();

  Error_param error;
  const string diagnostic_prefix = "error importing from XML file";

  try
  {
    // get the audio recording path
    const vector<Node*> recorder_nodes = XML_node->find("//recorder");
    on_error(!recorder_nodes.size(),
             new Record_error(error_location(),
                              Record_error::load,
                              diagnostic_prefix + string(", node: recorder not found"),
                              file_path));
    audio_recording_path_text = ih_text_for_field("audio_recording_path",
                                                  recorder_nodes[0],
                                                  file_path,
                                                  error);
    on_error(!audio_recording_path_text, new Record_error(error_location(),
                                                          Record_error::general,
                                                          error()));
    // and frame rate
    const char *audio_recording_frame_rate_text = ih_text_for_field("audio_recording_frame_rate",
                                                                    recorder_nodes[0],
                                                                    file_path,
                                                                    error);
    on_error(!audio_recording_frame_rate_text, new Record_error(error_location(),
                                                                Record_error::general,
                                                                error()));
    audio_recording_frame_rate = ih_long_int_from_text(audio_recording_frame_rate_text);
  }
  catch (const xmlpp::exception& e)
  {
    const string diagnostic = diagnostic_prefix + string(": ") + e.what();
    on_error(true, new Record_error(error_location(),
                                    Record_error::load,
                                    diagnostic,
                                    file_path));
  }

  return_audio_recording_path = audio_recording_path_text;
  return_audio_recording_frame_rate = audio_recording_frame_rate;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}

