// Movie_builder.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// Abstract base, movie builder class

#ifndef MOVIE_BUILDER_HPP_
#define MOVIE_BUILDER_HPP_

#include "common.hpp"
#include <vector>
#include <string>
#include <cstddef>

namespace Roan_trail
{
  class Error_param;
  class Logger;

  namespace Recorder
  {
    class Segment;
  }

  namespace Builder
  {
    class Movie_builder_config;
    class Movie_builder_model;
    class Sound_builder;
    class Movie;

    class Movie_builder
    {
    public:
      virtual ~Movie_builder();
      // accessors
      //   build state
      bool running() const { return m_running; }
      bool completed() const { return m_completed; }
      //   build counters
      Long_int current_segment_index() const { return m_current_segment_index; }
      double total_fraction_completed() const { return m_total_fraction_completed; }
      //   movie
      const std::vector<std::string>& output_movies() const { return m_output_movies; }
      const std::string& movie_file_name() const { return m_movie_file_name; }
      const Movie& movie() const { return *m_movie; }
      //   other
      const Sound_builder& sound_builder() const { return *m_sound_builder; }
      //
      // build functions
      // -----------------------------------------------------------
      //   step 1
      bool setup(Error_param& return_error);
      //   step 2
      bool start(Error_param& return_error);
      //   step 3 (repeat for all segments)
      bool add_next_segment(bool& return_more_segments, Error_param& return_error);
      //   step 4
      bool stop(bool completed, Error_param& return_error);
      //   step 5
      bool end(bool completed,
               std::vector<Error_param>& return_warnings,
               Error_param& return_error);
      // -----------------------------------------------------------
      //
      // other member functions
      Long_int total_video_frames(const std::vector<Roan_trail::Recorder::Segment>& segments,
                                  bool using_silence_gaps,
                                  Long_int silence_gap_frame_count) const;
      static std::string name_for_movie(const std::string& movie_name,
                                   size_t movie_index,
                                   Long_int segment_index,
                                   Long_int slide_index);
      // for prebuild checks
      bool movie_or_movies_exist() const;
      Long_int estimate_build_size();
    protected:
      Movie_builder(const Movie_builder_config& config, const Movie_builder_model& model);
      // frame cache (so we don't have to allocate a frame every time one is to be added)
      const uint8_t* m_video_frame_cache;
      // return_used_default parameter is set to true if image could not be found and a default was used:
      // (meant to be implemented by a subclass, using a specific image processing library)
      virtual const uint8_t* mf_generate_image(Long_int slide_index, bool& return_used_default) = 0;
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      const Movie_builder_config& m_config;
      const Movie_builder_model& m_model;
      Logger& m_logger;
      Sound_builder* m_sound_builder;
      // build state
      bool m_setup;
      bool m_running;
      bool m_completed;
      bool m_cancelled;
      // build counters
      Long_int m_total_destination_count;
      Long_int m_current_segment_index;
      double m_total_fraction_completed;
      Long_int m_current_source_audio_position;
      Long_int m_dest_frames_added;
      // segment values
      Long_int m_segment_count;
      Long_int m_current_segment_video_start_frame;
      Long_int m_current_segment_video_duration;
      double m_video_frames_per_reference_frame;
      // movie
      Movie* m_movie;
      std::vector<std::string> m_output_movies;
      std::string m_movie_file_name;
      std::vector<std::string> m_movie_paths; // cumulative paths of destination movies started
      std::vector<std::string> m_temporary_paths; // cumulative paths of temporary movies
      // build-related functions
      //   common
      void mf_pre_update();
      void mf_post_update();
      //     commit
      bool mf_move_movies_atomically(std::vector<Error_param>& return_warnings, Error_param& return_error);
      //     rollback
      void mf_remove_temporary_movies(std::vector<Error_param>& return_warnings);
      //   single movie
      bool mf_add_segment_to_single_movie(Error_param& return_error);
      //   multiple movies
      bool mf_add_segment_to_multiple_movies(Error_param& return_error);
      // other functions
      static bool mf_at_least_one_movie_exists_in(const std::vector<std::string>& movie_paths);
      void mf_video_parameters_for_segment(const Roan_trail::Recorder::Segment& segment,
                                           Long_int& video_start_frame,
                                           Long_int& video_frame_count) const;
      bool mf_check_source_audio_sample_rate(Error_param& return_error);
      // prevent compiler from generating
      Movie_builder(const Movie_builder& b);
      Movie_builder& operator=(const Movie_builder& b);
    };
  }
}

#endif // MOVIE_BUILDER_HPP_
