// Movie_builder.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// Format for movie with silence gaps enabled
//
// * - Video
// ~ - Audio
// 0 - Audio Fill (silence)
//
// T = silence gap length (video frames)
//
//
// >------------------------------------------------------------->
// >**|*************|*************|*************|*************|**>  Video Track
// >--+---       ---+---       ---+---       ---+---       ---+-->
// >~~|00|~~~~~~~|00|00|~~~~~~~|00|00|~~~~~~~|00|00|~~~~~~~|00|~~>  Audio Track
// >------------------------------------------------------------->
//     ^          ^
//     |          |
//     |          Silence Out (Length T)
//     |
//     Silence In (Length T)

#include "Movie_builder.hpp"
#include "Movie_builder_config.hpp"
#include "Movie_builder_model.hpp"
#include "AV_compat.hpp"
#include "Sound_builder.hpp"
#include "Movie.hpp"
#include "Movie_config.hpp"
#include "Sound_file_config.hpp"
#include "Segment.hpp"
#include "File_manager.hpp"
#include "Logger.hpp"
#include "error/Build_error.hpp"
#include "error/Posix_error.hpp"
#include <iomanip>
#include <string>
#include <sstream>
#include <vector>
#include <cmath>
#include <cstddef>
#include <sys/stat.h>

using std::endl;
using std::setw;
using std::setfill;
using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Builder::AV_compat;
using Roan_trail::Recorder::Segment;
using Roan_trail::Recorder::Sound_file_config;
using Roan_trail::Long_int;
using namespace Roan_trail::Builder;

//
// Internal helpers
//
namespace
{
  const double ic_estimate_multiple_movie_factor = 1.01;
}

Movie_builder::~Movie_builder()
{
  precondition(mf_invariant(false));

  delete m_movie;
  delete m_sound_builder;
}

//
// Other member functions
//

Long_int Movie_builder::total_video_frames(const vector<Segment>& segments,
                                           bool using_silence_gaps,
                                           Long_int silence_gap_frame_count) const
{
  precondition((silence_gap_frame_count >= 0)
               && mf_invariant());

  Long_int total_video_frames = 0;

  for (size_t i = 0; i < segments.size(); ++i)
  {
    const Segment& current_segment = segments[i];
    Long_int segment_start_frame;
    Long_int segment_frame_count;
    mf_video_parameters_for_segment(current_segment,
                                    segment_start_frame,
                                    segment_frame_count);
    if (using_silence_gaps)
    {
      segment_frame_count += silence_gap_frame_count;
    }
    total_video_frames += segment_frame_count;
  }

  postcondition(mf_invariant());
  return total_video_frames;
}

//
// Build member functions
//

bool Movie_builder::setup(Error_param& return_error)
{
  precondition(!m_setup
               && !m_running
               && !return_error()
               && mf_invariant());

  bool need_sound_builder_end = false;
  bool return_value = false;

  start_error_block();

  Error_param error;

  const bool sound_builder_started = m_sound_builder->start(m_config.movie->sound->file_name, error);
  on_error(!sound_builder_started, new Build_error(error_location(),
                                                   Build_error::general,
                                                   error()));
  need_sound_builder_end = true;

  const bool sound_sample_rate_compatible = mf_check_source_audio_sample_rate(error);
  on_error(!sound_sample_rate_compatible, new Build_error(error_location(),
                                                          Build_error::general,
                                                          error()));

  const bool movie_setup = m_movie->setup(m_sound_builder, error);
  on_error(!movie_setup, new Build_error(error_location(),
                                         Build_error::general,
                                         error()));

  // extension added back later
  m_movie_file_name = File_manager::replace_file_extension(m_config.output.movie_name);

  Frame_rate video_frame_rate;
  const bool use_high_resolution_video_timescale = AV_compat::need_high_resolution_video_timescale();
  Movie_config::frame_rate_for_frame_rate_type(m_config.movie->frame_rate,
                                               use_high_resolution_video_timescale,
                                               video_frame_rate);
  m_video_frames_per_reference_frame = (static_cast<double>(video_frame_rate.time_scale())
                                        / static_cast<double>(video_frame_rate.frame_length()))
    / static_cast<double>(m_model.audio_recording_frame_rate());
  m_total_destination_count = total_video_frames(m_model.segments(),
                                                 m_config.movie_attributes.silence_gaps,
                                                 m_config.movie_attributes.silence_gap_length);

  m_setup = true;
  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (need_sound_builder_end)
  {
    Error_param error(false);
    m_sound_builder->end(error); // ignore error
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(// success
                ((return_value && m_setup)
                 // failure
                 || (!return_value && !m_setup))
                && !m_running
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;

}

bool Movie_builder::start(Error_param& return_error)
{
  precondition(m_setup
               && !m_running
               && !return_error()
               && mf_invariant());

  bool need_movie_end = false;
  bool return_value = false;

  start_error_block();

  Error_param error;

  if (!m_config.output.create_multiple_movies)
  {
    const string current_movie_name = m_movie_file_name + string(".")
      + Movie_config::extension_for_movie_preset(m_config.movie->movie_preset);
    string movie_file_name = m_config.output.directory + string("/") + current_movie_name;
    const bool movie_started = m_movie->start(movie_file_name,
                                              m_sound_builder,
                                              error);
    on_error(!movie_started, new Build_error(error_location(),
                                             Build_error::general,
                                             error()));
    m_logger << Logger::info << "Opened temporary movie at path: ";
    m_logger << m_movie->temporary_movie_file_path() << endl;
    m_temporary_paths.push_back(m_movie->temporary_movie_file_path());
    need_movie_end = true;
  }

  m_total_fraction_completed = 0.0;
  m_dest_frames_added = 0;
  m_current_segment_index = 0;
  m_running = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (!m_config.output.create_multiple_movies)
  {
    if (need_movie_end)
    {
      Error_param error(false);
      m_movie->end(error); // ignore error
      m_logger << Logger::info;
      m_logger << "Performing error cleanup:" << endl;
      m_logger << "Closed temporary movie file" << endl;
      vector<Error_param> warnings;
      mf_remove_temporary_movies(warnings); // ignore warnings
    }
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(// success
                ((return_value && m_running)
                 // failure
                 || (!return_value && !m_running))
                && m_setup
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Movie_builder::add_next_segment(bool& return_more_segments, Error_param& return_error)
{
  precondition(m_setup
               && m_running
               && !return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;
  if (m_config.output.create_multiple_movies)
  {
    const bool added_segment = mf_add_segment_to_multiple_movies(error);
    on_error(!added_segment, new Build_error(error_location(),
                                             Build_error::general,
                                             error()));
  }
  else
  {
    const bool added_segment = mf_add_segment_to_single_movie(error);
    on_error(!added_segment, new Build_error(error_location(),
                                             Build_error::general,
                                             error()));
  }

  ++m_current_segment_index;
  const Long_int segment_count = static_cast<Long_int>(m_model.segments().size());
  return_more_segments = (m_current_segment_index < segment_count);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && m_running
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Movie_builder::stop(bool completed, Error_param& return_error)
{
  precondition(m_setup
               && m_running
               && !return_error()
               && mf_invariant());

  bool return_value = false;
  bool need_movie_end = true;

  start_error_block();

  m_running = false;

  Error_param error;

  if (!m_config.output.create_multiple_movies)
  {
    need_movie_end = false;
    const bool movie_ended = m_movie->end(error);
    on_error(!movie_ended, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));
    m_logger << Logger::info << "Closed temporary movie" << endl;
    if (completed)
    {
      // put the destination movie path in a list for copying atomically later
      m_movie_paths.push_back(m_movie->movie_file_path());
    }
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (need_movie_end)
  {
    Error_param error(false);
    m_movie->end(error); // ignore error
    m_logger << Logger::info;
    m_logger << "Performing error cleanup:" << endl;
    m_logger << "Closed temporary movie file" << endl;
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && !m_running
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Movie_builder::end(bool completed,
                        vector<Error_param>& return_warnings,
                        Error_param& return_error)
{
  precondition(m_setup
               && !m_running
               && !return_warnings.size()
               && !return_error()
               && mf_invariant());

  bool return_value = false;
  bool need_remove_temporary_movies = true;
  bool need_sound_builder_end = true;

  start_error_block();

  Error_param error;

  m_output_movies.clear();

  if (completed)
  {
    m_logger << Logger::info << "Completed building temporary movie(s)" << endl;
    // move the temporary files to their final destinations
    if (m_config.output.temporary_movies)
    {
      // ... and update list completed (just use temporaries)
      const vector<string>& movies = m_temporary_paths;
      m_output_movies = movies;
      need_remove_temporary_movies = false;
    }
    else
    {
      vector<Error_param> warnings;
      const bool movies_moved = mf_move_movies_atomically(warnings, error);
      on_error(!movies_moved, new Build_error(error_location(),
                                              Build_error::general,
                                              error()));
      if (warnings.size())
      {
        return_warnings = warnings;
      }

      /// ... and update list completed
      const vector<string>& movies = m_movie_paths;
      m_output_movies = movies;
      need_remove_temporary_movies = false;
    }
  }
  else
  {
    vector<Error_param> warnings;
    need_remove_temporary_movies = false;
    mf_remove_temporary_movies(warnings);
    if (warnings.size())
    {
      return_warnings = warnings;
    }
  }

  need_sound_builder_end = false;
  m_sound_builder->end(error); // ignore error

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (need_remove_temporary_movies)
  {
    vector<Error_param> warnings;
    mf_remove_temporary_movies(warnings); // ignore warnings
  }
  if (need_sound_builder_end)
  {
    Error_param error(false);
    m_sound_builder->end(error); // ignore error
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && !m_running
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// Other
//

string Movie_builder::name_for_movie(const string& movie_name,
                                     size_t movie_index,
                                     Long_int segment_index,
                                     Long_int slide_index)
{
  precondition((movie_index >= 0)
               && (segment_index >= 0)
               && (slide_index >= 0));

  stringstream movie_name_stream;

  movie_name_stream << movie_name << "_" << setw(6) << setfill('0') << (movie_index + 1);
  movie_name_stream << "_seg_" << setw(6) << setfill('0') << (segment_index + 1);
  movie_name_stream << "_slide_" << setw(6) << setfill('0') << (slide_index + 1);

  return movie_name_stream.str();
}

bool Movie_builder::movie_or_movies_exist() const
{
  precondition(mf_invariant());

  bool return_value = false;

  if (!m_config.output.temporary_movies)
  {
    const string extension = "." + Movie_config::extension_for_movie_preset(m_config.movie->movie_preset);
    if (m_config.output.create_multiple_movies)
    {
      vector<string> movie_names;
      const vector<Segment>& segments = m_model.segments();
      const size_t segments_count = segments.size();
      for (size_t i = 0; i < segments_count; ++i)
      {
        const Segment& current_segment = segments[i];
        const string movie_name = name_for_movie(m_movie_file_name,
                                                 i,
                                                 current_segment.ordinality(),
                                                 current_segment.index());
        const string movie_path = m_config.output.directory + string("/") + movie_name + extension;
        movie_names.push_back(movie_path);
      }
      return_value = mf_at_least_one_movie_exists_in(movie_names);
    }
    else
    {
      stringstream movie_path_name;
      movie_path_name << m_config.output.directory << "/" << m_movie_file_name + extension;

      return_value = File_manager::path_exists(movie_path_name.str());
    }
  }

  postcondition(mf_invariant());
  return return_value;
}

Long_int Movie_builder::estimate_build_size()
{
  precondition(m_setup
               && !m_running
               && mf_invariant());

  const vector<Segment>& segments = m_model.segments();
  // use the middle segment for estimate
  const size_t middle = segments.size() / 2;
  const Segment& middle_segment = segments[middle];
  // cache the image from the source
  bool default_used;
  mf_generate_image(middle_segment.index(), default_used);
  if (default_used)
  {
    m_logger << Logger::warning;
    m_logger << "When estimating build size, image for slide " << middle_segment.index() + 1;
    m_logger << " not found, using default image" << endl;
  }
  // use the cache as the sample frame
  const uint8_t* raw_frame = m_video_frame_cache;
  const Long_int video_frame_size = m_movie->estimate_video_frame_size(raw_frame);
  Long_int size_estimate = 0;
  vector<Segment> segment;
  for (size_t i = 0; i < segments.size(); ++i)
  {
    segment.clear();
    segment.push_back(segments[i]);
    const Long_int video_frame_count = total_video_frames(segment,
                                                          m_config.movie_attributes.silence_gaps,
                                                          m_config.movie_attributes.silence_gap_length);
    Long_int segment_size = m_movie->estimate_segment_size(video_frame_size, video_frame_count);
    if (m_config.output.create_multiple_movies)
    {
      segment_size *= ic_estimate_multiple_movie_factor;
    }
    size_estimate += segment_size;
  }

  postcondition((size_estimate >= 0)
                && m_setup
                && !m_running
                && mf_invariant());

  return size_estimate;
}

//
// Protected member functions
//

Movie_builder::Movie_builder(const Movie_builder_config& config, const Movie_builder_model& model)
  : m_video_frame_cache(0),
    m_config(config),
    m_model(model),
    m_logger(Logger::default_logger()),
    m_sound_builder(new Sound_builder),
    m_setup(false),
    m_running(false),
    m_total_destination_count(0),
    m_current_segment_index(0),
    m_total_fraction_completed(0.0),
    m_dest_frames_added(0),
    m_segment_count(0),
    m_current_segment_video_start_frame(0),
    m_current_segment_video_duration(0),
    m_video_frames_per_reference_frame(0.0),
    m_movie(new Movie(*m_config.movie)),
    m_output_movies(),
    m_movie_file_name(),
    m_movie_paths(),
    m_temporary_paths()
{
  postcondition(mf_invariant(false));
}

//
//   Invariant
//

bool Movie_builder::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_movie
      || !m_sound_builder)
  {
    goto exit_point;
  }


  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

//
//   common
//

void Movie_builder::mf_pre_update()
{
  const vector<Segment>& segments = m_model.segments();
  const Segment& current_segment = segments[m_current_segment_index];

  mf_video_parameters_for_segment(current_segment,
                                  m_current_segment_video_start_frame,
                                  m_current_segment_video_duration);
}

void Movie_builder::mf_post_update()
{
  m_dest_frames_added += m_current_segment_video_duration;
    + (2 * m_config.movie_attributes.silence_gap_length);

  m_total_fraction_completed = static_cast<double>(m_dest_frames_added)
    / static_cast<double>(m_total_destination_count);
}

bool Movie_builder::mf_move_movies_atomically(vector<Error_param>& return_warnings, Error_param& return_error)
{
  precondition(!return_error());

  vector<string> backup_paths;
  vector<string> original_paths;
  vector<string> completed_paths;
  bool return_value = false;

  start_error_block();

  // make a copy of the temporary paths first, because
  // we will remove them from the instance variable
  // as the files themselves are deleted
  const vector<string> temporary_paths = m_temporary_paths;

  assert((m_movie_paths.size() == m_temporary_paths.size()) // TODO: in invariant?
         && "error, destination movie path count does not equal source temporary path count");

  Error_param error;

  for (size_t i = 0; i < m_movie_paths.size(); ++i)
  {
    const string& dest_path = m_movie_paths[i];
    const string &source_path = temporary_paths[i];
    string backup_file;
    bool backup_existing = true;
    // (specify move to a backup file if the file already exists)
    m_logger << Logger::info;
    m_logger << "Moving temporary movie: " << source_path;
    m_logger << " to destination: " << dest_path << endl;
    const bool file_moved = File_manager::move_file_with_backup(source_path,
                                                                dest_path,
                                                                m_config.output.file_overwrite,
                                                                backup_existing,
                                                                backup_file,
                                                                error);
    if (backup_existing)
    {
      m_logger << Logger::info << "Overwrite, existing movie backed up to: " << backup_file << endl;
      // file manager move operation indicated that a backup was made, store it
      backup_paths.push_back(backup_file);
      original_paths.push_back(dest_path);
    }
    on_error(!file_moved, new Build_error(error_location(),
                                          Build_error::move_temporary_to_destination_movie,
                                          source_path,
                                          dest_path,
                                          "",
                                          error()));
    completed_paths.push_back(dest_path);

    // change permissions on the destination path
    m_logger << Logger::info << "Changing permissions to rw-r--r-- for: " <<  dest_path << endl;
    const int chmod_return = chmod(dest_path.c_str(), 0644); // rw-r--r--
    on_error(chmod_return, new Build_error(error_location(),
                                           Build_error::destination_movie_change_permissions,
                                           new Posix_error(chmod_return,
                                                           "chmod",
                                                           dest_path.c_str())));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  // remove partial files
  for (size_t i = 0; i < completed_paths.size(); ++i)
  {
    const string dest_path = completed_paths[i];
    unlink(dest_path.c_str());
  }
  // restore backup files
  for (size_t i = 0; i < completed_paths.size(); ++i)
  {
    const string backup_path = backup_paths[i];
    const string original_path = original_paths[i];
    Error_param error(false);
    // ignore return error
    File_manager::move_file(backup_path,
                            original_path,
                            true,
                            error);
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

void Movie_builder::mf_remove_temporary_movies(vector<Error_param>& return_warnings)
{
  precondition(!return_warnings.size());

  // make a copy of the temporary paths first, because
  // we will remove them from the instance variable
  // as the files themselves are deleted
  const vector<string> temporary_paths = m_temporary_paths;
  for (vector<string>::const_iterator i = temporary_paths.begin(); i != temporary_paths.end(); ++i)
  {
    m_logger << Logger::info << "Removing temporary movie: " << i->c_str() << endl;
    const int unlink_return = unlink(i->c_str());
    if (unlink_return)
    {
      Error_param warning;
      Error* posix_warning = new Posix_error(unlink_return,
                                             "unlink",
                                             i->c_str());
      warning = posix_warning;
      return_warnings.push_back(warning);
    }
  }
}

//
//   single movie
//

bool Movie_builder::mf_add_segment_to_single_movie(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  mf_pre_update();

  Error_param error;

  const vector<Segment>& segments = m_model.segments();
  const Segment& current_segment = segments[m_current_segment_index];

  // add single frame with calculated length

  // cache the image from the source
  m_logger << Logger::info;
  m_logger << "Loading image for segment" << endl;
  bool default_used;
  mf_generate_image(current_segment.index(), default_used);
  if (default_used)
  {
    m_logger << Logger::warning;
    m_logger << "When adding segment, image for slide " << current_segment.index() + 1;
    m_logger << " not found, using default image" << endl;
  }
  // use the cache
  const uint8_t* raw_frame = m_video_frame_cache;

  // add a silence in frame to the movie, if applicable
  if (m_config.movie_attributes.silence_gaps)
  {
    m_logger << "Adding " << m_config.movie_attributes.silence_gap_length;
    m_logger << " video frames of silence in" << endl;
    const bool added_frame = m_movie->add_silent(raw_frame,
                                                 m_config.movie_attributes.silence_gap_length,
                                                 error);
    on_error(!added_frame, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));
  }

  // add the frame and its corresponding audio to the movie
  if (m_current_segment_video_duration)
  {
    // add video
    m_logger << "Adding " << m_current_segment_video_duration << " video frames" << endl;
    const bool added_frame = m_movie->add(raw_frame,
                                          m_current_segment_video_start_frame,
                                          m_current_segment_video_duration,
                                          error);
    on_error(!added_frame, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));
  }

  // add a silence out frame to the movie, if applicable
  if (m_config.movie_attributes.silence_gaps)
  {
    m_logger << "Adding " << m_config.movie_attributes.silence_gap_length;
    m_logger << " video frames of silence out" << endl;
    const bool added_frame = m_movie->add_silent(raw_frame,
                                                 m_config.movie_attributes.silence_gap_length,
                                                 error);
    on_error(!added_frame, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));
  }

  mf_post_update();

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
//   multiple movies
//

bool Movie_builder::mf_add_segment_to_multiple_movies(Error_param& return_error)
{
  precondition(!return_error());

  bool need_movie_end = false;
  bool return_value = false;

  start_error_block();

  mf_pre_update();

  const Segment& current_segment = m_model.segments()[m_current_segment_index];
  const string current_movie_name = name_for_movie(m_movie_file_name,
                                                   m_current_segment_index,
                                                   current_segment.ordinality(),
                                                   current_segment.index());
  const string movie_file_name = m_config.output.directory + string("/") + current_movie_name
    + string(".") + Movie_config::extension_for_movie_preset(m_config.movie->movie_preset);

  Error_param error;

  const bool movie_started = m_movie->start(movie_file_name,
                                            m_sound_builder,
                                            error);
  on_error(!movie_started, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));

  m_logger << Logger::info;
  m_logger << "Opened temporary movie at path: ";
  m_logger << m_movie->temporary_movie_file_path() << endl;

  m_temporary_paths.push_back(m_movie->temporary_movie_file_path());
  need_movie_end = true;

  // add single frame with calculated length

  // cache the image from the source
  m_logger << Logger::info << "Loading image for segment" << endl;
  bool default_used;
  mf_generate_image(current_segment.index(), default_used);
  if (default_used)
  {
    m_logger << Logger::warning;
    m_logger << "When adding segment, image for slide " << current_segment.index() + 1;
    m_logger << " not found, using default image" << endl;
  }
  // use the cache
  const uint8_t* raw_frame = m_video_frame_cache;

  // add a silence in frame to the movie, if applicable
  if (m_config.movie_attributes.silence_gaps)
  {
    m_logger << "Adding " << m_config.movie_attributes.silence_gap_length;
    m_logger << " video frames of silence in" << endl;
    const bool added_frame = m_movie->add_silent(raw_frame,
                                                 m_config.movie_attributes.silence_gap_length,
                                                 error);
    on_error(!added_frame, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));
  }

  // add the frame to the movie
  if (m_current_segment_video_duration)
  {
    m_logger << "Adding " << m_current_segment_video_duration << " video frames" << endl;
    const bool added_frame = m_movie->add(raw_frame,
                                          m_current_segment_video_start_frame,
                                          m_current_segment_video_duration,
                                          error);
    on_error(!added_frame, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));
  }

  // add a silence out frame to the movie, if applicable
  if (m_config.movie_attributes.silence_gaps)
  {
    m_logger << "Adding " << m_config.movie_attributes.silence_gap_length;
    m_logger << " video frames of silence out" << endl;
    const bool added_frame = m_movie->add_silent(raw_frame,
                                                 m_config.movie_attributes.silence_gap_length,
                                                 error);
    on_error(!added_frame, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));
  }

  need_movie_end = false;
  const bool movie_ended = m_movie->end(error);
  on_error(!movie_ended, new Build_error(error_location(),
                                         Build_error::general,
                                         error()));
  m_logger << "Closed temporary movie" << endl;

  // put the destination movie path in a list for copying atomically later
  m_movie_paths.push_back(m_movie->movie_file_path());

  mf_post_update();

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (need_movie_end)
  {
    Error_param error(false);
    m_logger << Logger::info;
    m_logger << "Performing error cleanup:" << endl;
    m_logger << "Closing temporary movie file" << endl;
    m_movie->end(error); // ignore error
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

bool Movie_builder::mf_at_least_one_movie_exists_in(const vector<string>& movie_paths)
{
  bool a_movie_exists = false;

  for (vector<string>::const_iterator i = movie_paths.begin(); i != movie_paths.end(); ++i)
  {
    if (File_manager::path_exists(*i))
    {
      a_movie_exists = true;
      break;
    }
  }

  return a_movie_exists;
}

void Movie_builder::mf_video_parameters_for_segment(const Segment& segment,
                                                    Long_int& return_video_start_frame,
                                                    Long_int& return_video_frame_count) const
{
  Long_int video_start_frame = 0;
  Long_int video_end_frame = 0;
  const Long_int reference_start_frame = segment.start_frame();
  if (reference_start_frame > 0)
  {
    const Long_int previous_video_end_frame = round((reference_start_frame - 1)
      * m_video_frames_per_reference_frame);
    video_start_frame = previous_video_end_frame + 1;
    video_end_frame = round(segment.end_frame() * m_video_frames_per_reference_frame);
  }
  else
  {
    video_start_frame = 0;
    video_end_frame = round(segment.end_frame() * m_video_frames_per_reference_frame);
  }

  return_video_start_frame = video_start_frame;
  return_video_frame_count = video_end_frame - video_start_frame + 1;
  if (return_video_frame_count < 0)
  {
    return_video_frame_count = 0;
  }
}

bool Movie_builder::mf_check_source_audio_sample_rate(Error_param &return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  const Sound_file_config& sound_config = m_sound_builder->config();
  const Long_int audio_sample_rate = sound_config.sample_rate;
  const bool use_high_resolution_video_timescale = AV_compat::need_high_resolution_video_timescale();
  if (!Movie_config::audio_sample_rate_compatible_with_frame_rate_type(audio_sample_rate,
                                                                       m_config.movie->frame_rate,
                                                                       use_high_resolution_video_timescale))
  {
    stringstream diagnostic;
    Frame_rate movie_frame_rate;
    Movie_config::frame_rate_for_frame_rate_type(m_config.movie->frame_rate,
                                                 use_high_resolution_video_timescale,
                                                 movie_frame_rate);
    diagnostic << "The audio sample rate " << audio_sample_rate ;
    diagnostic << " must be evenly divisible by the time scale " << movie_frame_rate.time_scale();
    diagnostic << " to be compatible with the movie frame rate of ";
    diagnostic << Movie_config::string_for_frame_rate_type(m_config.movie->frame_rate) << "." << endl << endl;
    diagnostic << "Try a different movie frame rate or convert the audio file to a compatible sample rate.";
    diagnostic << "  The following movie frame rates are compatible with " << audio_sample_rate;
    diagnostic << ":" << endl;
    vector<Frame_rate_type> compatible_frame_rates;
    Movie_config::compatible_frame_rates_for_audio_sample_rate(audio_sample_rate,
                                                               use_high_resolution_video_timescale,
                                                               compatible_frame_rates);
    if (compatible_frame_rates.size())
    {
      for (vector<Frame_rate_type>::const_iterator i = compatible_frame_rates.begin();
           i != compatible_frame_rates.end();
           ++i)
      {
        diagnostic << "  " << Movie_config::string_for_frame_rate_type(*i) << endl;
      }
    }
    else
    {
      diagnostic << "  <None>" << endl;
    }
    on_error(true, new Build_error(error_location(),
                                   Build_error::setup,
                                   sound_config.file_name,
                                   "",
                                   diagnostic.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}
