// Movie.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef MOVIE_HPP_
#define MOVIE_HPP_

#include "common.hpp"
#include "Movie_types.hpp"
#include <string>

struct AVOutputFormat;
struct AVFormatContext;

namespace Roan_trail
{
  class Error_param;
  struct Rect_size;

  namespace Builder
  {
    class Movie_config;
    class Sound_source;
    class Video_track;
    class Sound_track;
    class AV_compat;

    class Movie
    {
    public:
      // constructor/destructor/initialization
      explicit Movie(const Movie_config& config);
      virtual ~Movie();
      // accessors
      const std::string& temporary_movie_file_path() const { return m_temporary_movie_file_path; }
      const std::string& movie_file_path() const { return m_movie_file_path; }
      const Frame_rate& video_frame_rate() const;
      const Sound_track* sound_track() const { return m_sound_track; }
      const Video_track* video_track() const { return m_video_track; }
      // movie building functions
      virtual bool setup(Sound_source* sound, Error_param& return_error);
      virtual bool start(const std::string& movie_path,
                         Sound_source* sound,
                         Error_param& return_error);
      virtual bool add(const uint8_t* raw_frame,
                       Long_int video_start_frame,
                       Long_int video_frame_count,
                       Error_param& return_error);
      virtual bool add_silent(const uint8_t* raw_frame,
                              Long_int video_frame_count,
                              Error_param& return_error);
      virtual bool end(Error_param& return_error);
      // helpers
      Long_int estimate_video_frame_size(const uint8_t* raw_frame);
      Long_int estimate_segment_size(Long_int video_frame_size, Long_int video_frame_count);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      const Movie_config& m_config;
      //
      bool m_setup;
      bool m_started;
      // tracks
      Video_track* m_video_track;
      Sound_track* m_sound_track;
      //
      int m_temporary_movie_file;
      // paths
      std::string m_movie_file_path; // current destination movie
      std::string m_temporary_movie_file_path; // current temporary movie
      // FFmpeg
      //   format
      AVOutputFormat* m_output_format;
      AVFormatContext* m_format_context;
      //   API compatibility
      AV_compat* m_AV_compat_object;
      //
      static bool m_class_initialized;
      //
      void mf_cleanup();
      // prevent compiler from generating
      Movie(const Movie& m);
      Movie& operator=(const Movie& m);
    };
  }
}

#endif // MOVIE_HPP_
