// Movie.cpp
//
// Copyright 2011-2013 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Movie.hpp"
#include "AV_compat.hpp"
#include "Image_types.hpp"
#include "Movie_types.hpp"
#include "Movie_config.hpp"
#include "Sound_source.hpp"
#include "Sound_track.hpp"
#include "Video_track.hpp"
#include "File_manager.hpp"
#include "error/Build_error.hpp"
#include "error/Posix_error.hpp"
#include <string>

using std::string;
using Roan_trail::Long_int;
using Roan_trail::Builder::AV_compat;
using namespace Roan_trail::Builder;

//
// Constructor/destructor
//

Movie::Movie(const Movie_config& config)
  : m_config(config),
    m_setup(false),
    m_started(false),
    m_video_track(0),
    m_sound_track(0),
    m_temporary_movie_file(),
    m_movie_file_path(),
    m_temporary_movie_file_path(),
    m_output_format(0),
    m_format_context(0),
    m_AV_compat_object(new AV_compat) // take care of any FFmpeg compatibility initialization
{
  if (!m_class_initialized)
  {
    m_class_initialized = true;
    // (TODO: consider adding only codecs supported?)
    av_register_all(); // initialize libavcodec, register all codecs and formats (check_code_igore)
  }

  postcondition(mf_invariant(false));
}

Movie::~Movie()
{
  precondition(mf_invariant(false));

  mf_cleanup();

  delete m_AV_compat_object; // take care of any FFmpeg compatibility de-initialization
}

//
// Accessors
//

const Frame_rate& Movie::video_frame_rate() const
{
  return m_video_track->frame_rate();
}

//
// Build member functions
//

bool Movie::setup(Sound_source* sound_source, Error_param& return_error)
{
  precondition(!m_setup
               && sound_source
               && !return_error()
               && mf_invariant());

  bool return_value = false;

  // used in error cleanup
  AVOutputFormat* output_format = 0;
  AVFormatContext* format_context = 0;
  Video_track* video_track = 0;
  Sound_track* sound_track = 0;

  start_error_block();

  // initialization for FFmpeg
  //   setup format
  const string extension = Movie_config::extension_for_movie_preset(m_config.movie_preset);
  output_format = AV_compat::guess_format(extension.c_str(),
                                          0,
                                          0);
  on_error(!output_format, new Build_error(error_location(),
                                           Build_error::setup,
                                           "",
                                           "",
                                           "movie extension: " + extension + ", format not supported"));
  format_context = avformat_alloc_context();
  on_error(!format_context, new Build_error(error_location(),
                                            Build_error::setup,
                                            "",
                                            "",
                                            "out of memory allocating format context"));
  format_context->oformat = output_format;

  Error_param error;

  // add and setup video track
  int track_ID = 0;
  Frame_rate video_frame_rate;
  const bool use_high_resolution_video_timescale = AV_compat::need_high_resolution_video_timescale();
  Movie_config::frame_rate_for_frame_rate_type(m_config.frame_rate,
                                               use_high_resolution_video_timescale,
                                               video_frame_rate);
  Rect_size output_frame_size;
  if (!Movie_config::output_size_for_movie_config(m_config, output_frame_size))
  {
    assert(false && "invalid movie config for output size");
  }
  video_track = new Video_track(track_ID,
                                video_frame_rate,
                                output_frame_size,
                                m_config.movie_preset,
                                m_config.keyframe_period);
  const bool video_track_setup = video_track->setup(format_context, error);
  on_error(!video_track_setup, new Build_error(error_location(),
                                               Build_error::general,
                                               error()));
  // add and setup sound track
  ++track_ID;
  sound_track = new Sound_track(track_ID,
                                video_frame_rate,
                                *sound_source);

  const bool sound_track_setup = sound_track->setup(format_context, error);
  on_error(!sound_track_setup, new Build_error(error_location(),
                                               Build_error::general,
                                               error()));

  // success, update data members
  mf_cleanup();
  //
  m_output_format = output_format;
  m_format_context = format_context;
  m_video_track = video_track;
  m_sound_track = sound_track;
  //
  m_setup = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (sound_track)
  {
    delete sound_track;
  }
  if (video_track)
  {
    delete video_track;
  }
  if (format_context)
  {
    av_free(format_context);
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Movie::start(const string& movie_path,
                  Sound_source* sound,
                  Error_param& return_error)
{
  precondition(m_setup
               && !m_started
               && sound
               && !return_error()
               && mf_invariant());

  bool need_video_end = false;
  bool need_sound_end = false;
  bool have_open_file = false;
  string temporary_file_path;
  bool return_value = false;

  start_error_block();

  Error_param error;

  const bool video_started = m_video_track->start(error);
  on_error(!video_started, new Build_error(error_location(),
                                   Build_error::general,
                                   error()));
  need_video_end = true;

  const bool sound_started = m_sound_track->start(error);
  on_error(!sound_started, new Build_error(error_location(),
                                           Build_error::general,
                                           error()));
  need_sound_end = true;

  // create a temporary movie file
  const Movie_preset_type movie_preset = m_config.movie_preset;
  const string file_extension = string(".") + Movie_config::extension_for_movie_preset(movie_preset);
  int file_descriptor;
  const bool created_temporary_file =
    File_manager::create_temporary_file(file_extension,
                                        File_manager::temporary_movie_file,
                                        temporary_file_path,
                                        file_descriptor,
                                        error);
  on_error(!created_temporary_file, new Build_error(error_location(),
                                                    Build_error::temporary_movie_create,
                                                    (("" == temporary_file_path) ?
                                                     string("<temporary movie path>")
                                                     : temporary_file_path),
                                                    "",
                                                    "",
                                                    error()));

  // open the temporary movie file for output
  const int opened_output_file = AV_compat::io_open(&m_format_context->pb,
                                                    temporary_file_path.c_str(),
                                                    AV_compat::write_only);
  close(file_descriptor); // don't need the file descriptor any more, ignore error
  on_error(opened_output_file, new Build_error(error_location(),
                                               Build_error::temporary_movie_create,
                                               temporary_file_path,
                                               "",
                                               "could not open file"));
  have_open_file = true;

  // reset dts position (needed for multiple movies)
  AV_compat::zero_dts(m_format_context, m_video_track->stream());

  // write stream header, if necessary
  const int wrote_header = AV_compat::write_header(m_format_context);
  on_error(wrote_header, new Build_error(error_location(),
                                         Build_error::temporary_movie_write,
                                         temporary_file_path,
                                         "",
                                         "could not write header"));

  // success, update data members
  m_movie_file_path = movie_path;
  m_temporary_movie_file_path = temporary_file_path;
  m_temporary_movie_file = file_descriptor;
  m_started = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (have_open_file)
  {
    AV_compat::io_close(m_format_context->pb); // ignore error
  }
  if (("" != temporary_file_path) && File_manager::path_exists(temporary_file_path))
  {
    unlink(temporary_file_path.c_str());
  }
  if (need_sound_end)
  {
    Error_param error(false);
    m_sound_track->end(error); // ignore error
  }
  if (need_video_end)
  {
    Error_param error(false);
    m_video_track->end(error); // ignore error
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(m_setup
                // success
                && ((return_value
                     && File_manager::path_exists(m_temporary_movie_file_path)
                     && m_started)
                    // failure
                    || (!return_value
                        && (("" == temporary_file_path)
                            || !File_manager::path_exists(temporary_file_path))
                        && !m_started))
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Movie::add(const uint8_t* raw_video_frame,
                Long_int video_start_frame,
                Long_int video_frame_count,
                Error_param& return_error)
{
  precondition(raw_video_frame
               && (video_start_frame >= 0)
               && (video_frame_count >= 0)
               && !return_error()
               && m_setup
               && m_started
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  if (video_frame_count < 1)
  {
    // nothing to do
    return_value = true;
    goto exit_point;
  }

  Error_param error;

  // write interleaved video/sound
  //   video
  const bool video_added = m_video_track->add(raw_video_frame,
                                              video_frame_count,
                                              error);
  on_error(!video_added, new Build_error(error_location(),
                                         Build_error::general,
                                         error()));
  //  sound
  const bool sound_added = m_sound_track->add(video_start_frame,
                                              video_frame_count,
                                              error);
  on_error(!sound_added, new Build_error(error_location(),
                                         Build_error::general,
                                         error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && m_started
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Movie::add_silent(const uint8_t* raw_video_frame,
                       Long_int video_frame_count,
                       Error_param& return_error)
{
  precondition(raw_video_frame
               && (video_frame_count >= 0)
               && !return_error()
               && m_setup
               && m_started
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  if (video_frame_count < 1)
  {
    // nothing to do
    return_value = true;
    goto exit_point;
  }

  Error_param error;

  // write interleaved video/sound
  //   video
  const bool video_added = m_video_track->add(raw_video_frame,
                                              video_frame_count,
                                              error);
  on_error(!video_added, new Build_error(error_location(),
                                         Build_error::general,
                                         error()));
  //  sound
  const bool sound_added = m_sound_track->add_silence(video_frame_count,
                                                      error);
  on_error(!sound_added, new Build_error(error_location(),
                                         Build_error::general,
                                         error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && m_started
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Movie::end(Error_param& return_error)
{
  precondition(m_setup
               && m_started
               && !return_error()
               && mf_invariant());

  bool need_end_sound = true;
  bool need_write_trailer = true;
  bool need_close = true;
  bool return_value = false;

  start_error_block();

  Error_param error;

  const bool video_ended = m_video_track->end(error);
  on_error(!video_ended, new Build_error(error_location(),
                                                 Build_error::general,
                                                 error()));
  need_end_sound = false;
  const bool sound_ended = m_sound_track->end(error);
  on_error(!sound_ended, new Build_error(error_location(),
                                         Build_error::general,
                                         error()));

  need_write_trailer = false;
  const int write_trailer_return = av_write_trailer(m_format_context);
  on_error(write_trailer_return, new Build_error(error_location(),
                                                 Build_error::temporary_movie_write,
                                                 m_temporary_movie_file_path,
                                                 "",
                                                 "could not write movie trailer"));

  need_close = false;
  const int close_return = AV_compat::io_close(m_format_context->pb);
  on_error(close_return, new Build_error(error_location(),
                                         Build_error::temporary_movie_close,
                                         m_temporary_movie_file_path,
                                         "",
                                         "could not close temporary movie"));
  m_started = false;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (need_end_sound)
  {
    Error_param error(false);
    m_sound_track->end(error); // ignore error
  }
  if (need_write_trailer)
  {
    av_write_trailer(m_format_context); // ignore error
  }
  if (need_close)
  {
    AV_compat::io_close(m_format_context->pb); // ignore error
  }
  m_started = false;
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(m_setup
                && !m_started
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

Long_int Movie::estimate_video_frame_size(const uint8_t* raw_video_frame)
{
  precondition(m_setup
               && mf_invariant());

  const Long_int frame_size = m_video_track->estimate_frame_size(raw_video_frame);

  precondition(m_setup
               && mf_invariant());
  return frame_size;
}

Long_int Movie::estimate_segment_size(Long_int video_frame_size, Long_int video_frame_count)
{
  precondition(m_setup
               && (video_frame_size >= 0)
               && (video_frame_count >= 0)
               && mf_invariant());

  const Long_int video_size = m_video_track->estimate_segment_size(video_frame_size, video_frame_count);
  const Long_int audio_size = m_sound_track->estimate_segment_size(video_frame_count);

  const Long_int total_size = video_size + audio_size;

  postcondition((total_size >= 0)
                && m_setup
                && mf_invariant());
  return total_size;
}

//
// Protected member functions
//

bool Movie::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_AV_compat_object)
  {
   goto exit_point;
  }

  if ((m_setup &&
       (!m_video_track
        || !m_sound_track
        || !m_output_format
        || !m_format_context))
      || (!m_setup
          && (m_video_track
              || m_sound_track
              || m_output_format
              || m_format_context)))
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private class data members
//

bool Movie::m_class_initialized = false;

//
// Private member functions
//

void Movie::mf_cleanup()
{
  if (m_started)
  {
    av_write_trailer(m_format_context); // ignore error
    AV_compat::io_close(m_format_context->pb);
    m_started = false;
  }
  if (m_sound_track)
  {
    delete m_sound_track;
    m_sound_track = 0;
  }
  if (m_video_track)
  {
    delete m_video_track;
    m_video_track = 0;
  }
  m_output_format = 0;
  if (m_format_context)
  {
    av_free(m_format_context);
    m_format_context = 0;
  }
}
