// Level_meter_view.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

//
// This code was based on jackmeter.c:
//
//   jackmeter.c
//   Simple console based Digital Peak Meter for JACK
//   Copyright (C) 2005  Nicholas J. Humfrey
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#ifndef LEVEL_METER_VIEW_HPP_
#define LEVEL_METER_VIEW_HPP_

#include "View.hpp"
#include <vector>

namespace Roan_trail
{
    class Level_meter_view : public View
    {
    public:
      // constructor/destructor/copy
      Level_meter_view(int channels,
                       int update_rate, // updates per second
                       bool peak_meter_mode = false,
                       bool vertical = false,
                       bool model_ballistics = true,
                       int attack_period = -1, // milliseconds for 99% deflection
                       int decay_period = -1,  // milliseconds for 99% return
                       int peak_hold_period = -1); // milliseconds to hold peak
      virtual ~Level_meter_view();
      // helpers
      static double scale_IEC(double dB);
      double convert_level_to_dB(double level) const;
      // accessors
      int channels() const { return m_channels; }
      bool peak_meter_mode() const { return m_peak_meter_mode; }
      double level(int channel) const { return m_levels[channel]; } // instantaneous level
      double scaled_level(int channel) const { return m_scaled_levels[channel]; } // levels with ballistics
      double peak(int channel) const { return m_peak[channel]; }
      bool clipped(int channel) const { return m_clipped[channel]; }
      bool vertical() const { return m_vertical; }
      // mutators
      void set_vertical(bool value) { m_vertical = value; }
      void set_peak_meter_mode(bool peak_meter_mode); // peak (peak_meter_mode == true) or RMS meter
      virtual void set_levels(const std::vector<double>& levels);
      virtual void update();
      void reset_clipped();
      // class constants
      static const int default_attack_period;
      static const int default_decay_period;
      static const int default_peak_hold_period;
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      int m_channels;
      bool m_vertical; // true, vertical meters / false, horizontal
      bool m_peak_meter_mode; // true, peak mode / false, RMS mode
      bool m_model_ballistics;
      std::vector<double> m_levels;
      std::vector<double> m_scaled_levels;
      std::vector<double> m_peak;
      std::vector<bool> m_clipped;
      //
      double m_attack_delta;
      double m_decay_delta;
      int m_peak_hold_count;
      std::vector<int> m_peak_hold_counts;
      //
      void mf_set_channels();
      void mf_clear_vectors();
      void mf_set_rates(int update_rate,
                        int attack_period,
                        int decay_period,
                        int peak_hold_period);
      // prevent compiler from generating
      Level_meter_view(const Level_meter_view& view);
      Level_meter_view& operator=(const Level_meter_view& view);
    };
}

// Notes on meter styles and ballistics:
//
// (1) VU style meter:
//     model_ballistics = true
//     attack_period = 300
//     decay_period = 300
//
// (2) PPM style meter
//     model_ballistics = true
//     attack_period = 10
//     decay_period = 3000
//
// (3) Instantaneous update meter
//     model_ballistics = false
//
// The default for this class is (1).

#endif // LEVEL_METER_VIEW_HPP_
