// File_manager.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef FILE_MANAGER_HPP_
#define FILE_MANAGER_HPP_

#include "common.hpp"
#include <string>
#include <vector>

namespace Roan_trail
{
  class Error_param;

  class File_manager
  {
  public:
    // typedef
    typedef int Temporary_file_type;
    // class member functions
    //   predicates
    static bool path_exists(const std::string& path_arg);
    static bool path_is_directory(const std::string& path_arg);
    static bool path_is_writable(const std::string& path_arg);
    //   accessors
    static std::string temporary_directory();
    //   file operations
    static bool create_temporary_file(const std::string& file_extension,
                                      Temporary_file_type file_type,
                                      std::string& return_file_name,
                                      int& return_file_descriptor,
                                      Error_param& return_error);
    static bool move_file(const std::string& source_file_path,
                          const std::string& dest_file_path,
                          bool overwrite_dest,
                          Error_param& return_error);
    static bool move_file_with_backup(const std::string& source_file_path,
                                      const std::string& dest_file_path,
                                      bool overwrite_dest,
                                      bool &return_backed_up,
                                      std::string &return_backup_file_path,
                                      Error_param& return_error);
    static std::string backup_path_for_original(const std::string& original_path);
    static bool remove_file(const std::string& file_path,
                            Error_param& return_error);
    //   file system attributes
    static bool file_size_for_path(const std::string& path_arg,
                                   Long_int& return_file_size,
                                   Error_param& return_error);
    static bool file_system_available_for_path(const std::string& path_arg,
                                               double& return_fraction_available,
                                               Long_int& return_available_bytes,
                                               Error_param& return_error);
    static bool contents_of_directory_at_path(const std::string& path_arg,
                                              std::vector<std::string>& return_paths);
    //   file name helpers
    static std::string parent_directory_for_path(const std::string& path_arg);
    static std::string file_extension_for_file(const std::string& file_path);
    static std::string file_name_for_file_path(const std::string& file_path);
    static std::string replace_file_extension(const std::string& file_path, const std::string& extension = "");
    static std::string system_absolute(const std::string& file_path);
    //   misc
    static bool executable_dir(std::string& return_dir);
    // class constants
    static const Temporary_file_type temporary_file       = 0;
    static const Temporary_file_type temporary_movie_file = 1;
    static const Temporary_file_type temporary_sound_file = 2;
  private:
    // prevent the compiler from generating
    File_manager();
    File_manager(const File_manager& manager);
    File_manager& operator=(const File_manager& manager);
  };
}

#endif // FILE_MANAGER_HPP_
