// Kinetophone_vox_config.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_vox_config.hpp"
#include "../base/Speech_synthesizer_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/File_manager.hpp"
#include "../base/error/Boost_error.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <boost/program_options.hpp>
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <cstdlib>

using std::endl;
using std::ifstream;
using std::string;
using std::stringstream;
using std::ostream;
using std::vector;
using Roan_trail::File_manager;
using Roan_trail::Recorder::Sound_file_config;
using Roan_trail::Recorder::Speech_synthesizer_config;
using namespace Roan_trail::Kinetophone;
namespace PO = boost::program_options;

//
// Internal helpers
//

namespace
{
  // option validation support
  template<class T> struct Ih_validator
  {
    BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
    BOOST_CONCEPT_ASSERT((boost::DefaultConstructible<T>));
    Ih_validator(const T& a) : v(a) {}
    Ih_validator() : v(T()) {}
    T v;
  };

  template<class T> void validate(boost::any& v, const vector<string>& xs, Ih_validator<T>*, int)
  {
    using namespace PO;
    check_first_occurrence(v);
    v = boost::any(Ih_validator<T>(boost::lexical_cast<T>(get_single_string(xs))));
  }

  template<class T> ostream& operator<<(ostream& os, const Ih_validator<T>& c)
  {
    return os << c.v;
  }

  ostream& operator<<(ostream& os, const Ih_validator<bool>& c)
  {
    return os << (c.v ? "yes" : "no");
  }

  void validate(boost::any& v,
                const std::vector<std::string>& values,
                Ih_validator<bool>*, int)
  {
    using namespace PO;

    check_first_occurrence(v);
    string s(get_single_string(values, true));

    for (string::size_type i = 0; i < s.size(); ++i)
    {
      s[i] = char(tolower(s[i]));
    }

    if (s.empty() || (s == "on") || (s == "yes") || (s == "1") || (s == "true"))
    {
      v = boost::any(Ih_validator<bool>(true));
    }
    else if ((s == "off") || (s == "no") || (s == "0") || (s == "false"))
    {
      v = boost::any(Ih_validator<bool>(false));
    }
    else
    {
      throw validation_error(validation_error::invalid_bool_value, s);
    }
  }

  typedef string (*Ih_lookup_ptr)(const string&);

  template<class T, Ih_lookup_ptr F> struct Ih_lookup_validator
  {
    BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
    Ih_lookup_validator(const T& a) : v(a) {}
    T v;
  };

  template<class T, Ih_lookup_ptr F> ostream& operator<<(ostream& os,
                                                         const Ih_lookup_validator<T, F>& c)
  {
    return os << c.v;
  }

  template<class T, Ih_lookup_ptr F> void validate(boost::any& v,
                const std::vector<std::string>& values,
                Ih_lookup_validator<T, F>*, int)
  {
    using namespace PO;

    validators::check_first_occurrence(v);
    // error if more than one string supplied
    const string& s = validators::get_single_string(values);

    if ("" != F(s))
    {
      v = boost::any(Ih_lookup_validator<T, F>(s));
    }
    else
    {
      throw validation_error(validation_error::invalid_option_value);
    }
  }

  typedef struct Ih_lookup_validator<string, Speech_synthesizer_config::setup_command_for_voice>
  Ih_voice_validator;

  template<class T, int i> struct Ih_cond_validator
  {
    BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
    BOOST_CONCEPT_ASSERT((boost::DefaultConstructible<T>));
    Ih_cond_validator(const T& a) : v(a) {}
    Ih_cond_validator() : v(T()) {}
    bool cond() const;
    T v;
  };

  template<class T, int i> ostream& operator<<(ostream& os, const Ih_cond_validator<T, i>& c)
  {
    return os << c.v;
  }

  template<class T, int i> void validate(boost::any& v,
                                         const std::vector<std::string>& values,
                                         Ih_cond_validator<T, i>*, int)
  {
    using namespace PO;

    check_first_occurrence(v);
    string s(get_single_string(values, true));

    T a = boost::lexical_cast<T>(s);
    Ih_cond_validator<T, i> cv(a);

    if (cv.cond())
    {
      v = boost::any(cv);
    }
    else
    {
      throw validation_error(validation_error::invalid_option_value);
    }
  }

  typedef struct Ih_cond_validator<int, 0> Ih_pos_int_validator;
  template<> bool Ih_pos_int_validator::cond() const
  {
    return v > 0;
  }
  typedef struct Ih_cond_validator<double, 0> Ih_fraction_validator;
  template<> bool Ih_fraction_validator::cond() const
  {
    return ((v >= 0.0) && (v <= 1.0));
  }

  struct Ih_validator_struct
  {
    Ih_validator_struct(const Kinetophone_vox_config& s)
      : image_directory(s.image_directory),
        vox_generation(s.vox_generation),
        detailed_error(Ih_validator<bool>(s.detailed_error)),
        quit_on_pregenerate_warnings(Ih_validator<bool>(s.quit_on_pregenerate_warnings)),
        custom_installation_dir(s.custom_installation_dir),
        speech_synthesizer_config(*s.speech_synthesizer_config) {}
    void update(Kinetophone_vox_config& return_cfg)
    {
      return_cfg.image_directory = image_directory;
      return_cfg.detailed_error = detailed_error.v;
      vox_generation.update(return_cfg.vox_generation);
      return_cfg.detailed_error = detailed_error.v;
      return_cfg.quit_on_pregenerate_warnings = quit_on_pregenerate_warnings.v;
      return_cfg.custom_installation_dir = custom_installation_dir;
      speech_synthesizer_config.update(*return_cfg.speech_synthesizer_config);
    }
    string image_directory;
    struct Vox_generation_validator_struct
    {
      Vox_generation_validator_struct(const Kinetophone_vox_config::Vox_generation_struct& s)
        : input_session_file(s.input_session_file),
          output_session_file(s.output_session_file),
          quiet(Ih_validator<bool>(s.quiet)),
          verbose(Ih_validator<bool>(s.verbose)),
          skip_generate_warnings(Ih_validator<bool>(s.skip_generate_warnings)),
          overwrite_output_session_file(Ih_validator<bool>(s.overwrite_output_session_file)),
          output_empty_slides(Ih_validator<bool>(s.output_empty_slides)) {}
      void update(Kinetophone_vox_config::Vox_generation_struct& return_s)
      {
        return_s.input_session_file = input_session_file;
        return_s.output_session_file = output_session_file;
        return_s.quiet = quiet.v;
        return_s.verbose = verbose.v;
        return_s.skip_generate_warnings = skip_generate_warnings.v;
        return_s.overwrite_output_session_file = overwrite_output_session_file.v;
        return_s.output_empty_slides = output_empty_slides.v;
      }
      string input_session_file;
      string output_session_file;
      Ih_validator<bool> quiet;
      Ih_validator<bool> verbose;
      Ih_validator<bool> skip_generate_warnings;
      Ih_validator<bool> overwrite_output_session_file;
      Ih_validator<bool> output_empty_slides;
    } vox_generation;
    Ih_validator<bool> detailed_error;
    Ih_validator<bool> quit_on_pregenerate_warnings;
    string custom_installation_dir;
    struct Speech_synthesizer_config_validator_struct
    {
      Speech_synthesizer_config_validator_struct(const Speech_synthesizer_config& s)
        : voice(Ih_voice_validator(s.voice)),
          sound_file(*s.sound_file),
          file_overwrite(s.file_overwrite) {}
      void update(Speech_synthesizer_config& return_s)
      {
        return_s.voice = voice.v;
        sound_file.update(*return_s.sound_file);
        return_s.file_overwrite = file_overwrite.v;
      }
      //
      Ih_voice_validator voice;
      struct Sound_file_config_validator_struct
      {
        Sound_file_config_validator_struct(const Sound_file_config& s)
          : sample_rate(Ih_pos_int_validator(static_cast<int>(s.sample_rate))),
            file_name(s.file_name) {}
        void update(Sound_file_config& return_s)
        {
          return_s.sample_rate = sample_rate.v;
          return_s.file_name = file_name;
        }
        //
        Ih_pos_int_validator sample_rate;
        string file_name;
      } sound_file;
      Ih_validator<bool> file_overwrite;
    } speech_synthesizer_config;
  };

  //
  // Auxiliary functions for checking input validity.
  //

  // Function used to check that 'opt1' and 'opt2' are not specified
  // at the same time.
  void ih_check_conflicting_options(const PO::variables_map& VM,
                                    const char* opt1,
                                    const char* opt2,
                                    bool condition = true,
                                    const string& message = "")
  {
    precondition(opt1
                 && opt2);

    if (condition &&
        (VM.count(opt1) && !VM[opt1].defaulted()
         && VM.count(opt2) && !VM[opt2].defaulted()))
    {
      string exception_message = string("conflicting options '")
        + opt1 + "' and '" + opt2 + "'";
      if ("" != message)
      {
        exception_message += string(": ") + message;
      }
      exception_message += ".";
      throw std::logic_error(exception_message);
    }
  }

  // function used to check that 'for_what' is specified, then
  // 'required_option' is specified too.
  void ih_check_option_dependency(const PO::variables_map& VM,
                                  const char* for_what,
                                  const char* required_option,
                                  bool condition = true)
  {
    precondition(for_what
                 && required_option);

    if (VM.count(for_what) && !VM[for_what].defaulted())
    {
      if ((VM.count(required_option) == 0)
          || VM[required_option].defaulted()
          || !condition)
      {
        throw std::logic_error(string("option '") + for_what
                               + "' requires option '" + required_option + "'.");
      }
    }
  }

  void ih_check_option_consistency(const Kinetophone_vox_config& cfg,
                                   const PO::variables_map& VM)
  {
    // conflicting option checks
    ih_check_conflicting_options(VM,
                                 "quiet",
                                 "verbose");
    // required option checks
    if ("" == cfg.vox_generation.input_session_file)
    {
      throw std::logic_error(string("input session file not specified"));
    }
    if ("" == cfg.vox_generation.output_session_file)
    {
      throw std::logic_error(string("output session file not specified"));
    }
  }

  typedef struct
  {
    PO::options_description& cmd_line_desc;
    PO::options_description& basic_desc;
    PO::options_description& advanced_desc;
  } Ih_desc_params;

  // sets up the program option description object
  void ih_setup_descriptions(Ih_desc_params& return_desc_params,
                             Ih_validator_struct& return_v,
                             string& return_config_file)
  {
    const char *home_dir = getenv("HOME"); // (check_code_ignore)
    string default_config_file = (!home_dir || (*home_dir == 0) || !File_manager::path_is_directory(home_dir)) ?
      string("") : string(home_dir) + string("/.kinetophone");

    return_desc_params.cmd_line_desc.add_options()
      ("help,h", "output help message")
      ("config,i", PO::value<string>(&return_config_file)->default_value(default_config_file),
       "specify configuration file")
      ("version,v", "print program version")
      ;

    return_desc_params.basic_desc.add_options()
      ("voice,w", PO::value<Ih_voice_validator>(&return_v.speech_synthesizer_config.voice)
       ->default_value(return_v.speech_synthesizer_config.voice),
       "set voice for speech synthesizer\n"
       "{ en_br_m (British English male)\n"
       "| en_us_m (American English male)\n"
       "| en_us_m_alt (American English male alternate)\n"
       "| it_m (Italian male)\n"
       "| it_f (Italian female)\n"
       "| es_m (Spanish male)}")
      ("input-session-file,I", PO::value<string>(&return_v.vox_generation.input_session_file)
       ->default_value(""),
       "set input session file")
      ("output-session-file,O", PO::value<string>(&return_v.vox_generation.output_session_file)
       ->default_value(""),
       "set input session file")
      ("image-directory,D", PO::value<string>(&return_v.image_directory)
       ->default_value(return_v.image_directory),
       "set image source directory (overrides session)")
      ("sound-file,f", PO::value<string>(&return_v.speech_synthesizer_config.sound_file.file_name)
       ->default_value(return_v.speech_synthesizer_config.sound_file.file_name),
       "set destination sound file path")
      ("overwrite,o", PO::value<Ih_validator<bool> >(&return_v.speech_synthesizer_config.file_overwrite)
       ->implicit_value(Ih_validator<bool>(true)),
       "overwrite output sound recording file")
      ("overwrite-session-file,W",
       PO::value<Ih_validator<bool> >(&return_v.vox_generation.overwrite_output_session_file)
       ->implicit_value(Ih_validator<bool>(true)),
       "overwrite output session file")
      ("sample-rate,s",
       PO::value<Ih_pos_int_validator>(&return_v.speech_synthesizer_config.sound_file.sample_rate)
       ->default_value(return_v.speech_synthesizer_config.sound_file.sample_rate),
       "set samples per second (must be > 0)")
      ;

    return_desc_params.advanced_desc.add_options()
      ("skip-generate-warnings,k",
       PO::value<Ih_validator<bool> >(&return_v.vox_generation.skip_generate_warnings)
       ->implicit_value(Ih_validator<bool>(true)),
       "do not output vox generation warnings (cannot use with --quit-on-pregenerate-warnings)")
      ("quiet,Q", PO::value<Ih_validator<bool> >(&return_v.vox_generation.quiet)
       ->implicit_value(Ih_validator<bool>(true)),
       "little output, if any, from the vox generation (cannot use with --verbose)")
      ("verbose,V", PO::value<Ih_validator<bool> >(&return_v.vox_generation.verbose)
       ->implicit_value(Ih_validator<bool>(true)),
       "talkative vox generation (cannot use with --quiet)")
      ("quit-on-pregenerate-warnings,p",
       PO::value<Ih_validator<bool> >(&return_v.quit_on_pregenerate_warnings)
       ->implicit_value(Ih_validator<bool>(true)),
       "quit, do not continue, if there are pre-generate warning(s) " // (check_code_ignore)
       "(cannot use with --skip-generate-warnings)")
      ("detailed-error,e", PO::value<Ih_validator<bool> >(&return_v.detailed_error)
       ->implicit_value(Ih_validator<bool>(true)),
       "display detailed diagnostics on error")
      ("config-directory,C", PO::value<string>(&return_v.custom_installation_dir)
       ->default_value(return_v.custom_installation_dir),
       "set custom program configuration directory")
      ("output-empty-slides,E", PO::value<Ih_validator<bool> >(&return_v.vox_generation.output_empty_slides)
       ->implicit_value(Ih_validator<bool>(true)),
       "output slides with no notes as segments")
      ;
  }
}

//
// Constructor/destructor/copy
//

Kinetophone_vox_config::Kinetophone_vox_config()
  // setup some reasonable defaults
  : help_message(""),
    image_directory(""), // left blank to indicate that the directory in the session file should be used
    vox_generation(Vox_generation_struct()),
    detailed_error(false),
    quit_on_pregenerate_warnings(false),
    custom_installation_dir(""),
    speech_synthesizer_config(new Speech_synthesizer_config)
{
}

Kinetophone_vox_config::~Kinetophone_vox_config()
{
  delete speech_synthesizer_config;
}

Kinetophone_vox_config::Vox_generation_struct::Vox_generation_struct()
  // setup some reasonable defaults
  : input_session_file(""),
    output_session_file(""),
    quiet(false),
    verbose(false),
    skip_generate_warnings(false),
    overwrite_output_session_file(false),
    output_empty_slides(false)
{
}

//
//  Class constants
//

//   default
//     command
const int Kinetophone_vox_config::command_error;
const int Kinetophone_vox_config::command_synthesize;
const int Kinetophone_vox_config::command_help;
const int Kinetophone_vox_config::command_output_version;

int Kinetophone_vox_config::parse_program_options(int argc,
                                                       const char** argv,
                                                       Error_param& return_error)
{
  precondition(!return_error());

  int return_value = Kinetophone_vox_config::command_error;

  help_message = "";

  start_error_block();

  try
  {
    PO::options_description cmd_line_desc("Command line only");
    stringstream basic_desc_label;
    basic_desc_label << "Command line and configuration file" << endl << "  Basic";
    PO::options_description basic_desc(basic_desc_label.str());
    PO::options_description advanced_desc("  Advanced");
    Ih_desc_params desc_params =
    {
      cmd_line_desc,
      basic_desc,
      advanced_desc
    };

    Ih_validator_struct validator(*this);
    string config_file;
    ih_setup_descriptions(desc_params,
                          validator,
                          config_file);

    // parse the command line to get config file
    PO::variables_map VM_config;
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(cmd_line_desc).allow_unregistered().run(), VM_config);
    notify(VM_config);

    PO::options_description config_desc;
    config_desc.add(basic_desc).add(advanced_desc);

    stringstream help_stream;
    help_stream << "Usage: " << endl;
    help_stream << "kinetophone_vox --help" << endl << endl;
    help_stream << "kinetophone_vox --list-devices" << endl << endl;
    help_stream << "kinetophone_vox input_session_file output_session_file [sound_file ] [ options ]";
    help_stream << endl << endl;

    PO::options_description all_desc("Options");
    all_desc.add(cmd_line_desc).add(basic_desc).add(advanced_desc);
    help_stream << all_desc;
    help_message = help_stream.str();

    PO::variables_map VM;

    // parse the command line (overrides config file)
    PO::positional_options_description pos_desc;
    pos_desc.add("input-session-file", 1);
    pos_desc.add("output-session-file", 1);
    pos_desc.add("sound-file", 1);
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(all_desc).positional(pos_desc).run(), VM);
    PO::notify(VM);

    // parse the config file
    if ("" != config_file)
    {
      ifstream ifs(config_file.c_str());
      bool have_error =  (!ifs && !VM_config["config"].defaulted());
      on_error(have_error, new Boost_error(error_location(), string("cannot open config file: ") + config_file));
      const bool allow_unregistered_options = true;
      PO::store(PO::parse_config_file(ifs, config_desc, allow_unregistered_options), VM);
      PO::notify(VM);
    }

    if (VM.count("help"))
    { // HELP
      return_value = Kinetophone_vox_config::command_help;
    }
    else if (VM.count("version"))
    {
      // PROGRAM VERSION
      return_value = Kinetophone_vox_config::command_output_version;
    }
    else
    { // SYNTHESIZE
      // update the config from the validator
      validator.update(*this);
      ih_check_option_consistency(*this, VM);
      return_value = Kinetophone_vox_config::command_synthesize;
    }
  }
  catch (const std::exception& e)
  {
    string diagnostic = e.what();
    stringstream s;
    s << "Reason: " << diagnostic << std::endl << std::endl;
    s << "Enter kinetophone_vox --help for valid program options.";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     new Boost_error(error_location(), s.str()));
    on_error(true, error);
  }
  catch (...)
  {
    string diagnostic = "exception parsing command line";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     diagnostic);
    on_error(true, error);
  }

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  return_value = Kinetophone_vox_config::command_error;
  goto exit_point;

 exit_point:
  return return_value;
}
