// Kinetophone_vox_app.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_vox_app.hpp"
#include "Kinetophone_vox_config.hpp"
#include "Kinetophone_vox_manager.hpp"
#include "Kinetophone_vox_model.hpp"
#include "Kinetophone_vox_view.hpp"
#include "../base/Logger.hpp"
#include "../base/Speech_synthesizer_config.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <memory>
#include <iostream>
#include <sstream>
#include <string>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION "(unknown)"
#endif

using std::cerr;
using std::endl;
using std::string;
using std::stringstream;
using Roan_trail::Logger;
using namespace Roan_trail::Kinetophone;

//
// Constructor/destructor
//

Kinetophone_vox_app::Kinetophone_vox_app(int argc, const char** argv)
  : Application(argc, argv),
    m_config(new Kinetophone_vox_config),
    m_model(new Kinetophone_vox_model(*m_config, "directory")),
    m_vox_manager(new Kinetophone_vox_manager(*m_config, *m_model))
{
  precondition(argv);

  postcondition(mf_invariant(false));
}

Kinetophone_vox_app::~Kinetophone_vox_app()
{
  precondition(mf_invariant(false));

  delete m_vox_manager;
  delete m_model;
  delete m_config;
}

int Kinetophone_vox_app::run()
{
  precondition(mf_invariant());

  int return_value = 1;
  Error_param error;
  int user_error_code = 0;

  start_error_block();

  // see what the user wants to do
  int vox_command = m_config->parse_program_options(argc(), argv(), error);
  on_error(Kinetophone_vox_config::command_error == vox_command,
           new Kinetophone_error(error_location(),
                                 Kinetophone_error::command_line,
                                 error()));

  // update the installation directory if the program options override the default one
  if ("" != m_config->custom_installation_dir)
  {
    Application::application()->set_installation_dir(m_config->custom_installation_dir);
  }

  switch (vox_command)
  {
  case Kinetophone_vox_config::command_help:
    // HELP
    Kinetophone_vox_view::output_message(m_config->help_message);
    break;
  case Kinetophone_vox_config::command_output_version:
    // VERSION
    {
      stringstream version;
      version << "Kinetophone Vox Version " << string(PACKAGE_VERSION) << endl;
      Kinetophone_vox_view::output_message(version.str());
    }
    break;
  case Kinetophone_vox_config::command_synthesize:
    // SYNTHESIZE
    {
      Error_param synthesize_error;
      const bool synthesize_success = mf_synthesize(synthesize_error);
      on_error(!synthesize_success, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::general,
                                                       synthesize_error()));
    }
    break;
  default:
    assert(0 && "Invalid command");
    break;
  }

  return_value =  0;
  goto exit_point;

  end_error_block();

 error_handler:
  {
    string error_message;
    user_error_code = handler_error->format_message_for_chain(m_config->detailed_error, error_message);
    Kinetophone_vox_view::output_error(error_message);
  }
  goto error_cleanup;

 error_cleanup:
  delete handler_error;
  return_value = user_error_code;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());
  return return_value;
}

void Kinetophone_vox_app::terminate(int code, const string& message)
{
  precondition(mf_invariant());

  // Note: when threading added need to call: m_vox_manager->cancel();
  //       and wait for it to cancel (with timeout)

  Application::terminate(code, message);

  // no postcondition, above terminate() call does not return
}

//
// Protected member functions
//

bool Kinetophone_vox_app::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  // object allocation checking
  if (!m_config
      || !m_model
      || !m_vox_manager)
  {
    goto exit_point;
  }

  return_value = true;

 exit_point:
  return return_value;
}

//
// Private member functions
//

bool Kinetophone_vox_app::mf_synthesize(Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  Error_param error;

  start_error_block();

  Logger& logger = Logger::default_logger();
  if (m_config->vox_generation.quiet)
  {
    logger.set_output_level(Logger::quiet);
  }
  else if (m_config->vox_generation.verbose)
  {
    logger.set_output_level(Logger::verbose);
  }
  else
  {
    logger.set_output_level(Logger::normal);
  }

  logger << Logger::info << "Loading input session file: " << m_config->vox_generation.input_session_file;
  logger << endl;

  const bool model_loaded = m_model->load_from_XML(m_config->vox_generation.input_session_file, error);
  on_error(!model_loaded, new Kinetophone_error(error_location(),
                                            Kinetophone_error::general,
                                            error()));
  
  Kinetophone_vox_view view;

  const bool setup_success = m_vox_manager->setup_for_generate(error);
  on_error(!setup_success, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::general,
                                                 error()));
  if (!m_vox_manager->pregenerate_checks(error))
  {
    logger << Logger::warning;
    logger << "Not continuing after warning(s)" << endl;
    return_value = true;
    goto exit_point;
  }

  // TODO: check this:
  // Note: minimize heap allocation after this point, should be limited to error handling
  //   and library functions (Ref. Standards Rule #206)

  const bool synth_success = m_vox_manager->generate(error);
  on_error(!synth_success, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::general,
                                                 error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  return return_value;
}

//////////
// main //
//////////

int main (int argc, const char **argv)
{
  Kinetophone_vox_app application(argc, argv);

  int return_code = application.run();

  return return_code;
}
