// Kinetophone_dir_slide_collection.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_dir_slide_collection.hpp"

// include *mm first to avoid conflicts
#include <gdkmm/pixbuf.h>
#include <gtkmm/invisible.h>
#include <gtkmm/stock.h>

#include "../base/Dir_slide_collection.hpp"
#include "../base/Slide.hpp"
#include "../base/File_manager.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <string>
#include <vector>

using Glib::RefPtr;
using Gdk::Pixbuf;
using std::string;
using std::vector;
using Roan_trail::Error_param;
using Roan_trail::Rect_size;
using Roan_trail::Source::Dir_slide_collection;
using Roan_trail::Source::Slide;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//

namespace
{
  const int ic_thumbnail_size = 64;
  RefPtr<Pixbuf> ih_broken_image()
  {
    Gtk::Invisible invisible_widget;
    return invisible_widget.render_icon(Gtk::Stock::MISSING_IMAGE, Gtk::ICON_SIZE_BUTTON);
  }
}

//
// Accessors
//

bool Kinetophone_dir_slide_collection::image_for_slide_at(int slide_index, RefPtr<Pixbuf>& return_image) const
{
  precondition(((slide_index >= 0) && (slide_index < slide_count()))
               && !return_image
               && mf_invariant());

  bool return_value = false;

  const Slide<RefPtr<Pixbuf> >* slide = m_slides[slide_index];

  try
  {
    return_image = Pixbuf::create_from_file(slide->path());
    return_value = true;
  }
  catch (...)
  {
    return_image = ih_broken_image();
  }

  postcondition(return_image
                && mf_invariant());

  return return_value;
}

void Kinetophone_dir_slide_collection::thumbnail_for_slide_at(int slide_index,
                                                              RefPtr<Pixbuf>& return_image) const
{
  precondition(((slide_index >= 0) && (slide_index < slide_count()))
               && !return_image
               && mf_invariant());

  const Slide<RefPtr<Pixbuf> >* slide = m_slides[slide_index];

  try
  {
    return_image = Pixbuf::create_from_file(slide->path(),
                                            ic_thumbnail_size,
                                            ic_thumbnail_size,
                                            true);
  }
  catch (...)
  {
    return_image = ih_broken_image();
  }

  postcondition(return_image
                && mf_invariant());
}

Rect_size Kinetophone_dir_slide_collection::max_bounds() const
{
  Rect_size bounds(1, 1);
  for (int i = 0; i < slide_count(); ++i)
  {
    RefPtr<Pixbuf> image;
    image_for_slide_at(i, image);
    const int image_width = image->get_width();
    if (image_width > bounds.width)
    {
      bounds.width = image_width;
    }
    const int image_height = image->get_height();
    if (image_height > bounds.height)
    {
      bounds.height = image_height;
    }
  }

  return bounds;
}

//
// Protected member functions
//

bool Kinetophone_dir_slide_collection::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  return_value = !check_base_class
    || Dir_slide_collection<RefPtr<Pixbuf> >::mf_invariant(check_base_class);

  goto exit_point;

 exit_point:
  return return_value;
}

// implement virtual function for specific image (Gdk::Pixbuf in this case)
bool Kinetophone_dir_slide_collection::mf_process_source_images_with_notes(const vector<string>& image_paths,
                                                                           const string& source_path,
                                                                           Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = true;

  start_error_block();

  int slide_index = 0;
  for (vector<string>::const_iterator i = image_paths.begin(); i != image_paths.end(); ++i)
  {
    try
    {
      string file_path = source_path + string("/") + *i;
      RefPtr<Pixbuf> pixel_buffer = Gdk::Pixbuf::create_from_file(file_path);
      // loaded OK, add to the list of files which are supported
      m_slides.push_back(new Slide<RefPtr<Pixbuf> >(slide_index, "", file_path));
      pixel_buffer.reset(); // get rid of the object now that we are done with it
      ++slide_index;
    }
    catch (...)
    {
      // assume the file isn't a supported image
    }
  }
  // make sure we have at least one supported file
  const bool no_supported_files = (m_slides.size() == 0);
  on_error(no_supported_files, new Source_error(error_location(),
                                                Source_error::invalid_source,
                                                string("Source directory contains no valid images: ")
                                                + source_path));

  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}
