// Kinetophone_dbus_server.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_dbus_server.hpp"
#include "Kinetophone_dbus_recorder.hpp"
#include "../base/error/DBus_error.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include "../base/File_manager.hpp"
#include <dbus-c++/dbus.h>
#include <iostream>
#include <map>
#include <string>
#include <vector>
#include <csignal> // Standards Rule 21 deviation (check_code_ignore)

using DBus::BusDispatcher;
using DBus::Connection;
using std::cerr;
using std::endl;
using std::map;
using std::string;
using std::vector;
using Roan_trail::Kinetophone::Kinetophone_error;
using Roan_trail::Error;
using Roan_trail::DBus_error;
using Roan_trail::File_manager;
using namespace Roan_trail::Kinetophone;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION "(unknown)"
#endif

//
// Internal helpers
//

namespace
{
  const string ic_kinetophone_dbus_server_name = "com.roantrail.dbus.kinetophone";
  const string ic_kinetophone_dbus_server_path = "/com/roantrail/dbus/kinetophone";
  BusDispatcher *iv_dispatcher = 0;

  void ih_signal_handler(int s)
  {
    assert(iv_dispatcher && "error, dispatcher object not created");
    iv_dispatcher->leave();
  }
}

//
// Constructor, destructor
//

Kinetophone_dbus_server::Kinetophone_dbus_server(Connection& connection)
  : ObjectAdaptor(connection, ic_kinetophone_dbus_server_path.c_str()),
    m_recorder_ID(0)
{
  postcondition(mf_invariant(false));
}


//
// DBus interface
//

//
//   control (messages)
//

void Kinetophone_dbus_server::create_recorder(string& return_recorder_path,
                                              bool& return_have_error,
                                              Error_dictionary& return_error)
{
  precondition(mf_invariant());

  return_have_error = false;

  start_error_block();
  handler_error = 0; // avoid unused variable warning

  if (m_recorders.size() > 0)
  {
    const string diagnostic = "A recorder has already been created (only one is allowed).";
    const Kinetophone_error *server_error = new Kinetophone_error(error_location(),
                                                                  Kinetophone_error::server,
                                                                  diagnostic);
    server_error->format_chain_as_dictionary(return_error);
    delete server_error;
    return_have_error = true;
    goto exit_point;
  }

  ++m_recorder_ID;

  const string recorder_path = Kinetophone_dbus_recorder::dbus_path(m_recorder_ID);
  Kinetophone_dbus_recorder *recorder = 0;
  try
  {
    recorder = new Kinetophone_dbus_recorder(conn(), recorder_path);
  }
  catch (const DBus::Error& dbus_error)
  {
    const Kinetophone_error *server_error = new Kinetophone_error(error_location(),
                                                                  Kinetophone_error::server,
                                                                  new DBus_error(error_location(),
                                                                                 dbus_error));
    server_error->format_chain_as_dictionary(return_error);
    delete server_error;
    return_have_error = true;
    goto exit_point;
  }

  m_recorders[recorder_path] = recorder;

  return_recorder_path = recorder_path;

  goto exit_point;

  end_error_block();

 exit_point:
  postcondition((m_recorders.size() > 0)
                && mf_invariant());
}

void Kinetophone_dbus_server::remove_recorder(const string& recorder_path,
                                              bool& return_have_error,
                                              Error_dictionary& return_error)
{
  precondition(mf_invariant());

  return_have_error = false;

  if (m_recorders.find(recorder_path) != m_recorders.end())
  {
    Kinetophone_dbus_recorder *recorder = m_recorders[recorder_path];
    recorder->remove(return_have_error, return_error);
    m_recorders.erase(recorder_path);
    delete recorder;
  }
  else
  {
    const string diagnostic = "Recorder path not found: " + recorder_path;
    const Kinetophone_error *server_error = new Kinetophone_error(error_location(),
                                                                  Kinetophone_error::server,
                                                                  diagnostic);
    server_error->format_chain_as_dictionary(return_error);
    delete server_error;
    return_have_error = true;
  }

  postcondition(mf_invariant());
}

void Kinetophone_dbus_server::list_recorders(vector<string>& return_recorder_paths,
                                             bool& return_have_error,
                                             Error_dictionary& return_error)
{
  precondition(mf_invariant());

  return_have_error = false;

  return_recorder_paths.clear();

  for (map<string, Kinetophone_dbus_recorder*>::const_iterator i = m_recorders.begin();
       i != m_recorders.end();
       ++i)
  {
    return_recorder_paths.push_back(i->first);
  }

  postcondition((m_recorders.size() == return_recorder_paths.size())
                && mf_invariant());
}

//
// Protected member functions
//

bool Kinetophone_dbus_server::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (m_recorder_ID < 0)
  {
    goto exit_point;
  }

  return_value = true;;
  goto exit_point;

 exit_point:
  return return_value;
}


//////////
// main //
//////////

int main(int arc, char* argv[])
{
  cerr << "*** Experimental Kinetophone DBus server, version " << PACKAGE_VERSION << " ***" << endl;
  try
  {
    iv_dispatcher = new BusDispatcher;
  }
  catch (const DBus::Error& e)
  {
    cerr << "Exception encountered creating bus dispatcher: " << e.what() << endl;;
    goto exit_point;
  }

  {
    // set the installation directory for error handling
    string directory;
    if (File_manager::executable_dir(directory))
    {
      directory += "/../share/kinetophone";
      if (!File_manager::path_exists(directory))
      {
        directory = ".";
      }
    }
    else
    {
      directory = ".";
    }
    Error::set_error_directory(directory);

    signal(SIGTERM, ih_signal_handler);
    signal(SIGINT, ih_signal_handler);

    try
    {
      // TODO: when using separate threads  DBus::_init_threading();

      DBus::default_dispatcher = iv_dispatcher;
      Connection connection = Connection::SessionBus();
      try
      {
        connection.request_name(ic_kinetophone_dbus_server_name.c_str());
      }
      catch (const Error& e)
      {
        cerr << "Error encountered requesting name: " << ic_kinetophone_dbus_server_name << endl;
        goto exit_point;
      }

      Kinetophone_dbus_server server(connection);

      DBus::default_dispatcher->enter();
    }
    catch (const DBus::Error& dbus_error)
    {
      Error::Error_dictionary dictionary;
      const Kinetophone_error* server_error = new Kinetophone_error(error_location(),
                                                                    Kinetophone_error::server,
                                                                    new DBus_error(error_location(),
                                                                                   dbus_error));
      server_error->format_chain_as_dictionary(dictionary);
      cerr << "Error: " << dictionary[Error::brief_description_error_key] << endl;
      cerr << "Code: " << dictionary[Error::code_number_error_key];
      cerr << " (" << dictionary[Error::code_string_error_key] << ")" << endl;
      cerr << "Details: " << dictionary[Error::detailed_description_error_key] << endl;
      cerr << "Recovery suggestion: " << dictionary[Error::recovery_description_error_key] << endl;
      delete server_error;
      goto exit_point;
    }
  }
  goto exit_point;

 exit_point:
  delete iv_dispatcher;
  return 0;
}
