// Kinetophone_console_config.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_console_config.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/Level_meter_view.hpp"
#include "../base/File_manager.hpp"
#include "../base/error/Boost_error.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <boost/program_options.hpp>
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <cstdlib>

using std::endl;
using std::ifstream;
using std::ostream;
using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::File_manager;
using Roan_trail::Recorder::Sound_file_config;
using namespace Roan_trail::Kinetophone;

namespace PO = boost::program_options;

//
// Internal helpers
//

namespace
{
  using Roan_trail::Recorder::Sound_recorder_config;

  // option validation support
  template<class T> struct Ih_validator
  {
    BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
    BOOST_CONCEPT_ASSERT((boost::DefaultConstructible<T>));
    Ih_validator(const T& a) : v(a) {}
    Ih_validator() : v(T()) {}
    T v;
  };

  template<class T> void validate(boost::any& v, const vector<string>& xs, Ih_validator<T>*, int)
  {
    using namespace PO;
    check_first_occurrence(v);
    v = boost::any(Ih_validator<T>(boost::lexical_cast<T>(get_single_string(xs))));
  }

  template<class T> ostream& operator<<(ostream& os, const Ih_validator<T>& c)
  {
    return os << c.v;
  }

  ostream& operator<<(ostream& os, const Ih_validator<bool>& c)
  {
    return os << (c.v ? "yes" : "no");
  }

  void validate(boost::any& v,
                const std::vector<std::string>& values,
                Ih_validator<bool>*, int)
  {
    using namespace PO;

    check_first_occurrence(v);
    string s(get_single_string(values, true));

    for (string::size_type i = 0; i < s.size(); ++i)
    {
      s[i] = char(tolower(s[i]));
    }

    if (s.empty() || (s == "on") || (s == "yes") || (s == "1") || (s == "true"))
    {
      v = boost::any(Ih_validator<bool>(true));
    }
    else if ((s == "off") || (s == "no") || (s == "0") || (s == "false"))
    {
      v = boost::any(Ih_validator<bool>(false));
    }
    else
    {
      throw validation_error(validation_error::invalid_bool_value, s);
    }
  }

  typedef bool (*Ih_lookup_ptr)(const string&, int&);

  template<class T, Ih_lookup_ptr F> struct Ih_lookup_validator
  {
    BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
    Ih_lookup_validator(const T& a) : v(a) {}
    T v;
  };

  template<class T, Ih_lookup_ptr F> ostream& operator<<(ostream& os,
                                                         const Ih_lookup_validator<T, F>& c)
  {
    return os << c.v;
  }

  template<class T, Ih_lookup_ptr F> void validate(boost::any& v,
                const std::vector<std::string>& values,
                Ih_lookup_validator<T, F>*, int)
  {
    using namespace PO;

    validators::check_first_occurrence(v);
    // error if more than one string supplied
    const string& s = validators::get_single_string(values);

    int type;
    if (F(s, type))
    {
      v = boost::any(Ih_lookup_validator<T, F>(s));
    }
    else
    {
      throw validation_error(validation_error::invalid_option_value);
    }
  }

  template<class T, int i> struct Ih_cond_validator
  {
    BOOST_CONCEPT_ASSERT((boost::CopyConstructible<T>));
    BOOST_CONCEPT_ASSERT((boost::DefaultConstructible<T>));
    Ih_cond_validator(const T& a) : v(a) {}
    Ih_cond_validator() : v(T()) {}
    bool cond() const;
    T v;
  };

  template<class T, int i> ostream& operator<<(ostream& os, const Ih_cond_validator<T, i>& c)
  {
    return os << c.v;
  }

  template<class T, int i> void validate(boost::any& v,
                                         const std::vector<std::string>& values,
                                         Ih_cond_validator<T, i>*, int)
  {
    using namespace PO;

    check_first_occurrence(v);
    string s(get_single_string(values, true));

    T a = boost::lexical_cast<T>(s);
    Ih_cond_validator<T, i> cv(a);

    if (cv.cond())
    {
      v = boost::any(cv);
    }
    else
    {
      throw validation_error(validation_error::invalid_option_value);
    }
  }

  bool ih_filter_file_type_string(const string& file_type_string, int& check_file_type)
  {
    bool return_value = false;

    if (Sound_file_config::file_type_for_string(file_type_string, check_file_type))
    {
      switch (check_file_type)
      {
      // of the ones supported by Kinetophone, only these are allowed on the command line
      case Sound_file_config::file_type_wav:
      case Sound_file_config::file_type_aiff:
      case Sound_file_config::file_type_flac:
      case Sound_file_config::file_type_ogg:
        return_value = true;
        break;
      default:
        break;
      }
    }

    return return_value;
  }

  typedef struct Ih_lookup_validator<string, ih_filter_file_type_string>
  Ih_file_type_validator;
  typedef struct Ih_lookup_validator<string, Sound_file_config::data_format_for_string>
  Ih_data_format_validator;
  typedef struct Ih_lookup_validator<string, Sound_file_config::endianness_for_string>
  Ih_endianness_validator;
  typedef struct Ih_cond_validator<int, 0> Ih_pos_int_validator;
  template<> bool Ih_pos_int_validator::cond() const
  {
    return v > 0;
  }
  typedef struct Ih_cond_validator<double, 0> Ih_fraction_validator;
  template<> bool Ih_fraction_validator::cond() const
  {
    return ((v >= 0.0) && (v <= 1.0));
  }

  struct Ih_validator_struct
  {
    Ih_validator_struct(const Kinetophone_console_config& s)
      : has_level_meter(Ih_validator<bool>(s.has_level_meter)),
        level_meter(s.level_meter),
        update_rate(Ih_pos_int_validator(s.update_rate)),
        show_file_size(Ih_validator<bool>(s.show_file_size)),
        show_available(Ih_validator<bool>(s.show_available)),
        show_audio_info(Ih_validator<bool>(s.show_audio_info)),
        show_overflow_count(Ih_validator<bool>(s.show_overflow_count)),
        detailed_error(Ih_validator<bool>(s.detailed_error)),
        custom_installation_dir(s.custom_installation_dir),
        sound_recorder_config(*s.sound_recorder_config) {}
    void update(Kinetophone_console_config& return_cfg)
    {
      return_cfg.has_level_meter = has_level_meter.v;
      level_meter.update(return_cfg.level_meter);
      return_cfg.update_rate = update_rate.v;
      return_cfg.show_file_size = show_file_size.v;
      return_cfg.show_available = show_available.v;
      return_cfg.show_audio_info = show_audio_info.v;
      return_cfg.show_overflow_count = show_overflow_count.v;
      return_cfg.detailed_error = detailed_error.v;
      return_cfg.custom_installation_dir = custom_installation_dir;
      sound_recorder_config.update(*return_cfg.sound_recorder_config);
      // calculate and assign remaining members
      string use_file_type = sound_recorder_config.sound_file.file_type.v;
      if ("" == use_file_type)
      {
        if ("" != sound_recorder_config.sound_file.file_name)
        {
          // file type not specified, but output file is
          // try to determine from output file extension
          use_file_type = File_manager::file_extension_for_file(sound_recorder_config.sound_file.file_name);
        }
        else
        {
          // neither specified, use default
          use_file_type =
            Sound_file_config::string_for_file_type(Sound_file_config::default_file_type);
        }
      }
      if (!Sound_file_config::file_type_for_string(use_file_type,
                                                   return_cfg.sound_recorder_config->sound_file->file_type))
      {
        // not an an understood file type, this will be an error
        return_cfg.sound_recorder_config->sound_file->file_type = Sound_file_config::file_type_unknown;
      }
      if (!Sound_file_config::data_format_for_string(sound_recorder_config.sound_file.data_format.v,
                                                     return_cfg.sound_recorder_config->sound_file->data_format))
      { // use default data format
        return_cfg.sound_recorder_config->sound_file->data_format = Sound_file_config::default_data_format;
      }
      if (!Sound_file_config::endianness_for_string(sound_recorder_config.sound_file.endianness.v,
                                                    return_cfg.sound_recorder_config->sound_file->endianness))
      { // use default endianness
        return_cfg.sound_recorder_config->sound_file->endianness = Sound_file_config::default_endianness;
      }
    }
    //
    Ih_validator<bool> has_level_meter;
    struct Level_meter_validator_struct
    {
      Level_meter_validator_struct(const Kinetophone_console_config::Level_meter_struct& s)
        : has_level(Ih_validator<bool>(s.has_level)),
          width(Ih_pos_int_validator(s.width)),
          use_max_width(Ih_validator<bool>(s.use_max_width)),
          model_ballistics(Ih_validator<bool>(s.model_ballistics)),
          attack_period(Ih_pos_int_validator(s.attack_period)),
          decay_period(Ih_pos_int_validator(s.decay_period)),
          peak_hold_period(Ih_pos_int_validator(s.peak_hold_period)) {}
      void update(Kinetophone_console_config::Level_meter_struct& return_s)
      {
        return_s.has_level = has_level.v;
        return_s.width = width.v;
        return_s.use_max_width = use_max_width.v;
        return_s.model_ballistics = model_ballistics.v;
        return_s.attack_period = attack_period.v;
        return_s.decay_period = decay_period.v;
        return_s.peak_hold_period = peak_hold_period.v;
      }
      //
      Ih_validator<bool> has_level;
      Ih_pos_int_validator width;
      Ih_validator<bool> use_max_width;
      Ih_validator<bool> model_ballistics;
      Ih_pos_int_validator attack_period;
      Ih_pos_int_validator decay_period;
      Ih_pos_int_validator peak_hold_period;
    } level_meter;
    Ih_pos_int_validator update_rate;
    Ih_validator<bool> show_file_size;
    Ih_validator<bool> show_available;
    Ih_validator<bool> show_audio_info;
    Ih_validator<bool> show_overflow_count;
    Ih_validator<bool> detailed_error;
    string custom_installation_dir;
    struct Sound_recorder_config_validator_struct
    {
      Sound_recorder_config_validator_struct(const Sound_recorder_config& s)
        : input_device(s.input_device),
          frames_per_buffer(Ih_pos_int_validator(s.frames_per_buffer)),
          write_buffer_factor(Ih_pos_int_validator(s.write_buffer_factor)),
          RMS_level_integration_period(Ih_pos_int_validator(s.RMS_level_integration_period)),
          sound_file(*s.sound_file),
          file_overwrite(s.file_overwrite) {}
      void update(Sound_recorder_config& return_s)
      {
        return_s.input_device = input_device;
        return_s.frames_per_buffer = frames_per_buffer.v;
        return_s.write_buffer_factor = write_buffer_factor.v;
        return_s.RMS_level_integration_period = RMS_level_integration_period.v;
        sound_file.update(*return_s.sound_file);
        return_s.file_overwrite = file_overwrite.v;
      }
      //
      int input_device;
      Ih_pos_int_validator frames_per_buffer;
      Ih_pos_int_validator write_buffer_factor;
      Ih_pos_int_validator RMS_level_integration_period;
      struct Sound_file_config_validator_struct
      {
        Sound_file_config_validator_struct(const Sound_file_config& s)
          : channels(Ih_pos_int_validator(s.channels)),
            sample_rate(Ih_pos_int_validator(static_cast<int>(s.sample_rate))),
            VBR_quality(Ih_fraction_validator(s.VBR_quality)),
            file_name(s.file_name),
            file_type(Ih_file_type_validator("")),
            data_format(Ih_data_format_validator(Sound_file_config::string_for_data_format(s.data_format))),
            endianness(Ih_endianness_validator(Sound_file_config::string_for_endianness(s.endianness))) {}
        void update(Sound_file_config& return_s)
        {
          return_s.channels = channels.v;
          return_s.sample_rate = sample_rate.v;
          return_s.VBR_quality = VBR_quality.v;
          return_s.file_name = file_name;
          if (!Sound_file_config::file_type_for_string(file_type.v, return_s.file_type))
          {
            return_s.file_type = Sound_file_config::file_type_unknown;
          }
          if (!Sound_file_config::data_format_for_string(data_format.v, return_s.data_format))
          {
            assert(false && "invalid data format string");
          }
          if (!Sound_file_config::endianness_for_string(endianness.v, return_s.endianness))
          {
            assert(false && "invalid endianness string");
          }
        }
        //
        Ih_pos_int_validator channels;
        Ih_pos_int_validator sample_rate;
        Ih_fraction_validator VBR_quality;
        string file_name;
        Ih_file_type_validator file_type;
        Ih_data_format_validator data_format;
        Ih_endianness_validator endianness;
      } sound_file;
      Ih_validator<bool> file_overwrite;
    } sound_recorder_config;
  };

  //
  // Auxiliary functions for checking input validity.
  //

  // Function used to check that 'opt1' and 'opt2' are not specified
  // at the same time.
  void ih_check_conflicting_options(const PO::variables_map& VM,
                                    const char* opt1,
                                    const char* opt2,
                                    bool condition = true,
                                    const string& message = "")
  {
    precondition(opt1
                 && opt2);

    if (condition &&
        (VM.count(opt1) && !VM[opt1].defaulted()
         && VM.count(opt2) && !VM[opt2].defaulted()))
    {
      string exception_message = string("conflicting options '")
        + opt1 + "' and '" + opt2 + "'";
      if ("" != message)
      {
        exception_message += string(": ") + message;
      }
      exception_message += ".";
      throw std::logic_error(exception_message);
    }
  }

  // function used to check that 'for_what' is specified, then
  // 'required_option' is specified too.
  void ih_check_option_dependency(const PO::variables_map& VM,
                                  const char* for_what,
                                  const char* required_option,
                                  bool condition = true)
  {
    precondition(for_what
                 && required_option);

    if (VM.count(for_what) && !VM[for_what].defaulted())
    {
      if ((VM.count(required_option) == 0)
          || VM[required_option].defaulted()
          || !condition)
      {
        throw std::logic_error(string("option '") + for_what
                               + "' requires option '" + required_option + "'.");
      }
    }
  }

  void ih_check_option_consistency(const Kinetophone_console_config& cr_cfg,
                                   const PO::variables_map& VM)
  {
    // conflicting option checks
    ih_check_conflicting_options(VM,
                                 "level-meter-width",
                                 "level-meter-max-width",
                                 cr_cfg.level_meter.use_max_width);
    ih_check_conflicting_options(VM,
                                 "file-type",
                                 "overwrite",
                                 (cr_cfg.sound_recorder_config->file_overwrite
                                  && (Sound_file_config::file_type_flac
                                      == cr_cfg.sound_recorder_config->sound_file->file_type)),
                                 "FLAC file format cannot be specified with file overwrite");
    // option dependency checks
    ih_check_option_dependency(VM,
                               "level-meter-level",
                               "level-meter",
                               cr_cfg.has_level_meter);
    ih_check_option_dependency(VM,
                               "level-meter-width",
                               "level-meter",
                               cr_cfg.has_level_meter);
    ih_check_option_dependency(VM,
                               "level-meter-max-width",
                               "level-meter",
                               cr_cfg.has_level_meter);
    ih_check_option_dependency(VM,
                               "level-meter-ballistics",
                               "level-meter",
                               cr_cfg.has_level_meter);
    ih_check_option_dependency(VM,
                               "level-meter-attack-period",
                               "level-meter-ballistics",
                               cr_cfg.level_meter.model_ballistics);
    ih_check_option_dependency(VM,
                               "level-meter-decay-period",
                               "level-meter-ballistics",
                               cr_cfg.level_meter.model_ballistics);
    // invalid file type
    if (Sound_file_config::file_type_unknown == cr_cfg.sound_recorder_config->sound_file->file_type)
    {
      throw std::logic_error("could not determine a valid sound file type");
    }
  }

  typedef struct
  {
    PO::options_description& cmd_line_desc;
    PO::options_description& basic_desc;
    PO::options_description& advanced_desc;
  } Ih_desc_params;

  // sets up the program option description object
  void ih_setup_descriptions(Ih_desc_params& return_desc_params,
                             Ih_validator_struct& return_v,
                             string& return_config_file)
  {
    const char *home_dir = getenv("HOME"); // (check_code_ignore)
    string default_config_file = (!home_dir || (*home_dir == 0) || !File_manager::path_is_directory(home_dir)) ?
      string("") : string(home_dir) + string("/.kinetophone");

    return_desc_params.cmd_line_desc.add_options()
      ("help,h", "output help message")
      ("list-devices,l", "generate a list of devices")
      ("config,i", PO::value<string>(&return_config_file)->default_value(default_config_file),
       "specify configuration file")
      ("version,v", "print program version")
      ;

    return_desc_params.basic_desc.add_options()
      ("audio-info,I", PO::value<Ih_validator<bool> >(&return_v.show_audio_info)
       ->implicit_value(Ih_validator<bool>(true)),
       "show audio information when recording")
      ("device,d", PO::value<int>(&return_v.sound_recorder_config.input_device)
       ->default_value(return_v.sound_recorder_config.input_device),
       "set input device number (from --list-devices option)")
      ("sound-file,f", PO::value<string>(&return_v.sound_recorder_config.sound_file.file_name)
       ->default_value(return_v.sound_recorder_config.sound_file.file_name),
       "set destination sound file path")
      ("overwrite,o", PO::value<Ih_validator<bool> >(&return_v.sound_recorder_config.file_overwrite)
       ->implicit_value(Ih_validator<bool>(true)),
       "overwrite sound recording file")
      ("level-meter,m", PO::value<Ih_validator<bool> >(&return_v.has_level_meter)
       ->implicit_value(Ih_validator<bool>(true)),
       "show level meter on console")
      ("level-meter-level,L", PO::value<Ih_validator<bool> >(&return_v.level_meter.has_level)
       ->implicit_value(Ih_validator<bool>(true)),
       "show numerical level on level meter (requires --level-meter or -m)")
      ("level-meter-width,w", PO::value<Ih_pos_int_validator>(&return_v.level_meter.width)
       ->default_value(return_v.level_meter.width),
       "set width in characters of level meter (must be > 0, requires --level-meter or -m)")
      ("level-meter-max-width,x", PO::value<Ih_validator<bool> >(&return_v.level_meter.use_max_width)
       ->implicit_value(Ih_validator<bool>(true)),
       "make level meter span width of the console (requires --level-meter or -m)")
      ("channels,c", PO::value<Ih_pos_int_validator>(&return_v.sound_recorder_config.sound_file.channels)
       ->default_value(return_v.sound_recorder_config.sound_file.channels),
       "set channels to record (must be > 0)")
      ("sample-rate,s", PO::value<Ih_pos_int_validator>(&return_v.sound_recorder_config.sound_file.sample_rate)
       ->default_value(return_v.sound_recorder_config.sound_file.sample_rate),
       "set samples per second (must be > 0)")
      ("file-type,t", PO::value<Ih_file_type_validator>(&return_v.sound_recorder_config.sound_file.file_type)
       ->default_value(return_v.sound_recorder_config.sound_file.file_type),
       "set audio file type\n"
       "{ wav | aiff | flac | ogg }") // (check_code_ignore)
      ("data-format,a", PO::value<Ih_data_format_validator>(&return_v.sound_recorder_config.sound_file
                                                            .data_format)
       ->default_value(return_v.sound_recorder_config.sound_file.data_format),
       "set format of samples in the audio file\n(signed values)\n"
       "{ pcm8 | pcm16 | pcm24 | pcm32 | float | double | vorbis }")
      ("show-file-size,F", PO::value<Ih_validator<bool> >(&return_v.show_file_size)
       ->implicit_value(Ih_validator<bool>(true)),
       "display the size of the output file")
      ("show-available,A", PO::value<Ih_validator<bool> >(&return_v.show_available)
       ->implicit_value(Ih_validator<bool>(true)),
       "display the available space/time for the output file")
      ;

    return_desc_params.advanced_desc.add_options()
      ("endian,E", PO::value<Ih_endianness_validator>(&return_v.sound_recorder_config.sound_file.endianness)
       ->default_value(return_v.sound_recorder_config.sound_file.endianness),
       "set endian order in the output file\n"
       "\t{ file | little | big | cpu }")
      ("frames-per-buffer,b", PO::value<Ih_pos_int_validator>(&return_v.sound_recorder_config.frames_per_buffer)
       ->default_value(return_v.sound_recorder_config.frames_per_buffer),
       "set number of samples processed at a time (must be > 0)")
      ("write-buffer-factor,W",
       PO::value<Ih_pos_int_validator>(&return_v.sound_recorder_config.write_buffer_factor)
       ->default_value(return_v.sound_recorder_config.write_buffer_factor),
       "set factor to multiply frames per buffer for sizing write buffer (must be > 0)")
      ("RMS-integration-period,R",
       PO::value<Ih_pos_int_validator>(&return_v.sound_recorder_config.RMS_level_integration_period)
       ->default_value(return_v.sound_recorder_config.RMS_level_integration_period),
       "set the period in milliseconds used for RMS level averaging (must be > 0)")
      ("update-rate,U", PO::value<Ih_pos_int_validator>(&return_v.update_rate)
       ->default_value(return_v.update_rate),
       "set number of times display is refreshed per second (must be > 0)")
      ("level-meter-ballistics,B", PO::value<Ih_validator<bool> >(&return_v.level_meter.model_ballistics)
       ->implicit_value(Ih_validator<bool>(return_v.level_meter.model_ballistics)),
       "model ballistics in level meter (requires --level-meter or -m)")
      ("level-meter-attack-period,P", PO::value<Ih_pos_int_validator>(&return_v.level_meter.attack_period)
       ->default_value(return_v.level_meter.attack_period),
       "set the number of milliseconds for the level meter to travel forward 0.99X full scale (must be > 0, "
       "requires --level-meter-ballistics)")
      ("level-meter-decay-period,p", PO::value<Ih_pos_int_validator>(&return_v.level_meter.decay_period)
       ->default_value(return_v.level_meter.decay_period),
       "set the number of milliseconds for the level meter to decay back from 0.99X full scale (must be > 0, "
       "requires --level-meter-ballistics)")
      ("level-meter-peak-hold-period,H", PO::value<Ih_pos_int_validator>(&return_v.level_meter.peak_hold_period)
       ->default_value(return_v.level_meter.peak_hold_period),
       "set the number of milliseconds for the level meter to maintain peak (must be > 0)")
      ("detailed-error,e", PO::value<Ih_validator<bool> >(&return_v.detailed_error)
       ->implicit_value(Ih_validator<bool>(true)),
       "display detailed diagnostics on error")
      ("overflow-count,O", PO::value<Ih_validator<bool> >(&return_v.show_overflow_count)
       ->implicit_value(Ih_validator<bool>(true)),
       "show input/output overflow count")
      ("VBR-quality,q", PO::value<Ih_fraction_validator>(&return_v.sound_recorder_config.sound_file.VBR_quality)
       ->default_value(Sound_file_config::highest_VBR_quality),
       "set variable bit rate (VBR) quality (0.0 = lowest to 1.0 = highest)")
      ("config-directory,C", PO::value<string>(&return_v.custom_installation_dir)
       ->default_value(return_v.custom_installation_dir),
       "set custom program configuration directory")
      ;
  }
}

//
// Constructor/destructor/copy
//

Kinetophone_console_config::Kinetophone_console_config()
  // setup some reasonable defaults
  : help_message(""),
    has_level_meter(false),
    level_meter(Level_meter_struct()),
    update_rate(default_update_rate),
    show_file_size(false),
    show_available(false),
    show_audio_info(false),
    show_overflow_count(false),
    detailed_error(false),
    custom_installation_dir(""),
    sound_recorder_config(new Sound_recorder_config)
{
}

Kinetophone_console_config::~Kinetophone_console_config()
{
  delete sound_recorder_config;
}

Kinetophone_console_config::Level_meter_struct::Level_meter_struct()
  // setup some reasonable defaults
  : has_level(false),
    width(default_level_meter_width),
    use_max_width(false),
    model_ballistics(true),
    attack_period(Level_meter_view::default_attack_period),
    decay_period(Level_meter_view::default_decay_period),
    peak_hold_period(Level_meter_view::default_peak_hold_period)
{
}

//
//  Class constants
//

//   default
const int Kinetophone_console_config::default_update_rate = 30;
const int Kinetophone_console_config::default_level_meter_width = 45;
//   command
const int Kinetophone_console_config::command_error;
const int Kinetophone_console_config::command_record;
const int Kinetophone_console_config::command_help;
const int Kinetophone_console_config::command_list_devices;
const int Kinetophone_console_config::command_output_version;

int Kinetophone_console_config::parse_program_options(int argc,
                                                      const char** argv,
                                                      Error_param& return_error)
{
  precondition(!return_error());

  int return_value = Kinetophone_console_config::command_error;

  help_message = "";

  start_error_block();

  try
  {
    PO::options_description cmd_line_desc("Command line only");
    stringstream basic_desc_label;
    basic_desc_label << "Command line and configuration file" << endl << "  Basic";
    PO::options_description basic_desc(basic_desc_label.str());
    PO::options_description advanced_desc("  Advanced");
    Ih_desc_params desc_params =
    {
      cmd_line_desc,
      basic_desc,
      advanced_desc
    };

    Ih_validator_struct validator(*this);
    string config_file;
    ih_setup_descriptions(desc_params,
                          validator,
                          config_file);

    // parse the command line to get config file
    PO::variables_map VM_config;
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(cmd_line_desc).allow_unregistered().run(), VM_config);
    notify(VM_config);

    PO::options_description config_desc;
    config_desc.add(basic_desc).add(advanced_desc);

    stringstream help_stream;
    help_stream << "Usage:  ";
    help_stream << "kinetophone_console --help | --list-devices | [ output_file ] [ options ]" << endl;
    help_stream << endl;

    PO::options_description all_desc("Options");
    all_desc.add(cmd_line_desc).add(basic_desc).add(advanced_desc);
    help_stream << all_desc;
    help_message = help_stream.str();

    PO::variables_map VM;

    // parse the command line (overrides config file)
    PO::positional_options_description pos_desc;
    pos_desc.add("sound-file", 1);
    PO::store(PO::command_line_parser(argc, const_cast<char**>(argv)).
              options(all_desc).positional(pos_desc).run(), VM);
    PO::notify(VM);

    // parse the config file
    if ("" != config_file)
    {
      ifstream ifs(config_file.c_str());
      bool have_error =  (!ifs && !VM_config["config"].defaulted());
      on_error(have_error, new Boost_error(error_location(), string("cannot open config file: ") + config_file));
      const bool allow_unregistered_options = true;
      PO::store(PO::parse_config_file(ifs, config_desc, allow_unregistered_options), VM);
      PO::notify(VM);
    }

    if (VM.count("help"))
    { // HELP
      return_value = Kinetophone_console_config::command_help;
    }
    else if (VM.count("list-devices"))
    { // LIST DEVICES
      return_value = Kinetophone_console_config::command_list_devices;
    }
    else if (VM.count("version"))
    {
      // PROGRAM VERSION
      return_value = Kinetophone_console_config::command_output_version;
    }
    else
    { // RECORD
      // update the config from the validator
      validator.update(*this);
      ih_check_option_consistency(*this, VM);
      return_value = Kinetophone_console_config::command_record;
    }
  }
  catch (const std::exception& e)
  {
    string diagnostic = e.what();
    stringstream s;
    s << "Reason: " << diagnostic << std::endl << std::endl;
    s << "Enter kinetophone --help for valid program options.";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     new Boost_error(error_location(), s.str()));
    on_error(true, error);
  }
  catch (...)
  {
    string diagnostic = "exception parsing command line";
    Kinetophone_error* error = new Kinetophone_error(error_location(),
                                                     Kinetophone_error::command_line,
                                                     diagnostic);
    on_error(true, error);
  }

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  return_value = Kinetophone_console_config::command_error;
  goto exit_point;

 exit_point:
  return return_value;
}
