// Kinetophone_movie_builder.cpp
//
// Copyright 2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_movie_builder.hpp"

// include *mm first to avoid conflicts
#include <gdkmm/pixbuf.h>

#include "Kinetophone_builder_config.hpp"
#include "Kinetophone_builder_model.hpp"
#include "../base/Movie_builder_config.hpp"
#include "../base/Movie_config.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/Segment.hpp"
#include <cmath>

using Glib::RefPtr;
using Gdk::Pixbuf;
using Roan_trail::Color;
using Roan_trail::Rectangle;
using Roan_trail::Rect_origin;
using Roan_trail::Rect_size;
using Roan_trail::Builder::Movie_config;
using Roan_trail::Builder::Movie_builder_config;
using Roan_trail::Source::Slide_collection;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//
namespace
{
  const int ic_bits_per_sample = 8;

  // calculate parameters for transforming source image to destination image
  // for cropped compositing
  void ih_scale_src_cropped_to_dest(Rect_size src_size,
                                    Rect_size dest_size,
                                    double& return_scale,
                                    Rect_origin& return_src_offset)
  {
    const double src_width = static_cast<double>(src_size.width);
    const double src_height = static_cast<double>(src_size.height);
    const double src_aspect = src_width / src_height;
    const double dest_width = static_cast<double>(dest_size.width);
    const double dest_height = static_cast<double>(dest_size.height);
    const double dest_aspect = dest_width / dest_height;

    if (src_aspect < dest_aspect)
    {
      // scale width
      return_scale = dest_width / src_width;
      return_src_offset = Rect_origin(0, rint((src_height * return_scale - dest_height) / 2.0));
    }
    else
    {
      // scale height
      return_scale = dest_height / src_height;
      return_src_offset = Rect_origin(rint((src_width * return_scale - dest_width) / 2.0), 0);
    }
  }

  // calculate parameters for transforming source image to destination image
  // for fitted compositing
  void ih_scale_src_fit_in_dest(Rect_size src_size,
                                Rect_size dest_size,
                                Rectangle& return_dest_rect,
                                double& return_scale)
  {
    const double src_width = static_cast<double>(src_size.width);
    const double src_height = static_cast<double>(src_size.height);
    const double src_aspect = src_width / src_height;
    const double dest_width = static_cast<double>(dest_size.width);
    const double dest_height = static_cast<double>(dest_size.height);
    const double dest_aspect = dest_width / dest_height;

    if (src_aspect < dest_aspect)
    {
      // scale height
      return_scale = dest_height / src_height;
      Rect_origin offset(rint((dest_width - return_scale * src_width) / 2.0), 0);
      return_dest_rect = Rectangle(offset, Rect_size(rint(src_width * return_scale),
                                                     rint(src_height * return_scale)));
    }
    else
    {
      // scale width
      return_scale = dest_width / src_width;
      Rect_origin offset(0, rint((dest_height - return_scale * src_height) / 2.0));
      return_dest_rect = Rectangle(offset, Rect_size(rint(src_width * return_scale),
                                                     rint(src_height * return_scale)));
    }
  }
}

//
// Constructor/destructor
//

Kinetophone_movie_builder::Kinetophone_movie_builder(const Kinetophone_builder_config& config,
                                                     const Kinetophone_builder_model& model)
  : Movie_builder(*config.movie_builder_config, model),
    m_config(config),
    m_model(model),
    m_pixbuf_cache(0),
    m_pixbuf_convert_cache(0)
{
  const Movie_config& movie_config = *m_config.movie_builder_config->movie;
  Rect_size input_size;
  if (!Movie_config::input_size_for_movie_config(movie_config, input_size))
  {
    assert(false && "invalid movie config for input size");
  }
  Rect_size output_size;
  if (!Movie_config::output_size_for_movie_config(movie_config, output_size))
  {
    assert(false && "invalid movie config for output size");
  }
  m_pixbuf_cache = Pixbuf::create(Gdk::COLORSPACE_RGB,
                                  false,
                                  ic_bits_per_sample,
                                  output_size.width,
                                  output_size.height);

  if (input_size != output_size)
  {
    Rect_size convert_size;
    if (m_config.movie_builder_config->movie_attributes.square_pixels)
    {
      if (!Movie_config::output_size_for_movie_config(movie_config, convert_size))
      {
        assert(false && "invalid movie config for output size (convert)");
      }
    }
    else
    {
      if (!Movie_config::input_size_for_movie_config(movie_config, convert_size))
      {
        assert(false && "invalid movie config for input size (convert)");
      }
    }
    m_pixbuf_convert_cache = Pixbuf::create(Gdk::COLORSPACE_RGB,
                                            false,
                                            ic_bits_per_sample,
                                            convert_size.width,
                                            convert_size.height);
  }

  postcondition(((input_size == output_size) || m_pixbuf_convert_cache)
                && mf_invariant(false));
}

Kinetophone_movie_builder::~Kinetophone_movie_builder()
{
  precondition(mf_invariant(false));
}

//
// Protected member functions
//

const uint8_t* Kinetophone_movie_builder::mf_generate_image(Long_int slide_index, bool& return_used_default)
{
  precondition((slide_index >= 0)
               && (slide_index < m_model.slides()->slide_count()));

  const Slide_collection<RefPtr<Pixbuf> >* slides = m_model.slides();
  RefPtr<Pixbuf> src_image;
  const bool image_found = slides->image_for_slide_at(slide_index, src_image);
  return_used_default = !image_found;
  const Rect_size src_image_size(src_image->get_width(), src_image->get_height());

  const Movie_builder_config& config = *m_config.movie_builder_config;
  const Movie_config& movie_config = *config.movie;
  Rect_size input_size;
  if (m_config.movie_builder_config->movie_attributes.square_pixels)
  {
    if (!Movie_config::output_size_for_movie_config(movie_config, input_size))
    {
      assert(false && "invalid movie config for output size");
    }
  }
  else
  {
    if (!Movie_config::input_size_for_movie_config(movie_config, input_size))
    {
      assert(false && "invalid movie config for input size");
    }
  }
  Rect_size output_size;
  if (!Movie_config::output_size_for_movie_config(movie_config, output_size))
  {
      assert(false && "invalid movie config for output size");
  }

  Rect_origin offset(0, 0);
  Rectangle dest_rect(offset, input_size);
  double scale;
  if (config.movie_attributes.crop_image_to_frame)
  {
    ih_scale_src_cropped_to_dest(src_image_size,
                                 input_size,
                                 scale,
                                 offset);
  }
  else
  {
    ih_scale_src_fit_in_dest(src_image_size,
                             input_size,
                             dest_rect,
                             scale);
  }
  const double scale_x = scale;
  const double scale_y = scale;
  const int32_t pixel_color = config.movie_attributes.aspect_fill_color.scalar_value();
  if (input_size == output_size)
  {
    m_pixbuf_cache->fill(pixel_color);
    // pixels are square:  scale and fit image to output rect
    src_image->composite(m_pixbuf_cache,
                         dest_rect.origin.x,
                         dest_rect.origin.y,
                         dest_rect.size.width,
                         dest_rect.size.height,
                         dest_rect.origin.x - offset.x,
                         dest_rect.origin.y - offset.y,
                         scale_x,
                         scale_y,
                         config.movie_attributes.image_interpolation ? Gdk::INTERP_HYPER : Gdk::INTERP_TILES,
                         255);
  }
  else
  {
    m_pixbuf_convert_cache->fill(pixel_color);
    // pixels are non-square:  need to scale and fit to the input rect first
    src_image->composite(m_pixbuf_convert_cache,
                         dest_rect.origin.x,
                         dest_rect.origin.y,
                         dest_rect.size.width,
                         dest_rect.size.height,
                         dest_rect.origin.x - offset.x,
                         dest_rect.origin.y - offset.y,
                         scale_x,
                         scale_y,
                         config.movie_attributes.image_interpolation ? Gdk::INTERP_HYPER : Gdk::INTERP_TILES,
                         255);
    // then a squeeze/stretch to the output rect to get the correct pixel aspect
    const double convert_scale_x = static_cast<double>(output_size.width)
      / static_cast<double>(input_size.width);
    const double convert_scale_y = static_cast<double>(output_size.height)
      / static_cast<double>(input_size.height);
    m_pixbuf_convert_cache->scale(m_pixbuf_cache,
                                  0,
                                  0,
                                  output_size.width,
                                  output_size.height,
                                  0,
                                  0,
                                  convert_scale_x,
                                  convert_scale_y,
                                  config.movie_attributes.image_interpolation ?
                                  Gdk::INTERP_HYPER : Gdk::INTERP_TILES);

  }
  m_video_frame_cache = m_pixbuf_cache->get_pixels();

  return 0;
}

bool Kinetophone_movie_builder::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_pixbuf_cache)
  {
    goto exit_point;
  }

  return_value = !check_base_class || Movie_builder::mf_invariant(check_base_class);
  goto exit_point;

 exit_point:
  return return_value;
}
