// Kinetophone_builder_model.cpp
//
// Copyright 2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_builder_model.hpp"

// include *mm first to avoid conflicts
#include <gdkmm/pixbuf.h>

#include "Kinetophone_builder_config.hpp"
#include "../narrator/Kinetophone_dir_slide_collection.hpp"
#include "../base/Segment.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include "../base/error/Posix_error.hpp"
#include <libxml++/exceptions/exception.h>
#include <libxml++/nodes/node.h>
#include <libxml++/parsers/domparser.h>
#include <libxml2/libxml/tree.h>
#include <string>
#include <cerrno> // (check_code_ignore)

using Glib::RefPtr;
using Gdk::Pixbuf;
using xmlpp::DomParser;
using xmlpp::Node;
using std::string;
using Roan_trail::Long_int;
using Roan_trail::Error_param;
using Roan_trail::Recorder::Segment;
using Roan_trail::Source::Slide_collection;
using namespace Roan_trail::Kinetophone;

//
// Constructor/destructor
//

Kinetophone_builder_model::Kinetophone_builder_model(const Kinetophone_builder_config& builder_config,
                                                     const string& slides_source_type)
  : m_builder_config(builder_config),
    m_slides(0),
    m_slides_source_type(slides_source_type)
{
  postcondition(mf_invariant(false));
}

Kinetophone_builder_model::~Kinetophone_builder_model()
{
  precondition(mf_invariant(false));

  delete m_slides;
}

//
// Protected member functions
//

bool Kinetophone_builder_model::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  return_value = true;

  goto exit_point;

 exit_point:
  return return_value;
}

//
// Load
//

bool Kinetophone_builder_model::load_from_XML(const string& file_path, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;
  Slide_collection<RefPtr<Pixbuf> >* slides = 0;

  start_error_block();

  Error_param error;
  const string& diagnostic_prefix = "error loading XML file";

  try
  {
    DomParser parser(file_path); // no XML validation by default
    parser.set_substitute_entities(); // substitute text for entity references
    parser.parse_file(file_path);
    on_error(!parser, new Kinetophone_error(error_location(),
                                            Kinetophone_error::file_IO,
                                            diagnostic_prefix + string(", could not parse"),
                                            file_path));

    const Node* root_node = parser.get_document()->get_root_node();
    on_error(!root_node, new Kinetophone_error(error_location(),
                                               Kinetophone_error::file_IO,
                                               diagnostic_prefix + string(", could not parse"),
                                               file_path));

    // import into the slide collection
    string custom_source_path;
    if ("directory" == m_slides_source_type)
    {
      custom_source_path = m_builder_config.image_directory;
      slides = new Kinetophone_dir_slide_collection;
    }
    else
    {
      assert(false && "invalid slides source type when creating Kinetophone narrator model");
    }
    const bool slides_loaded = slides->load_from_XML(file_path,
                                                     root_node,
                                                     custom_source_path,
                                                     error);
    on_error(!slides_loaded, new Kinetophone_error(error_location(),
                                                   Kinetophone_error::build,
                                                   error()));

    // import base class
    const bool base_model_loaded = Movie_builder_model::load_from_XML(file_path,
                                                                      root_node,
                                                                      error);
    on_error(!base_model_loaded, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::build,
                                                       error()));
  }
  catch (const xmlpp::exception& e)
  {
    const string diagnostic = diagnostic_prefix + string(": ") + e.what();
    on_error(true, new Kinetophone_error(error_location(),
                                         Kinetophone_error::file_IO,
                                         diagnostic,
                                         file_path));
  }

  // success, update data members
  delete m_slides;
  m_slides = slides;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (slides)
  {
    delete slides;
  }
  goto exit_point;

 exit_point:
  postcondition(((return_value && m_slides)
                 || !return_value)
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}
