// Kinetophone_build_manager.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_build_manager.hpp"

// include *mm first to avoid conflicts
#include <gdkmm/pixbuf.h>

#include "Kinetophone_builder_config.hpp"
#include "Kinetophone_builder_model.hpp"
#include "Kinetophone_movie_builder.hpp"
#include "../base/File_manager.hpp"
#include "../base/Logger.hpp"
#include "../base/Movie.hpp"
#include "../base/Movie_builder_config.hpp"
#include "../base/Movie_config.hpp"
#include "../base/Segment.hpp"
#include "../base/Slide.hpp"
#include "../base/Slide_collection.hpp"
#include "../base/Sound_builder.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/Sound_track.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include "../base/error/Build_error.hpp"
#include <iomanip>
#include <sstream>
#include <string>
#include <vector>

using Glib::RefPtr;
using Gdk::Pixbuf;
using std::stringstream;
using std::endl;
using std::ios_base;
using std::setfill;
using std::setw;
using std::string;
using std::vector;
using Roan_trail::Logger;
using Roan_trail::Recorder::Segment;
using Roan_trail::Source::Slide;
using Roan_trail::Recorder::Sound_file_config;
using Roan_trail::Builder::Movie_builder_config;
using Roan_trail::Builder::Movie_config;
using Roan_trail::Builder::Sound_track;
using Roan_trail::Builder::Build_error;
using namespace Roan_trail::Kinetophone;

//
// Internal helpers
//
namespace
{
  const double ic_build_estimate_factor = 1.01;
  const double ic_output_space_warning_threshold = 0.95;
}

//
// Constructor/destructor
//

Kinetophone_build_manager::Kinetophone_build_manager(const Kinetophone_builder_config& config,
                                                     const Kinetophone_builder_model& model)
  : m_config(config),
    m_model(model),
    m_logger(Logger::default_logger()),
    m_movie_builder(0),
    m_running(0),
    m_cancelled(0),
    m_setup(false)
{
  postcondition(mf_invariant(false));
}

Kinetophone_build_manager::~Kinetophone_build_manager()
{
  precondition(mf_invariant(false));

  delete m_movie_builder;
}

//
// Control
//

bool Kinetophone_build_manager::setup_for_build(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  Kinetophone_movie_builder* movie_builder = 0;
  bool return_value = false;

  start_error_block();

  m_logger << Logger::info << "Setting up for build." << endl;

  movie_builder = new Kinetophone_movie_builder(m_config, m_model);

  Error_param error;
  const bool movie_builder_setup = movie_builder->setup(error);
  on_error(!movie_builder_setup, new Kinetophone_error(error_location(),
                                                       Kinetophone_error::build,
                                                       error()));

  mf_log_config(*movie_builder);
  mf_log_model(*movie_builder);

  // success, update data members
  delete m_movie_builder;
  m_movie_builder = movie_builder;
  m_setup = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (!movie_builder)
  {
    delete movie_builder;
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

// Perform various checks before the potentially lengthy
// build to lessen frustration levels.  The check
// is not a guarantee that the build will work, however,
// because conditions might change after the check.
bool Kinetophone_build_manager::prebuild_checks(Error_param& return_error)
{
  precondition(m_setup
               && !return_error()
               && mf_invariant());

  bool return_value = false;
  bool have_warning = false;

  m_logger << Logger::info << "Estimating build size:" << endl;
  const Long_int build_size_estimate = m_movie_builder->estimate_build_size()
    * ic_build_estimate_factor;
  m_logger << Logger::info << "Movie(s) will use approximately " << build_size_estimate << " (bytes) " << endl;
  if (!m_config.movie_builder_config->skip_build_warnings)
  {
    // check for output directory accessibility
    if (!m_config.movie_builder_config->output.temporary_movies)
    {
      const string output_dir = m_config.movie_builder_config->output.directory;
      const bool accessible = File_manager::path_exists(output_dir)
        && File_manager::path_is_directory(output_dir)
        && File_manager::path_is_writable(output_dir);
      if (!accessible)
      {
        Kinetophone_error* output_dir_warning =
          new Kinetophone_error(error_location(),
                                Kinetophone_error::build_warning,
                                new Build_error(error_location(), Build_error::prebuild_output_dir_access));
        mf_log_warning(output_dir_warning);
        delete output_dir_warning;
        have_warning = true;
      }
      else
      {
        // check for sufficient output space
        double fraction_avail = 0.0;
        Long_int output_space_avail = 0;
        Error_param error(false);
        if (!File_manager::file_system_available_for_path(output_dir,
                                                          fraction_avail,
                                                          output_space_avail,
                                                          error)
            || ((output_space_avail * ic_output_space_warning_threshold) < build_size_estimate))
        {
          Kinetophone_error* output_space_warning =
            new Kinetophone_error(error_location(),
                                  Kinetophone_error::build_warning,
                                  new Build_error(error_location(),
                                                  Build_error::prebuild_insuff_output_space));
          mf_log_warning(output_space_warning, false);
          m_logger << Logger::warning;
          m_logger << "output space available: " << output_space_avail;
          m_logger << " (bytes) in output directory:" << endl;
          m_logger << "  " << output_dir << endl;
          m_logger << "approximate size of output movie(s): " << build_size_estimate << endl;
          m_logger << "----" << endl;
          delete output_space_warning;
          have_warning = true;
        }
      }
    }

    // check output movie existence
    if (m_movie_builder->movie_or_movies_exist())
    {
      Kinetophone_error* exists_warning =
        new Kinetophone_error(error_location(),
                              Kinetophone_error::build_warning,
                              new Build_error(error_location(),
                                              (m_config.movie_builder_config->output.file_overwrite ?
                                               Build_error::prebuild_movie_exists_overwrite
                                               : Build_error::prebuild_movie_exists_no_overwrite)));
      mf_log_warning(exists_warning);
      delete exists_warning;
      have_warning = true;
    }

    // check for sufficient temporary space
    const string temp_dir = m_config.movie_builder_config->output.temporary_directory;
    double fraction_avail = 0.0;
    Long_int temp_space_avail = 0;
    Error_param error(false);
    if (!File_manager::file_system_available_for_path(temp_dir,
                                                      fraction_avail,
                                                      temp_space_avail,
                                                      error)
        || ((temp_space_avail * ic_output_space_warning_threshold) < build_size_estimate))
    {
      Kinetophone_error* temp_space_warning =
        new Kinetophone_error(error_location(),
                              Kinetophone_error::build_warning,
                              new Build_error(error_location(),
                                              Build_error::prebuild_insuff_temp_space));

      mf_log_warning(temp_space_warning, false);
      m_logger << Logger::warning;
      m_logger << "temporary space available: " << temp_space_avail;
      m_logger << " (bytes) in directory:" << endl;
      m_logger << "  " << temp_dir << endl;
      m_logger << "approximate size of output movie(s): " << build_size_estimate << endl;
      m_logger << "----" << endl;
      delete temp_space_warning;
      have_warning = true;
    }
  }
  return_value = (!have_warning || !m_config.quit_on_prebuild_warnings);

  goto exit_point;

 exit_point:
  postcondition(m_setup
                && !return_error()
                && mf_invariant());
  return return_value;
}

bool Kinetophone_build_manager::build(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  __sync_or_and_fetch(&m_running, 0x1);

  vector<Error_param> warnings;
  bool need_builder_stop = false;
  bool need_builder_end = false;
  bool cancelled = false;
  bool return_value = false;

  start_error_block();

  m_logger << Logger::message << "Starting build" << endl;

  Error_param error;

  const bool builder_started = m_movie_builder->start(error);
  on_error(!builder_started, new Kinetophone_error(error_location(),
                                                    Kinetophone_error::build,
                                                    error()));
  need_builder_stop = true;
  need_builder_end = true;

  // build loop
  bool more_segments = true;
  while (more_segments)
  {
    // check for cancellation
    cancelled = is_cancelled();
    if (cancelled)
    {
      m_logger << Logger::message << "Build cancelled" << endl;
      goto break_out;
    }

    // segment status
    mf_log_segment();

    // add another segment
    const bool segment_added = m_movie_builder->add_next_segment(more_segments, error);
    on_error(!segment_added, new Kinetophone_error(error_location(),
                                                    Kinetophone_error::build,
                                                    error()));

    // check for cancellation
    cancelled = is_cancelled();
    if (cancelled)
    {
      m_logger << Logger::message << endl << "Build cancelled" << endl;
      goto break_out;
    }

    mf_log_percent_complete();
  }

 break_out: // normal loop exit point (complete or cancelled build)
  m_logger << Logger::message << "Finishing" << endl;

  need_builder_stop = false;
  const bool builder_stopped = m_movie_builder->stop(!cancelled,
                                                     error);
  on_error(!builder_stopped, new Kinetophone_error(error_location(),
                                                    Kinetophone_error::build,
                                                    error()));

  need_builder_end = false;
  bool builder_ended = m_movie_builder->end(!cancelled,
                                            warnings,
                                            error);
  for (vector<Error_param>::const_iterator i = warnings.begin(); i != warnings.end(); ++i)
  {
    mf_log_warning((*i)());
  }
  on_error(!builder_ended, new Kinetophone_error(error_location(),
                                                 Kinetophone_error::build,
                                                 error()));
  mf_log_temporary_movies(*m_movie_builder);

  m_logger << Logger::message;
  m_logger << "-------------------" << endl;
  m_logger << "| Build completed |" << endl;
  m_logger << "-------------------" << endl;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  m_logger << Logger::info << "Error encountered while building" << endl;
  if (need_builder_end)
  {
    Error_param error(false);
    if (need_builder_stop)
    {
      m_movie_builder->stop(false, error); // ignore error
    }
    vector<Error_param> warnings;
    m_movie_builder->end(false,
                         warnings, // ignore warnings
                         error); // ignore error
  }
  return_value = false;
  goto exit_point;

 exit_point:
  // clean up the warnings, if any
  for (vector<Error_param>::iterator i = warnings.begin(); i != warnings.end(); ++i)
  {
    delete (*i)();
  }
  warnings.clear();

  postcondition(!warnings.size()
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());

  __sync_and_and_fetch(&m_running, 0x0);

  return return_value;
}

//
// Protected member functions
//

bool Kinetophone_build_manager::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (m_setup
      && !m_movie_builder)
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

//
//   messages
//

void Kinetophone_build_manager::mf_log_segment()
{
  // normal message
  m_logger << Logger::message;
  const Long_int current_segment_index = m_movie_builder->current_segment_index();
  const Long_int segment_count = static_cast<Long_int>(m_model.segments().size());
  m_logger << "Processing segment: " << (current_segment_index + 1) << " of " << segment_count;
  // additional info
  m_logger << Logger::info;
  const Segment& current_segment = m_model.segments()[current_segment_index];
  const Slide<RefPtr<Pixbuf> >* current_slide = m_model.slides()->operator[](current_segment.index());
  const Long_int slide_index = current_slide->index();
  m_logger << ", slide: " << (slide_index + 1);
  const Long_int segment_ordinality = current_segment.ordinality();
  m_logger << ", ordinality: " << (segment_ordinality + 1);
  m_logger << ", image: " << File_manager::file_name_for_file_path(current_slide->path());
  if (current_segment.is_retake())
  {
    m_logger << " [retake]";
  }
  if (m_config.movie_builder_config->output.create_multiple_movies)
  {
    m_logger << endl << "Current movie: ";
    string movie_name = m_movie_builder->name_for_movie(m_movie_builder->movie_file_name(),
                                                        current_segment_index,
                                                        segment_ordinality,
                                                        slide_index)
      + string(".")
      + Movie_config::extension_for_movie_preset(m_config.movie_builder_config->movie->movie_preset);
    m_logger << movie_name;
  }
  //
  m_logger << Logger::message << endl;
}

void Kinetophone_build_manager::mf_log_percent_complete()
{
  stringstream status_stream;
  status_stream.setf(std::ios_base::fixed, std::ios_base::floatfield);
  status_stream.precision(1);
  const double percent_completed = m_movie_builder->total_fraction_completed() * 100.0;
  const int star_count = min(10, max(1, static_cast<int>(percent_completed / 10.0)));
  const string stars(star_count, '*');
  const string spaces(10 - star_count, ' ');
  status_stream << stars << spaces << " (build " << setw(5) << setfill(' ') << percent_completed;
  status_stream << "% completed)";
  m_logger << Logger::message << status_stream.str() << endl;
}

void Kinetophone_build_manager::mf_log_config(const Kinetophone_movie_builder& builder)
{
  m_logger << Logger::info;
  m_logger << "Session info loaded from: " << m_config.session_file_path << endl;

  // source sound file
  const Sound_file_config& sound_config = builder.sound_builder().config();
  m_logger << endl;
  m_logger << "Source sound file configuration" << endl;
  m_logger << "-------------------------------" << endl;
  m_logger << "  path: " << sound_config.file_name << endl;
  m_logger << "  sample rate: " << sound_config.sample_rate << " samples/sec" << endl;
  m_logger << "  channels: " << sound_config.channels << endl;
  m_logger << "  type: " << Sound_file_config::string_for_file_type(sound_config.file_type) << endl;
  m_logger << "  format: " << Sound_file_config::string_for_data_format(sound_config.data_format);
  m_logger << " (signed)" << endl;
  m_logger << "  endian: " << Sound_file_config::string_for_endianness(sound_config.endianness) << endl;
  const Sound_track* sound_track = builder.movie().sound_track();
  assert(sound_track && "error, sound_track is null");
  if (sound_track->endian_conversion())
  {
    m_logger << "  * endianness will be swapped due to movie format limitations" << endl;
  }
  m_logger << endl;

  // movie
  const Movie_builder_config& movie_config = *m_config.movie_builder_config;
  m_logger << "Movie configuration" << endl;
  m_logger << "-------------------" << endl;
  //   format
  m_logger << "  Format" << endl;
  m_logger << "    container: QT MOV" << endl;
  m_logger << "    codec: qtrle (RLE \"Animation\")" << endl;
  m_logger << "    video colorspace: RGB" << endl;
  const Movie_config& movie = *movie_config.movie;
  m_logger << "    extension: " << Movie_config::extension_for_movie_preset(movie.movie_preset) << endl;
  m_logger << "    frame rate: " << Movie_config::string_for_frame_rate_type(movie.frame_rate) << endl;
  m_logger << "    frame format: " << Movie_config::string_for_frame_format(movie.frame_format) << endl;
  Rect_size dest_size;
  if (!Movie_config::output_size_for_movie_config(movie, dest_size))
  {
    assert(false && "invalid movie config for output size");
  }
  m_logger << "      destination frame size: " << dest_size.width << " X " << dest_size.height << endl;
  if ((Movie_config::frame_format_original != movie.frame_format)
      && !movie_config.movie_attributes.square_pixels)
  {
    Rect_size src_size;
    if (!Movie_config::input_size_for_movie_config(movie, src_size))
    {
      assert(false && "invalid movie config for input size");
    }
    m_logger << "      source frame size: " << src_size.width << " X " << src_size.height << endl;
  }
  m_logger << "    keyframe period: " << movie.keyframe_period << endl;
  const Movie_builder_config::Output_struct& output = movie_config.output;
  //   output
  m_logger << "  Output" << endl;
  if (output.create_multiple_movies)
  {
    m_logger << "    multiple movies" << endl;
    m_logger << "    multiple movie prefix: ";
    m_logger << builder.movie_file_name() << endl;
  }
  else
  {
    m_logger << "    single movie" << endl;
    m_logger << "    movie name: " << builder.movie_file_name() << ".";
    m_logger << Movie_config::extension_for_movie_preset(movie.movie_preset) << endl;
  }
  m_logger << "    temporary directory: " << output.temporary_directory << endl;
  if (!output.temporary_movies)
  {
    m_logger << "    destination directory: " << output.directory << endl;
  }
  m_logger << "    overwrite existing movies: " << (output.file_overwrite ? "yes" : "no") << endl;
  const Movie_builder_config::Movie_attributes_struct& attributes = movie_config.movie_attributes;
  //   attributes
  m_logger << "  Attributes" << endl;
  m_logger << "    square pixels: " << (attributes.square_pixels ? "yes" : "no") << endl;
  m_logger << "    silence gaps: " << (attributes.silence_gaps ? "yes" : "no") << endl;
  if (attributes.silence_gaps)
  {
    m_logger << "    silence gap length: " << attributes.silence_gap_length << " (frames)" << endl;
  }
  if (Movie_config::frame_format_original == movie.frame_format)
  {
    m_logger << "    scale for original image: " << attributes.full_resolution_scale << endl;
  }
  stringstream hex_stream;
  hex_stream.setf(ios_base::hex);
  hex_stream << setw(6) << setfill('0');
  hex_stream << attributes.aspect_fill_color.scalar_value();
  m_logger << "    aspect fill color (RRGGBB hex): " << hex_stream.str() << endl;
  m_logger << "    crop source image to frame: " << (attributes.crop_image_to_frame ? "yes" : "no") << endl;
  m_logger << "    high quality image interpolation when scaling: ";
  m_logger << (attributes.image_interpolation ? "yes" : "no") << endl;
  m_logger << endl;
}

void Kinetophone_build_manager::mf_log_model(const Kinetophone_movie_builder& builder)
{
  m_logger << Logger::info;
  m_logger << "Session" << endl;
  m_logger << "-------" << endl;
  m_logger << "  images loaded from: " << m_model.slides()->source() << endl;
  // audio sample rate
  const Long_int original_sample_rate = m_model.audio_recording_frame_rate();
  m_logger << "  original audio sample rate: " << original_sample_rate << endl;;
  const double dest_sample_rate = builder.sound_builder().config().sample_rate;
  if (static_cast<const Long_int>(dest_sample_rate) != original_sample_rate)
  {
    m_logger << "  * will be converted to:  " << dest_sample_rate << endl;
  }
  // segments
  const vector<Segment>& segments = m_model.segments();
  m_logger << "  movie segments: " << segments.size() << endl;
  m_logger << endl;
}

void Kinetophone_build_manager::mf_log_temporary_movies(const Kinetophone_movie_builder& builder)
{
  if (m_config.movie_builder_config->output.temporary_movies)
  {
    const vector<string>& output_movies = builder.output_movies();
    if (output_movies.size())
    {
      m_logger << Logger::info;
      m_logger << "Temporary movies located at:" << endl;
      for (vector<string>::const_iterator i = output_movies.begin(); i != output_movies.end(); ++i)
      {
        m_logger << "  " << *i << endl;
      }
    }
  }
}

void Kinetophone_build_manager::mf_log_warning(const Error* warning, bool output_separator)
{
  precondition(m_setup
               && warning);

  if (!m_config.movie_builder_config->skip_build_warnings)
  {
    stringstream warning_stream;
    warning_stream << "Build warning: " << endl;

    const Error* user_warning = warning->user_error_for_chain();

    bool have_detail = false;

    if (user_warning)
    {
      const string detailed_description =
        user_warning->dictionary_entry_for_chain(Error::detailed_description_error_key);
      if ("" != detailed_description)
      {
        have_detail = true;
        warning_stream << detailed_description << endl;
      }

      const string recovery_description =
        user_warning->dictionary_entry_for_chain(Error::recovery_description_error_key);
      if ("" != recovery_description)
      {
        have_detail = true;
        warning_stream << recovery_description << endl;
      }

      const string diagnostic = user_warning->dictionary_entry_for_chain(Error::diagnostic_error_key);
      if ("" != recovery_description)
      {
        have_detail = true;
        warning_stream << diagnostic << endl;
      }
    }

    if (!user_warning || !have_detail)
    {
      warning_stream << "Warning class: " << warning->error_class() << endl;
      warning_stream << "Code: " << warning->code();
      const Error::Error_dictionary& dictionary = warning->error_dictionary();
      Error::Error_dictionary::const_iterator code_string = dictionary.find(Error::code_string_error_key);
      if (dictionary.end() != code_string)
      {
        if ("" != code_string->second)
        {
          warning_stream << " (" << code_string->second << ")";
        }
      }
      warning_stream << endl;
    }

    if (output_separator)
    {
      warning_stream << "----" << endl;
    }

    m_logger << Logger::warning << warning_stream.str();
  }

  postcondition(m_setup);
}

