// Error.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Error.hpp"
#include "../File_manager.hpp"
#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>
#include <boost/foreach.hpp>
#include <iostream>
#include <sstream>
#include <string>

using std::endl;
using std::ostream;
using std::string;
using std::stringstream;
using boost::property_tree::ptree;
using namespace Roan_trail;

//
// Internal helpers
//

namespace
{
  static const string ih_default_dictionary_entry = "";
}

Error::~Error()
{
  precondition(mf_invariant(false));

  delete m_base_error;
}

//
// Operators
//

Error& Error::operator=(const Error& error)
{
  precondition(mf_invariant());

  if (this == &error)
  {
    goto exit_point;
  }

  m_error_dictionary = error.m_error_dictionary;

  // const cast here is OK because
  // because we are in operator=
  *const_cast<string*>(&m_file) = error.m_file;
  *const_cast<string*>(&m_function) = error.m_function;
  *const_cast<int*>(&m_line) = error.m_line;
  *const_cast<int*>(&m_code) = error.m_code;

  delete m_base_error;
  if (error.m_base_error)
  {
    m_base_error = error.m_base_error->clone();
  }
  else
  {
    m_base_error = 0;
  }

 exit_point:
  postcondition(mf_invariant());

  return *this;
}

//
// Operators
//

bool Error::operator==(const Error& error) const
{
  precondition(mf_invariant());

  bool return_value = false;

  const Error* current_error = this;
  const Error* cmp_current_error = &error;
  while (current_error && cmp_current_error)
  {
    if ((current_error->m_error_dictionary == cmp_current_error->m_error_dictionary)
        && (current_error->m_function == cmp_current_error->m_file)
        && (current_error->m_line == cmp_current_error->m_line)
        && (current_error->m_file == cmp_current_error->m_file)
        && (current_error->m_code == cmp_current_error->m_code))
    {
      current_error = current_error->base_error();
      cmp_current_error = cmp_current_error->base_error();
    }
    else
    {
      goto exit_point;
    }
  }

  return_value = !current_error && !cmp_current_error;

 exit_point:
  return return_value;
}

bool Error::operator!=(const Error& error) const
{
  precondition(mf_invariant());

  // operator== checks precondition
  return !operator==(error);
}

//
// Mutators
//

void Error::set_base_error(const Error* base_error)
{
  precondition(mf_invariant());

  delete m_base_error;
  m_base_error = base_error;

  postcondition(mf_invariant());
}

//
// Error keys
//

const string Error::code_string_error_key = "code_string_error_key";
const string Error::code_number_error_key = "code_number_error_key";
const string Error::diagnostic_error_key = "diagnostic_error_key";
const string Error::file_path_error_key = "file_path_error_key";
const string Error::to_path_error_key = "to_path_error_key";
const string Error::brief_description_error_key = "brief_description_error_key";
const string Error::detailed_description_error_key = "detailed_description_error_key";
const string Error::recovery_description_error_key = "recovery_description_error_key";
const string Error::error_dump_error_key = "error_dump_error_key";

//
// Error chain
//

void Error::format_chain_as_dictionary(Error_dictionary& return_dictionary) const
{
  return_dictionary.clear();
  const Error* user_error = user_error_for_chain();
  if (user_error)
  {
    return_dictionary = user_error->error_dictionary();
  }
  else
  {
    return_dictionary = error_dictionary();
  }
  stringstream error_dump;
  error_dump << *this;
  return_dictionary[error_dump_error_key] = error_dump.str();
}

const string& Error::dictionary_entry_for_chain(const string& key) const
{
  precondition(mf_invariant());

  const string* return_value_ptr = &ih_default_dictionary_entry;

  const Error* current_error = this;
  while (current_error)
  {
    const Error_dictionary& dict = current_error->m_error_dictionary;
    Error_dictionary::const_iterator p;
    p = dict.find(key);
    if (p != dict.end())
    {
      return_value_ptr = &p->second;
      goto exit_point;
    }
    current_error = current_error->m_base_error;
  }

 exit_point:
  return *return_value_ptr;
}

// returns 0 if no user error

const Error* Error::user_error_for_chain() const
{
  precondition(mf_invariant());

  const Error *return_value = 0;

  const Error* current_error = this;
  while (current_error)
  {
    if (current_error->mf_is_user_error())
    {
      return_value = current_error;
      goto exit_point;
    }
    current_error = current_error->base_error();
  }

 exit_point:
  return return_value;
}

int Error::format_message_for_chain(bool use_details, string& return_message)
{
  stringstream error_stream;

  error_stream << endl;

  const Error* user_error_in_chain = user_error_for_chain();
  const Error* user_error = user_error_in_chain ? user_error_in_chain : this;
  if (!user_error_in_chain)
  {
    error_stream << "(No user error found, check installation.)" << endl;
  }

  bool have_text = false;
  const Error::Error_dictionary& error_dict = user_error->error_dictionary();
  Error::Error_dictionary::const_iterator i;
  i = error_dict.find(Error::brief_description_error_key);
  if (error_dict.end() != i)
  {
    string brief_description = i->second;
    if ("" != brief_description)
    {
      error_stream << brief_description << endl;
      have_text = true;
    }
  }
  i = error_dict.find(Error::detailed_description_error_key);
  if (error_dict.end() != i)
  {
    string detailed_description = i->second;
    if ("" != detailed_description)
    {
      error_stream << detailed_description << endl << endl;
      have_text = true;
    }
  }
  i = error_dict.find(Error::recovery_description_error_key);
  if (error_dict.end() != i)
  {
    string recovery_description = i->second;
    if ("" != recovery_description)
    {
      error_stream << recovery_description << endl;
      have_text = true;
    }
  }

  if (!have_text)
  {
    error_stream << "(Unknown error encountered, check installation.)" << endl;
    error_stream << "  class: " << user_error->error_class() << endl;
    error_stream << "  code: " << user_error->code() << endl;
    const string& diagnostic = user_error->dictionary_entry_for_chain(Error::diagnostic_error_key);
    if ("" != diagnostic)
    {
      error_stream << "  diagnostic: " << diagnostic << endl;
    }
  }

  if (use_details)
  {
    error_stream << "Detailed error:" << endl;
    error_stream << *user_error;
  }

  return_message = error_stream.str();
  return user_error->code();
}

//
// Protected member functions
//

//
//   Constructors
//

Error::Error(const char* file,
             const char* function,
             int line,
             int code,
             const Error* base_error)
  : m_error_dictionary(),
    m_file(file ? file : string("")),
    m_function(function ? function : string("")),
    m_line(line),
    m_code(code),
    m_base_error(base_error)
{
  mf_set_error_code_number(code);

  postcondition(mf_invariant(false));
}

Error::Error(int code, const Error* base_error)
  : m_error_dictionary(Error_dictionary()),
    m_file(),
    m_function(),
    m_line(0),
    m_code(code),
    m_base_error(base_error)
{
  mf_set_error_code_number(code);

  postcondition(mf_invariant(false));
}

Error::Error(const Error& error)
  : m_error_dictionary(error.m_error_dictionary),
    m_file(error.m_file), m_function(error.m_function),
    m_line(error.m_line), m_code(error.m_code), m_base_error(0)
{
  if (error.m_base_error)
  {
    m_base_error = error.m_base_error->clone();
  }

  postcondition(mf_invariant(false));
}

//
//   Invariant
//

bool Error::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  Error_dictionary::const_iterator i = m_error_dictionary.find(Error::code_number_error_key);
  if (i == m_error_dictionary.end())
  {
    goto exit_point;
  }
  else
  {
    const string& code_entry = i->second;
    if ("" == code_entry)
    {
      goto exit_point;
    }
    else
    {
      stringstream s;
      int code;
      s << code_entry;
      s >> code;
      if (m_code != code)
      {
        goto exit_point;
      }
      else
      {
        return_value = true;
      }
    }
  }

 exit_point:
  return return_value;
}

void Error::mf_set_descriptions_from_file(const string &file)
{
  const string &error_code_key = m_error_dictionary[Error::code_string_error_key];
  assert(error_code_key != "" && "invalid error code when reading descriptions");

  ptree pt;

  string errors_file = m_error_directory + "/" + file;
  try
  {
    read_xml(errors_file, pt);
  }
  catch (...)
  {
    // if in development, errors file will be in the data directory of the source, so try again
    try
    {
      errors_file = m_error_directory + "/../../data/" + file;
      read_xml(errors_file, pt);
    }
    catch (std::exception& e)
    {
      std::cerr << "exception parsing error description file:" << endl;
      std::cerr << errors_file << endl;
      std::cerr << "reason: " << e.what() << endl;
    }
    catch (...)
    {
      std::cerr << "exception parsing error description file :" << endl;
      std::cerr << errors_file << endl;
    }
  }

  try
  {
    // set brief description
    string brief_description = pt.get(error_code_key + ".brief_description", "");
    if ("" != brief_description)
    {
      m_error_dictionary[brief_description_error_key] = brief_description;
    }

    // set detailed description
    string detailed_description;
    ptree default_child_pt;
    BOOST_FOREACH(ptree::value_type &v,
                  pt.get_child(error_code_key + ".detailed_description", default_child_pt))
    {
      if (v.first == "string")
      {
        detailed_description += v.second.data();
      }
      else if (v.first == "argument")
      {
        string argument = m_error_dictionary[v.second.data()];
        detailed_description += argument;
      }
      else if (v.first == "optional")
      {
        string optional_key = v.second.get("optional_key", "");
        if (optional_key != "")
        {
          string optional_entry = dictionary_entry_for_chain(optional_key);
          if (optional_entry != "")
          {
            BOOST_FOREACH(ptree::value_type &u, v.second)
            {
              if (u.first == "string")
              {
                detailed_description += u.second.data();
              }
              else if (u.first == "argument")
              {
                string argument = dictionary_entry_for_chain(u.second.data());
                detailed_description += argument;
              }
              // no final else needed (additional entries are ignored)
            }
          }
        }
      }
      // no final else needed (additional entries are ignored)
    }
    if ("" != detailed_description)
    {
      m_error_dictionary[detailed_description_error_key] = detailed_description;
    }
    // set recovery description
    string recovery_description = pt.get(error_code_key + ".recovery_description", "");
    if ("" != recovery_description)
    {
      m_error_dictionary[recovery_description_error_key] = recovery_description;
    }
  }
  catch (const std::exception& e)
  {
    std::cerr << "exception parsing error description file:" << endl;
    std::cerr << errors_file << endl;
    std::cerr << "reason: " << e.what() << endl;
  }
  catch (...)
  {
    std::cerr << "exception parsing error description file :" << endl;
    std::cerr << errors_file << endl;
  }
}

//
// Private member functions
//

void Error::mf_set_error_code_number(int code)
{
  stringstream code_string;
  code_string << code;
  m_error_dictionary[Error::code_number_error_key] = code_string.str();
}

//
// Private static members
//

string Error::m_error_directory;

//
// Operators
//

ostream& Roan_trail::operator<<(ostream& s, const Error &e)
{
  precondition(e.mf_invariant());

  s << "file: " << e.m_file << ":" << e.m_line << ": " << e.error_class() << endl;
  s << "function: " << e.m_function << endl;
  s << "error class: " << e.error_class() << endl;
  s << "code: " << e.m_code << endl;
  s << "error dictionary:" << endl;
  for (Error::Error_dictionary::const_iterator i = e.m_error_dictionary.begin();
       i != e.m_error_dictionary.end();
       ++i)
  {
    s << "  key: " << i->first << endl;
    s << "  value: " << i->second << endl;
  }

  if (e.m_base_error)
  {
    s << endl << "(base error:) -->" << endl;
    s << *e.m_base_error;
  }

  postcondition(e.mf_invariant());

  return s;
}

//
// Error_param helper class
//

//
// Constructor/destructor
//

Error_param::Error_param(bool need_error)
  : m_need_error(need_error),
    m_error(0)
{
  postcondition(mf_invariant(false));
}

//
// Operators
//

Error_param& Error_param::operator=(Error* error)
{
  precondition(!m_error
               && m_need_error
               && mf_invariant());

  m_error = error;

  postcondition(m_error == error
                && mf_invariant());

  return *this;
}

//
// Protected member functions
//

bool Error_param::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (!m_need_error && m_error)
  {
    goto exit_point;
  }

  return_value = true;

  goto exit_point;

 exit_point:
  return return_value;
}
