// Speech_synthesizer.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Speech_synthesizer.hpp"
#include "Speech_synthesizer_config.hpp"
#include "Sound_file_config.hpp"
#include "Sndfile_compat.hpp"
#include "File_manager.hpp"
#include "Logger.hpp"
#include <festival.h> // need this there to avoid conflict with Roan Trail error macro
#include "error/Record_error.hpp"
#include "error/Posix_error.hpp"
#include "error/Sndfile_error.hpp"
#include <boost/integer_traits.hpp>
#include <sndfile.h>
#include <iostream>
#include <iomanip>
#include <sstream>
#include <string>
#include <cmath>
#include <cstring>
#include <cerrno> // Standards Rule 17 exception (check_code_ignore)
#include <strings.h>
#include <fcntl.h>

using std::cerr;
using std::endl;
using std::setfill;
using std::setw;
using std::string;
using std::stringstream;
using boost::integer_traits;
using Roan_trail::Long_int;
using Roan_trail::Error;
using namespace Roan_trail::Recorder;

//
// Internal constants/variables
//

namespace
{
  const double ic_remaining_adjustment = 0.90;
  const double ic_fraction_available_warning_threshold = 0.10;
  const double ic_seconds_remaining_warning_threshold = 30.0;

  const int ic_festival_heap_size = 210000;
  const int ic_frames_per_buffer = 512;
  const int ic_default_channels = 1;

  bool iv_time_digits_cache_set = false;
  char iv_time_digits_cache[100][3];

  void ih_setup_time_digits_cache()
  {
    if (!iv_time_digits_cache_set)
    {
      stringstream time_digits;
      for (size_t i = 0; i < 100; ++i)
      {
        time_digits.str("");
        time_digits << setw(2) << setfill('0') << i;
        const string time_digits_str = time_digits.str();
        assert((2 == time_digits_str.length()) && "time digits string length not equal to 2");
        const char *digits_buf = time_digits_str.c_str();
        iv_time_digits_cache[i][0] = digits_buf[0];
        iv_time_digits_cache[i][1] = digits_buf[1];
        iv_time_digits_cache[i][2] = 0;
      }
      iv_time_digits_cache_set = true;
    }
  }
}

//
// Constructor/destructor
//

Speech_synthesizer::Speech_synthesizer()
  : m_frames_written_count(0),
    m_speech_initialized(false),
    m_config(new Speech_synthesizer_config),
    m_logger(Logger::default_logger()),
    m_have_recording(false),
    m_frames_per_second(0),
    m_frames_per_minute(0),
    m_frames_per_hour(0)
{
  ih_setup_time_digits_cache();

  postcondition(mf_invariant(false));
}

Speech_synthesizer::~Speech_synthesizer()
{
  precondition(mf_invariant(false));

  if (m_speech_initialized)
  {
    Error_param e(false);
    shutdown(false, e); // ignore error
  }

  delete m_config;
}

//
// Control
//

//
//   ...startup/shutdown
//

bool Speech_synthesizer::startup(const Speech_synthesizer_config& config, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  bool festival_initialized = false;
  bool file_open = false;
  bool created_temp_file = false;
  Speech_synthesizer_config save_config = *m_config;

  start_error_block();

  *m_config = config;

  if (m_speech_initialized)
  {
    return_value = true;
    goto exit_point;
  }
  else
  {
    // initialize Festival speech synthesizer
    const int load_init_files = 1;
    m_logger << Logger::info << "Initializing speech synthesizer subsystem" << endl;
    festival_initialize(load_init_files, ic_festival_heap_size);
    //   voice
    if ("" != m_config->voice)
    {
      // not using Festival's default voice, use the one specified
      m_logger << Logger::info << "Setting voice to: " << m_config->voice << endl;
      const string voice_setup_command = Speech_synthesizer_config::setup_command_for_voice(m_config->voice);
      on_error("" == voice_setup_command, new Record_error(error_location(),
                                                           Record_error::startup,
                                                           string("could not locate Festival voice: ")
                                                           + m_config->voice));
      m_logger << Logger::info << "Sending voice setup command: " << voice_setup_command << endl;
      EST_String voice_command(voice_setup_command.c_str());
      const int voice_evaluated = festival_eval_command(voice_command);
      on_error(!voice_evaluated, new Record_error(error_location(),
                                                  Record_error::startup,
                                                  string("could not evaluate Festival voice setup command: ")
                                                  + voice_setup_command));
    }
    else
    {
      m_logger << Logger::info << "Using default voice" << endl;
    }


    festival_initialized = true;

    // open output file
    m_logger << Logger::info << "Opening sound file" << endl;
    Error_param error;
    string sound_file_path;
    SNDFILE* sound_file = mf_open_sound_file(sound_file_path,
                                             created_temp_file,
                                             error);
    on_error(!sound_file, new Record_error(error_location(),
                                           Record_error::general,
                                           error()));
    file_open = true;

    m_output_file = sound_file;

    m_frames_per_second = static_cast<Long_int>(m_config->sound_file->sample_rate);
    m_frames_per_minute = m_frames_per_second * 60;
    m_frames_per_hour = m_frames_per_minute * 60;
  }

  __sync_and_and_fetch(&m_frames_written_count, 0x0);
  m_have_recording = false;

  m_speech_initialized = true;
  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }

  if (festival_initialized)
  {
    // intentionally left blank
  }

  m_frames_per_second = 0;
  m_frames_per_minute = 0;
  m_frames_per_hour = 0;
  *m_config = save_config;

  m_speech_initialized = false;
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Speech_synthesizer::shutdown(bool completed, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  if (!m_speech_initialized)
  {
    return_value = true;
    goto exit_point;
  }

  string output_file_path = output_file();

  int close_ret = sf_close(m_output_file);
  if (!completed && !close_ret)
  {
    if ("" != output_file_path) // paranoia
    {
      m_logger << Logger::info;
      m_logger <<"Synthesis not completed, removing output sound file: " << endl;
      m_logger << output_file_path << endl;

      unlink(output_file_path.c_str());
    }
  }

  m_frames_per_second = 0;
  m_frames_per_minute = 0;
  m_frames_per_hour = 0;

  m_output_file = 0;
  m_temporary_file_path = "";
  m_speech_initialized = false;

  on_error(close_ret, new Record_error(error_location(),
                                       Record_error::shutdown,
                                       new Sndfile_error(close_ret,
                                                         "sf_close",
                                                         0)));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(!m_speech_initialized
                && return_value
                && !return_error()
                && mf_invariant());
  return return_value;
}

//
//   ...basic synthesis functions
//

bool Speech_synthesizer::speak(const string& text, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  on_error(!m_speech_initialized,
           new Record_error(error_location(),
                            Record_error::record,
                            string("Error, speech synthesizer not initialized when attempting to speak.")));

  EST_Wave wave;

  m_logger << Logger::info;
  m_logger << "*** Creating audio for the following text:" << endl;
  m_logger << text << endl;
  m_logger << "*** End of spoken text." << endl;

  EST_String convert_text(text.c_str());

  const int got_wave = festival_text_to_wave(convert_text, wave);
  on_error(!got_wave, new Record_error(error_location(),
                                       Record_error::record,
                                       string("Error, voice synthesizer could not convert text.")));

  // Festival generates at its own default sample rate,
  // resample to the required sample rate
  if (EST_Wave::default_sample_rate != m_config->sound_file->sample_rate)
  {
    wave.resample(m_config->sound_file->sample_rate);
  }

  const int items_to_write = wave.num_samples() * wave.num_channels();
  const sf_count_t items_written = sf_write_short(m_output_file,
                                                  wave.values().memory(),
                                                  items_to_write);
  if (items_written != items_to_write)
  {
    // amount of data written doesn't match the amount requested
    // figure out what the error is
    const int sound_file_error = sf_error(m_output_file);
    if (SF_ERR_NO_ERROR == sound_file_error)
      {
        stringstream diagnostic;
        diagnostic << "file write failed, items written: " << items_written;
        diagnostic << " does not equal requested items to write: " << items_to_write;
        on_error(true, new Record_error(error_location(),
                                        Record_error::record,
                                        diagnostic.str()));
      }
    else
    {
      const string output_file_path = output_file();
      if (SF_ERR_SYSTEM == sound_file_error)
      {
        const string parent_directory = File_manager::parent_directory_for_path(output_file_path);
        double dummy_fraction = 0.0;
        Long_int available_bytes = 0L;
        Error_param error(false);
        if (File_manager::file_system_available_for_path(parent_directory,
                                                         dummy_fraction,
                                                         available_bytes,
                                                         error) && (0 == available_bytes))
        {
          stringstream diagnostic;
          diagnostic << "The file system is full for file: " << output_file_path;
          on_error(true, new Record_error(error_location(),
                                          Record_error::record,
                                          diagnostic.str()));
        }
      }
      on_error(true, new Record_error(error_location(),
                                      Record_error::record,
                                      new Sndfile_error(sound_file_error,
                                                        "sf_write_float",
                                                        output_file_path.c_str())));
    }
  }
  sf_write_sync(m_output_file); // force writing of all file cache buffers
  __sync_add_and_fetch(&m_frames_written_count, wave.num_samples());

  m_have_recording = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(( // function success
                 (return_value
                  && m_have_recording)
                 // function failure
                 || !return_value)
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// Accessors
//

bool Speech_synthesizer::is_started() const
{
  precondition(mf_invariant());

  return m_speech_initialized;
}

bool Speech_synthesizer::can_record() const
{
  precondition(mf_invariant());

  return m_speech_initialized;
}

void Speech_synthesizer::frames_to_recorded_time(Long_int frames,
                                                 char separator,
                                                 string& return_recorded_time) const
{
  precondition((frames >= 0)
               && mf_invariant());

  const Long_int seconds = (frames / m_frames_per_second) % 60;
  const Long_int minutes = (frames / m_frames_per_minute) % 60;
  Long_int hours = frames / m_frames_per_hour;
  if (hours > 99)
  {
    hours = 0;
  }
  assert((seconds < 60)
         && (minutes < 60)
         && (hours < 100)
         && "error, invalid HMS");

  stringstream recorded_time;
  recorded_time << iv_time_digits_cache[hours] << separator;
  recorded_time << iv_time_digits_cache[minutes] << separator;
  recorded_time << iv_time_digits_cache[seconds];

  return_recorded_time = recorded_time.str();

  postcondition((return_recorded_time.length() == string("HH:MM:SS").length()));
}

void Speech_synthesizer::recorded_time(string& return_recorded_time, char separator) const
{
  precondition(mf_invariant());

  const Long_int frames_recorded = frames_written_count();

  frames_to_recorded_time(frames_recorded, separator, return_recorded_time);
}

string Speech_synthesizer::output_file() const
{
  precondition(mf_invariant());

  const string blank("");
  const string *return_value_ptr = &blank;
  string temporary_output_file;

  if (!is_started())
  {
    goto exit_point;
  }
  else if (m_config->sound_file->file_name != "")
  {
    return_value_ptr = &m_config->sound_file->file_name;
    goto exit_point;
  }
  else
  {
    return_value_ptr = &m_temporary_file_path;
  }
  goto exit_point;

 exit_point:
  postcondition(((is_started() && (*return_value_ptr != ""))
                 || (!is_started() && (*return_value_ptr == ""))));
  return *return_value_ptr;
}

// returns false if available space/time is less than thresholds
// return_bytes_available, return_fraction_available, and return_seconds_remaining are set
// to 0 if the directory isn't found
bool Speech_synthesizer::space_available(Long_int& return_bytes_available,
                                         double& return_fraction_available,
                                         double& return_seconds_remaining) const
{
  precondition(mf_invariant());

  bool return_value = false;

  double fraction_available = 0.0;
  Long_int available_bytes = 0;
  double seconds_remaining = 0.0;
  const string directory = File_manager::parent_directory_for_path(output_file());
  Error_param e(false);
  if (!File_manager::file_system_available_for_path(directory,
                                                    fraction_available,
                                                    available_bytes,
                                                    e))
  {
    return_bytes_available = 0;
    return_fraction_available = 0.0;
    return_seconds_remaining = 0.0;
    return_value = true;
    goto exit_point;
  }
  else
  {
    const double rate = m_config->sound_file->sample_rate;
    const double channels = ic_default_channels;
    size_t bytes_per_sample;
    if (!Sound_file_config::sample_size_for_data_format(m_config->sound_file->data_format,
                                                        bytes_per_sample))
    {
      assert(false && "invalid data format");
    }

    seconds_remaining = (static_cast<double>(available_bytes)
                         / (rate * static_cast<double>(bytes_per_sample) * channels))
      * ic_remaining_adjustment;

    return_bytes_available = available_bytes;
    return_fraction_available = fraction_available;
    return_seconds_remaining = seconds_remaining;

    return_value = ((fraction_available > ic_fraction_available_warning_threshold)
                    && (seconds_remaining > ic_seconds_remaining_warning_threshold));
  }

 exit_point:
  return return_value;
}

// currently always returns true
// return_file_size and return_data_rate are set to 0 if there is no output file
bool Speech_synthesizer::output_file_size(Long_int& return_file_size,
                                          double& return_data_rate) const
{
  precondition(mf_invariant());

  Long_int file_size = 0;
  double data_rate = 0.0;

  string file_path = output_file();
  if ("" == file_path)
  {
    return_file_size = 0;
    return_data_rate = 0.0;

    goto exit_point;
  }
  else
  {
    Error_param e(false);
    if (!File_manager::file_size_for_path(file_path,
                                          file_size,
                                          e))
    {
      goto exit_point;
    }

    const double rate = m_config->sound_file->sample_rate;
    const double channels = ic_default_channels;
    size_t bytes_per_sample;
    if (!Sound_file_config::sample_size_for_data_format(m_config->sound_file->data_format,
                                                        bytes_per_sample))
    {
      assert(false && "invalid data format");
    }

    data_rate = rate * channels * static_cast<double>(bytes_per_sample);

    if (m_config->sound_file->is_compressed())
    {
      data_rate /= 2;
    }

    return_file_size = file_size;
    return_data_rate = data_rate;
  }

 exit_point:
  return true;
}

//
// Protected member functions
//

// invariant check
bool Speech_synthesizer::mf_invariant(bool check_base_class) const
{
  bool return_value = false;
  {
    // member objects allocated check
    if (!m_config)
    {
      goto exit_point;
    }

    if ((m_speech_initialized
         && (m_frames_per_second < 1)
         && (m_frames_per_minute < 1)
         && (m_frames_per_hour < 1))
        || (!m_speech_initialized
            && (m_frames_per_second != 0)
            && (m_frames_per_minute != 0)
            && (m_frames_per_hour != 0)))
    {
      goto exit_point;
    }

    if ((m_speech_initialized
         && (!m_output_file
             || (((m_config->sound_file->file_name == "") && (m_temporary_file_path == ""))
                 || ((m_config->sound_file->file_name != "") && (m_temporary_file_path != "")))))
        || (!m_speech_initialized
            && (m_output_file
                || ("" != m_temporary_file_path))))
    {
      goto exit_point;
    }

    return_value = true;
  }

 exit_point:
  return return_value;
}

//
// private member functions
//

SNDFILE* Speech_synthesizer::mf_open_sound_file(string& return_output_file_path,
                                                bool& return_created_temp_file,
                                                Error_param& return_error)
{
  precondition(!return_error()
               && ("" == m_temporary_file_path));

  return_created_temp_file = false;

  SNDFILE* return_value = false;

  // variable declarations for error cleanup
  bool created_temp_file = false;
  string temporary_file_name = "";
  SNDFILE* file = 0;

  start_error_block();

  // use libsndfile to open the file
  SF_INFO sf_info;
  sf_info.samplerate = static_cast<int>(m_config->sound_file->sample_rate);
  sf_info.frames = ic_frames_per_buffer;
  sf_info.channels = ic_default_channels;
  if (!Sndfile_compat::sf_format_for_config(*m_config->sound_file, sf_info.format)
      || !sf_format_check(&sf_info))
  {
    // invalid sf_info structure
    string prefix("  ");
    stringstream diagnostic;
    diagnostic << "The combination:" << endl <<  m_config->sound_file->format_string(prefix);
    diagnostic << "is invalid.  Not all combinations of these options are allowed.";
    Record_error *format_error = new Record_error(error_location(),
                                                  Record_error::open,
                                                  diagnostic.str());
    on_error(true, format_error);
  }

  string output_file = m_config->sound_file->file_name;
  bool overwrite = m_config->file_overwrite;

  int close_on_sf_close = 1;
  string error_file_path;
  if (output_file == "")
  {
    // create temporary output file
    int fd;
    string file_extension = string(".")
      + Sound_file_config::string_for_file_type(m_config->sound_file->file_type);
    Error_param error;
    created_temp_file = File_manager::create_temporary_file(file_extension,
                                                            File_manager::temporary_sound_file,
                                                            temporary_file_name,
                                                            fd,
                                                            error);
    on_error(!created_temp_file, new Record_error(error_location(),
                                                  Record_error::open,
                                                  error()));
    return_created_temp_file = true;
    file = sf_open_fd(fd, SFM_WRITE, &sf_info, close_on_sf_close);
    error_file_path = string(temporary_file_name.c_str());
    on_error_with_label(!file,
                        sf_open_fd_error_handler,
                        new Sndfile_error(sf_error(0),
                                          "sf_open_fd",
                                          error_file_path.c_str()));

    m_temporary_file_path = temporary_file_name;
    output_file = temporary_file_name;
  }
  else
  {
    // output file specified
    const char* file_path = output_file.c_str();
    error_file_path = string(file_path);
    int oflag = O_CREAT;
    int mode = S_IRUSR | S_IWUSR;
    if (!overwrite)
    { // no overwrite:
      oflag |= O_WRONLY;
      // using O_CREAT and O_EXCL flags together will cause an error to be returned
      // if the file exists
      oflag |= O_EXCL;
      int fd = open(file_path,
                    oflag,
                    mode);
      if (Posix_error::error_return == fd)
      {
        if (EEXIST == errno) // (check_code_ignore)
        {
          Error *overwrite_error = new Record_error(error_location(),
                                                    Record_error::open,
                                                    string("output file exists and overwrite not requested"));
          overwrite_error->error_dictionary()[Error::file_path_error_key] = output_file;
          on_error(true, overwrite_error);
        }
        else
        {
          on_error(true, new Record_error(error_location(),
                                          Record_error::open,
                                          new Posix_error(errno, // (check_code_ignore)
                                                          "open",
                                                          error_file_path.c_str())));
        }
      }
      file = sf_open_fd(fd,
                        SFM_WRITE,
                        &sf_info,
                        close_on_sf_close);
      on_error_with_label(!file,
                          sf_open_fd_error_handler,
                          new Sndfile_error(sf_error(0),
                                            "sf_open_fd",
                                            error_file_path.c_str()));
    }
    else
    {
      // overwrite
      oflag |= O_RDWR;
      int fd = open(file_path, oflag, mode);
      on_error(Posix_error::error_return == fd, new Record_error(error_location(),
                                                                 Record_error::open,
                                                                 new Posix_error(errno, // (check_code_ignore)
                                                                                 "open",
                                                                                 error_file_path.c_str())));
      file = sf_open_fd(fd,
                        SFM_RDWR,
                        &sf_info,
                        close_on_sf_close);
      on_error_with_label(!file,
                          sf_open_fd_error_handler,
                          new Sndfile_error(sf_error(0),
                                            "sf_open_fd",
                                            error_file_path.c_str()));
      sf_count_t count = sf_seek(file,
                                 0,
                                 SEEK_SET);
      on_error(-1 == count, new Record_error(error_location(),
                                             Record_error::open,
                                             new Sndfile_error(sf_error(file),
                                                               "sf_seek",
                                                               error_file_path.c_str())));
    }
  }

  return_value = file;
  goto exit_point;

  end_error_block();

 sf_open_fd_error_handler:
  if (return_error.need_error())
  {
    if (SF_ERR_SYSTEM == handler_error->code())
    {
      const string &output_file = handler_error->error_dictionary()[Error::file_path_error_key];
      if ("" != output_file)
      {
        const string parent_directory = File_manager::parent_directory_for_path(output_file);
        double dummy_fraction = 0.0;
        Long_int available_bytes = 0L;
        Error_param error(false);
        if (File_manager::file_system_available_for_path(parent_directory,
                                                         dummy_fraction,
                                                         available_bytes,
                                                         error) && (0 == available_bytes))
        {
          stringstream diagnostic;
          diagnostic << "The file system is full for file: " << output_file;
          return_error = new Record_error(error_location(),
                                          Record_error::open,
                                          diagnostic.str(),
                                          string(""),
                                          handler_error);
          goto error_cleanup;
        }
      }
    }

    return_error = new Record_error(error_location(),
                                    Record_error::open,
                                    handler_error);
  }
  goto error_cleanup;

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (file)
  {
    sf_close(file); // ignore error
  }
  if (created_temp_file)
  {
    if ("" != temporary_file_name)
    {
      unlink(temporary_file_name.c_str()); // ignore error
    }
    m_temporary_file_path = "";
  }
  return_value = 0;
  goto exit_point;

 exit_point:
  postcondition((return_value
                 || (!return_value && ("" == m_temporary_file_path)))
                && return_error.is_valid_at_return(return_value != 0));


  return return_value;
}

