// Sound_recorder_config.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Sound_recorder_config.hpp"
#include "Sound_file_config.hpp"
#include "common.hpp"
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>
#include <map>
#include <string>

using boost::lexical_cast;
using boost::algorithm::to_lower;
using std::map;
using std::string;
using namespace Roan_trail::Recorder;

//
// Constructor/destructor
//

Sound_recorder_config::Sound_recorder_config()
  // setup some reasonable defaults
  : input_device(default_input_device),
    frames_per_buffer(default_frames_per_buffer),
    write_buffer_factor(default_write_buffer_factor),
    RMS_level_integration_period(default_RMS_level_integration_period),
    sound_file(new Sound_file_config),
    file_overwrite(false)
{
}

Sound_recorder_config::~Sound_recorder_config()
{
  delete sound_file;
}

Sound_recorder_config::Sound_recorder_config(const Sound_recorder_config& r)
  : input_device(r.input_device),
    frames_per_buffer(r.frames_per_buffer),
    write_buffer_factor(r.write_buffer_factor),
    RMS_level_integration_period(r.RMS_level_integration_period),
    sound_file(new Sound_file_config),
    file_overwrite(r.file_overwrite)
{
  *sound_file = *r.sound_file;
}

Sound_recorder_config& Sound_recorder_config::operator=(const Sound_recorder_config& r)
{
  if (&r != this)
  {
    input_device = r.input_device;
    frames_per_buffer = r.frames_per_buffer;
    write_buffer_factor = r.write_buffer_factor;
    RMS_level_integration_period = r.RMS_level_integration_period;
    *sound_file = *r.sound_file;
    file_overwrite = r.file_overwrite;
  }

  return *this;
}

//
// Operators
//

bool Sound_recorder_config::operator==(const Sound_recorder_config& config) const
{
  bool eq = (config.input_device == input_device)
    && (config.frames_per_buffer == frames_per_buffer)
    && (config.write_buffer_factor == write_buffer_factor)
    && (config.RMS_level_integration_period == RMS_level_integration_period)
    && (*config.sound_file == *sound_file)
    && (config.file_overwrite == config.file_overwrite);

  return eq;
}

bool Sound_recorder_config::operator!=(const Sound_recorder_config& config) const
{
  // operator== checks precondition
  return !operator==(config);
}

bool Sound_recorder_config::from_map(const map<string, string>& config_map)
{
  bool found_error = false;

  for (map<string, string>::const_iterator i = config_map.begin(); i != config_map.end(); ++i)
  {
    try
    {
      string key = i->first;
      to_lower(key);
      string entry = i->second;
      if ("input_device" == key)
      {
        input_device = lexical_cast<int>(entry);
      }
      else if ("frames_per_buffer" == key)
      {
        frames_per_buffer = lexical_cast<int>(entry);
      }
      else if ("write_buffer_factor" == key)
      {
        write_buffer_factor = lexical_cast<int>(entry);
      }
      else if ("RMS_level_integration_period" == key)
      {
        RMS_level_integration_period = lexical_cast<int>(entry);
      }
      else if ("file_overwrite" == key)
      {
        file_overwrite = lexical_cast<bool>(entry);
      }
      else
      {
        found_error = sound_file->from_map(config_map);
      }
    }
    catch (...)
    {
      found_error = true;
    }
  }

  return !found_error;
}

bool Sound_recorder_config::to_map(map<string, string>& return_config_map) const
{
  bool found_error = false;

  try
  {
    return_config_map["input_device"] = lexical_cast<string>(input_device);
  }
  catch (...)
  {
    found_error = true;
  }
  try
  {
    return_config_map["frames_per_buffer"] = lexical_cast<string>(frames_per_buffer);
  }
  catch (...)
  {
    found_error = true;
  }
  try
  {
    return_config_map["write_buffer_factor"] = lexical_cast<string>(write_buffer_factor);
  }
  catch (...)
  {
    found_error = true;
  }
  try
  {
    return_config_map["RMS_level_integration_period"] = lexical_cast<string>(RMS_level_integration_period);
  }
  catch (...)
  {
    found_error = true;
  }
  try
  {
    return_config_map["overwrite"] = lexical_cast<string>(file_overwrite);
  }
  catch (...)
  {
    found_error = true;
  }
  if (!sound_file->to_map(return_config_map))
  {
    found_error = true;
  }
  return !found_error;
}

//
// Class constants
//

// default constants (some from libsndfile)
const int Sound_recorder_config::default_input_device = -1;
const int Sound_recorder_config::default_frames_per_buffer = 512;
const int Sound_recorder_config::default_write_buffer_factor = 32;
const int Sound_recorder_config::default_RMS_level_integration_period = 10; // milliseconds
