// Sound_recorder.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef SOUND_RECORDER_HPP_
#define SOUND_RECORDER_HPP_

#include "common.hpp"
#include "Recorder.hpp"
#include <string>
#include <vector>

struct PaStreamCallbackTimeInfo;
typedef void PaStream;
typedef struct SNDFILE_tag SNDFILE;

namespace Roan_trail
{
  class Error;
  class Error_param;

  namespace Recorder
  {
    class Async_audio_file_writer;
    struct Sound_recorder_config;

    class Sound_recorder : public Recorder
    {
    public:
      // constructor/destructor
      Sound_recorder();
      virtual ~Sound_recorder();
      // control
      //
      //  Note: The Sound_recorder class is meant to be used directly from a library,
      //        so the control functions are forgiving of errors such as recording state.
      //        They will generally return an error rather than using an assertion.
      //
      //   startup/shutdown functions
      bool startup(const Sound_recorder_config &config, Error_param& return_error);
      bool shutdown(Error_param& return_error);
      //   basic Recorder functions
      bool record(Error_param& return_error);
      bool pause(Error_param& return_error);
      bool stop(Error_param& return_error);
      //   sound recorder functions
      bool mute(Error_param& return_error); // toggles muting
      // accessors
      bool is_RMS_metering_enabled() const
      { return __sync_fetch_and_add(const_cast<int32_t*>(&m_calc_levels_RMS), 0x0); }
      //   caller supplies return_level array with sufficient (# channels) elements
      void sound_levels_RMS(std::vector<double>& return_levels) const;
      bool is_peak_metering_enabled() const
      { return __sync_fetch_and_add(const_cast<int32_t*>(&m_calc_levels_peak), 0x0); }
      int metering_channels() const
      { return __sync_fetch_and_add(const_cast<int32_t*>(&m_metering_channels), 0x0); }
      //   caller supplies return_level array with sufficient (# channels) elements
      void sound_levels_peak(std::vector<double>& return_levels, bool reset_peaks = true) const;
      Long_int frames_written_count() const;
      bool is_monitoring() const  { return __sync_fetch_and_add(const_cast<int32_t*>(&m_monitoring), 0x0); }
      bool is_muted() const  { return __sync_fetch_and_add(const_cast<int32_t*>(&m_muted), 0x0); }
      bool is_started() const;
      bool can_record() const;
      Long_int input_overflow_count() const
      { return __sync_fetch_and_add(const_cast<int64_t*>(&m_input_overflow_count), 0x0); }
      Long_int output_overflow_count() const
      { return __sync_fetch_and_add(const_cast<int64_t*>(&m_output_overflow_count), 0x0); }
      void frames_to_recorded_time(const Long_int frames,
                                   char separator,
                                   std::string& return_recorded_time) const;
      void recorded_time(std::string &return_recorded_time, char separator = ':') const;
      const Error* record_error() const;
      const Sound_recorder_config* config() const { return m_config; }
      bool have_recording() const { return m_have_recording; }
      std::string output_file() const;
      bool space_available(Long_int& return_bytes_available,
                           double& return_fraction_available,
                           double& return_seconds_remaining) const;
      bool output_file_size(Long_int& return_file_size, double& return_data_rate) const;
      // mutators
      void enable_RMS_metering(bool enable);
      void enable_peak_metering(bool enable);
      void set_metering_channels(int channel_count);
      void reset_input_overflow_count() { __sync_and_and_fetch(&m_input_overflow_count, 0x0); }
      void reset_output_overflow_count() { __sync_and_and_fetch(&m_output_overflow_count, 0x0); }
      void clear_record_error();
      // misc
      bool cleanup_temporary_file();
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      // aligned data members used for synchronized access
      int32_t __attribute__ ((aligned (4))) m_monitoring;
      int32_t __attribute__ ((aligned (4))) m_muted;
      int32_t __attribute__ ((aligned (4))) m_calc_levels_RMS;
      int32_t __attribute__ ((aligned (4))) m_calc_levels_peak;
      int32_t __attribute__ ((aligned (4))) m_metering_channels;
      mutable int32_t __attribute__ ((aligned (4))) m_reset_peaks;
      int64_t __attribute__ ((aligned (8))) m_input_overflow_count;
      int64_t __attribute__ ((aligned (8))) m_output_overflow_count;
      int32_t __attribute__ ((aligned (4))) m_have_callback_error;
      //
      Stream_conv_int* m_level_RMS;
      Stream_conv_int* m_level_peak;
      bool m_sound_initialized;
      Sound_recorder_config *m_config;
      bool m_overflow_logging_enabled;
      Async_audio_file_writer* m_AAFW;
      PaStream* m_PA_stream;
      int m_callback_record_error_code;
      bool m_have_recording;
      char *m_muted_buffer;
      Long_int m_frames_per_second;
      Long_int m_frames_per_minute;
      Long_int m_frames_per_hour;
      struct RMS_level_calc_struct
      {
        RMS_level_calc_struct();
        int integration_period; // milliseconds in RMS level average window
        int integration_count; // samples in RMS level average window (derived from period)
        double *square_sum; // array count is the number of channels
        double *square_cache; // array count is the number of channels times the integration period
        int square_cache_position; // range: 0 to integration period - 1
      } m_RMS_level_calc;
      // callbacks
      static void mf_stream_finished_callback(void* user_data);
      static int mf_audio_callback(const void* input_buffer,
                                   void* output_buffer,
                                   unsigned long frames_per_buffer,
                                   const PaStreamCallbackTimeInfo* time_info,
                                   unsigned long status_flags,
                                   void* user_data);
      // prevent compiler from generating
      Sound_recorder(const Sound_recorder& recorder);
      Sound_recorder& operator=(const Sound_recorder& recorder);
    };
  }
}

#endif // SOUND_RECORDER_HPP_
