// Sound_recorder.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.
//
// The code in the audio callback for maintaining a running sum of
// squares is based on src/envelopes.c from the Meterbridge project:
//   Released under GPL version 2
//   By Steve Harris 2002 steve -at- plugin.org.uk

#include "Sound_recorder.hpp"
#include "Sound_recorder_config.hpp"
#include "Sound_file_config.hpp"
#include "Async_audio_file_writer.hpp"
#include "File_manager.hpp"
#include "error/Record_error.hpp"
#include "error/AAFW_error.hpp"
#include "error/Portaudio_error.hpp"
#include "error/Posix_error.hpp"
#include <boost/integer_traits.hpp>
#include <portaudio.h>
#include <iomanip>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <cerrno> // Standards Rule 17 exception (check_code_ignore)
#include <cmath>
#include <cstring>

using std::cerr;
using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using boost::integer_traits;
using Roan_trail::Long_int;
using Roan_trail::Error;
using namespace Roan_trail::Recorder;

// PORTING: uses "memcpy" to copy (alias) int -> float and float -> int
// PORTING: look at assembler output to make sure memcpy is not
// PORTING: called as a function, but is optimized to a move
// PORTING: (currently gcc with -O2 correctly optimizes)
// PORTING: (use "objdump -S <obj_file>" where obj_file has been
// PORTING:  compiled with -g)

//
// Internal constants/variables
//

namespace
{
  const double ic_remaining_adjustment = 0.90;
  const double ic_fraction_available_warning_threshold = 0.10;
  const double ic_seconds_remaining_warning_threshold = 30.0;

  bool iv_time_digits_cache_set = false;
  char iv_time_digits_cache[100][3];

  void ih_setup_time_digits_cache()
  {
    if (!iv_time_digits_cache_set)
    {
      stringstream time_digits;
      for (size_t i = 0; i < 100; ++i)
      {
        time_digits.str("");
        time_digits << std::setw(2) << std::setfill('0') << i;
        const string time_digits_str = time_digits.str();
        assert((2 == time_digits_str.length()) && "time digits string length not equal to 2");
        const char *digits_buf = time_digits_str.c_str();
        iv_time_digits_cache[i][0] = digits_buf[0];
        iv_time_digits_cache[i][1] = digits_buf[1];
        iv_time_digits_cache[i][2] = 0;
      }
      iv_time_digits_cache_set = true;
    }
  }
}

//
// Static member functions
//

void Sound_recorder::mf_stream_finished_callback(void* user_data)
{
  Sound_recorder* sr = reinterpret_cast<Sound_recorder*>(user_data);
  static_cast<void>(sr); // Avoid unused warning
  // NOTE: setup for future use
}

////////////////////
// audio callback //
////////////////////
int Sound_recorder::mf_audio_callback(const void* input_buffer,
                                      void* output_buffer,
                                      unsigned long frames_per_buffer,
                                      const PaStreamCallbackTimeInfo* time_info,
                                      unsigned long status_flags,
                                      void* user_data)
{
  precondition(input_buffer
               && (frames_per_buffer >= 0)
               && user_data);

  // Avoid unused warnings
  static_cast<void>(output_buffer);
  static_cast<void>(time_info);
  static_cast<void>(status_flags);

  int return_value = paContinue;

  Sound_recorder* sr = reinterpret_cast<Sound_recorder*>(user_data);
  if (status_flags & paInputOverflow) // Note: paInputOverflow declaration uses an old-style cast
  {
    __sync_add_and_fetch(&sr->m_input_overflow_count, 0x1);
  }
  else if (status_flags)
  {
    if (!__sync_fetch_and_or(const_cast<int32_t*>(&sr->m_have_callback_error), 0x1))
    {
      sr->m_callback_record_error_code = Record_error::record_callback;
    }

    return_value = paAbort;
    goto exit_point;
  }
  // no final else needed, we've handled the status flags

  { // anonymous block to hide declarations from goto
    const bool muted = __sync_fetch_and_add(&sr->m_muted, 0x0);
    const void *buffer = (muted ? sr->m_muted_buffer
                          : input_buffer);

    // levels update
    const bool calc_levels_RMS = __sync_fetch_and_add(&sr->m_calc_levels_RMS, 0x0);
    const bool calc_levels_peak = __sync_fetch_and_add(&sr->m_calc_levels_peak, 0x0);

    if (calc_levels_RMS || calc_levels_peak)
    {
      Stream_conv_int cur_level;
      Stream_conv_int convert_level;
      Stream_float cur_sample;

      const int32_t metering_channel_count = __sync_fetch_and_add(&sr->m_metering_channels, 0x0);
      const size_t channel_count = min(static_cast<size_t>(sr->m_config->sound_file->channels),
                                       static_cast<size_t>(metering_channel_count));

      if (calc_levels_RMS)
      {
        Stream_float float_level;
        int position = 0;
        // calculate RMS levels
        for (size_t channel = 0; channel < channel_count; ++channel)
        {
          position = sr->m_RMS_level_calc.square_cache_position;
          const Stream_float* samples = reinterpret_cast<const Stream_float*>(buffer) + channel;
          const int row_index = sr->m_RMS_level_calc.integration_count * channel;
          for (unsigned long i = 0; i < frames_per_buffer; ++i)
          {
            cur_sample = *samples;

            // update the running sum of squares for this channel
            //   remove the previous square at the current position from the channel's sum
            int cache_index = row_index + position;
            sr->m_RMS_level_calc.square_sum[channel] -= sr->m_RMS_level_calc.square_cache[cache_index];
            double sample_squared = cur_sample * cur_sample;
            //   add the square of the current sample to the channel's sum
            sr->m_RMS_level_calc.square_sum[channel] += sample_squared;
            if (sr->m_RMS_level_calc.square_sum[channel] < 0.0)
            {
              sr->m_RMS_level_calc.square_sum[channel] = 0.0; // "massage" if we get a numerical cramp
            }
            //   keep a copy of it so we can remove it in the next callback (as above)
            sr->m_RMS_level_calc.square_cache[cache_index] = sample_squared;

            // update the position in the running sum of squares
            // (the integration count is a power of 2, so it's value - 1 is used as a mask)
            position = (position + 1) & (sr->m_RMS_level_calc.integration_count - 1);
            samples += channel_count;

            assert(((position >= 0) && (position < sr->m_RMS_level_calc.integration_count))
                   && "error, RMS level sample position is out of range");

          }
          float_level =
            static_cast<Stream_float>(sqrt(sr->m_RMS_level_calc.square_sum[channel]
                                           / static_cast<double>(sr->m_RMS_level_calc.integration_count)));
          memcpy(&convert_level,
                 &float_level,
                 sizeof(Stream_conv_int));  // copy the current float level to an int
          // get current level for channel (as int) atomically
          cur_level = __sync_fetch_and_add(&(sr->m_level_RMS[channel]), 0x0);
          // xor old and new levels
          convert_level = convert_level ^ cur_level;
          // xor old level with "update xor" atomically
          __sync_xor_and_fetch(&(sr->m_level_RMS[channel]), convert_level);
        }
        // pick up where we left off with the running average in the next callback
        sr->m_RMS_level_calc.square_cache_position = position;
      }
      if (calc_levels_peak)
      {
        Stream_float peak_level;
        bool reset_peaks = __sync_fetch_and_and(&(sr->m_reset_peaks), 0x0);
        // calculate peak levels
        for (size_t channel = 0; channel < channel_count; ++channel)
        {
          // get current peak level for channel (as int) atomically
          cur_level = __sync_fetch_and_add(&(sr->m_level_peak[channel]), 0x0);
          if (reset_peaks)
          {
            peak_level = 0.0; // start with comparing new samples to 0
          }
          else
          {
            // convert current peak level to a float and use to compare to new samples
            memcpy(&peak_level,
                   &cur_level,
                   sizeof(Stream_float));
          }
          const Stream_float* samples = reinterpret_cast<const Stream_float*>(buffer) + channel;
          for (unsigned long i = 0; i < frames_per_buffer; ++i)
          {
            cur_sample = fabsf(*samples);
            if (cur_sample > peak_level)
            {
              peak_level = cur_sample;
            }
          }
          // copy the current peak level to an int
          memcpy(&convert_level,
                 &peak_level,
                 sizeof(Stream_conv_int));
          // xor old and new levels
          convert_level = convert_level ^ cur_level;
          // xor old level with "update xor" atomically
          __sync_xor_and_fetch(&(sr->m_level_peak[channel]), convert_level);
        }
      }
    }

    // get monitoring status atomically
    bool monitoring = __sync_fetch_and_add(&sr->m_monitoring, 0x0);
    if (!monitoring)
    {
      // recording, write buffer to disk
      bool have_overflow = false;
      if (!sr->m_AAFW->write(static_cast<int>(frames_per_buffer), buffer, have_overflow))
      {
        if (have_overflow)
        {
          __sync_add_and_fetch(&sr->m_output_overflow_count, 0x1);
        }
        else
        {
          if (!__sync_fetch_and_or(const_cast<int32_t*>(&sr->m_have_callback_error), 0x1))
          {
            sr->m_callback_record_error_code = Record_error::record_write;

            return_value = paAbort;
            goto exit_point;
          }
        }
      }
    }
  }

  return_value = paContinue;

 exit_point:
  return return_value;
}

//
// Constructor/destructor
//

Sound_recorder::Sound_recorder()
  : m_monitoring(1),
    m_muted(0),
    m_calc_levels_RMS(0),
    m_calc_levels_peak(0),
    m_metering_channels(0),
    m_reset_peaks(0),
    m_input_overflow_count(0),
    m_output_overflow_count(0),
    m_have_callback_error(0),
    m_level_RMS(0),
    m_level_peak(0),
    m_sound_initialized(false),
    m_config(new Sound_recorder_config),
    m_overflow_logging_enabled(false),
    m_AAFW(0),
    m_PA_stream(0),
    m_callback_record_error_code(Record_error::none),
    m_have_recording(false),
    m_muted_buffer(0),
    m_frames_per_second(0),
    m_frames_per_minute(0),
    m_frames_per_hour(0),
    m_RMS_level_calc(RMS_level_calc_struct())
{
  ih_setup_time_digits_cache();

  postcondition(mf_invariant(false));
}

Sound_recorder::RMS_level_calc_struct::RMS_level_calc_struct()
  : integration_period(0),
    integration_count(0),
    square_sum(0),
    square_cache(0),
    square_cache_position(0)
{
}

Sound_recorder::~Sound_recorder()
{
  precondition(mf_invariant(false));

  if (m_sound_initialized)
  {
    Error_param e(false);
    shutdown(e); // ignore error
  }

  delete [] m_RMS_level_calc.square_sum;
  delete [] m_RMS_level_calc.square_cache;
  delete [] m_muted_buffer;
  delete m_AAFW;
  delete m_config;
  free(m_level_peak);
  free(m_level_RMS);
}

//
// Control
//

//
//   ...startup/shutdown
//

bool Sound_recorder::startup(const Sound_recorder_config& config, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  bool PA_initialized = false;
  bool PA_stream_opened = false;
  bool AAFW_started = false;
  Sound_recorder_config new_config = config;
  Sound_recorder_config old_config = *m_config;

  start_error_block();

  if (m_sound_initialized)
  {
    return_value = true;
    goto exit_point;
  }
  else
  {
    m_monitoring = 1;
    m_muted = false;
    delete(m_AAFW);
    m_AAFW = 0;
    m_input_overflow_count = 0;
    m_output_overflow_count = 0;
    m_have_callback_error = 0;
    m_callback_record_error_code = Record_error::none;
    m_have_recording = false;
    free(m_level_peak);
    m_level_peak = 0;
    free(m_level_RMS);
    m_level_RMS = 0;
    delete [] m_muted_buffer;
    m_muted_buffer = 0;
    delete [] m_RMS_level_calc.square_sum;
    delete [] m_RMS_level_calc.square_cache;
    m_RMS_level_calc = RMS_level_calc_struct();

    PaError PA_error = Pa_Initialize();
    on_error(paNoError != PA_error, new Record_error(error_location(),
                                                     Record_error::startup,
                                                     new Portaudio_error(PA_error, "Pa_Intialize")));
    PA_initialized = true;

    // don't clip, since we won't be getting out of range samples
    const PaStreamFlags input_stream_flags = paClipOff; // Note: paClipOff declaration uses an old-style cast

    PaStreamParameters PA_input_parameters;
    if (Sound_recorder_config::default_input_device == new_config.input_device)
    {
      PA_input_parameters.device = Pa_GetDefaultInputDevice();
      new_config.input_device = PA_input_parameters.device;
    }
    else
    {
      PA_input_parameters.device = new_config.input_device;
    }

    const PaDeviceInfo* input_device_info = Pa_GetDeviceInfo(PA_input_parameters.device);
    if (!input_device_info)
    {
      stringstream diagnostic;

      const PaDeviceIndex device_count = Pa_GetDeviceCount();
      if (device_count == 0)
      {
        diagnostic << "no sound devices detected, check your configuration";
      }
      else if (device_count > 0)
      {
        diagnostic << "invalid device specified: " << PA_input_parameters.device << " (valid range is 0 to ";
        diagnostic << device_count - 1 << ")";
      }
      else
      {
        diagnostic << "error getting information for device: " << PA_input_parameters.device;
      }
      on_error(true, new Record_error(error_location(),
                                      Record_error::startup,
                                      diagnostic.str()));
    }

    if (Sound_file_config::default_channels == new_config.sound_file->channels)
    {
      PA_input_parameters.channelCount = input_device_info->maxInputChannels;
      new_config.sound_file->channels = PA_input_parameters.channelCount;
    }
    else
    {
      PA_input_parameters.channelCount = new_config.sound_file->channels;
    }

    // alignment parameter must be a power of two and a multiple of sizeof(void*)
    // Standards Rule #163 deviation
    const size_t alignment = sizeof(Stream_conv_int) * 2;
    assert((((alignment - 1) & alignment) == 0)
           && "Error alignment for posix_memalign is not a power of 2");
    assert((alignment % sizeof(void*)) == 0
           && "Error alignment for posix_memalign is not a multiple of sizeof(void*)");
    {
      size_t number_channels = static_cast<size_t>(new_config.sound_file->channels);
      // Standards Rule #163 deviation
      int posix_error = posix_memalign(reinterpret_cast<void**>(&m_level_RMS), // (check_code_ignore)
                                       alignment,
                                       number_channels * sizeof(Stream_conv_int));
      assert((posix_error != EINVAL) && "Error, invalid alignment for posix_memalign");
      on_error(posix_error, new Record_error(error_location(), Record_error::startup,
                                             new Posix_error(posix_error,
                                                             "posix_memalign",
                                                             0)));
      // Standards Rule #163 deviation
      posix_error = posix_memalign(reinterpret_cast<void**>(&m_level_peak), // (check_code_ignore)
                                   alignment,
                                   number_channels * sizeof(Stream_conv_int));
      assert((posix_error != EINVAL) && "Error, invalid alignment for posix_memalign");
      on_error(posix_error, new Record_error(error_location(), Record_error::startup,
                                             new Posix_error(posix_error,
                                                             "posix_memalign",
                                                             0)));
    }
    // zero out levels
    const Stream_float zero = 0.0;
    for (int channel = 0; channel < new_config.sound_file->channels; ++ channel)
    {
      // RMS
      memcpy(&m_level_RMS[channel],
             &zero,
             sizeof(Stream_conv_int));
      // peak
      memcpy(&m_level_peak[channel],
             &zero,
             sizeof(Stream_conv_int));
    }

    // always use 32 bit floating point for read from input device
    PA_input_parameters.sampleFormat = paFloat32; // Note: paFloat32 declaration uses an old-style cast
    PA_input_parameters.suggestedLatency = input_device_info->defaultLowInputLatency;
    PA_input_parameters.hostApiSpecificStreamInfo = 0;

    PaStream* PA_stream;
    PA_error = Pa_OpenStream(&PA_stream,
                             &PA_input_parameters,
                             0, // only read from PortAudio subsystem
                             new_config.sound_file->sample_rate,
                             static_cast<long unsigned int>(new_config.frames_per_buffer),
                             input_stream_flags,
                             Sound_recorder::mf_audio_callback,
                             this);
    on_error(paNoError != PA_error, new Record_error(error_location(),
                                                     Record_error::open_input_stream,
                                                     new Portaudio_error(PA_error, "Pa_OpenStream")));

    PA_stream_opened = true;
    m_PA_stream = PA_stream;

    Pa_SetStreamFinishedCallback(m_PA_stream, mf_stream_finished_callback);

    // assign the sound recorder config before starting the stream
    // since it is needed by the audio callback
    *m_config = new_config;

    m_overflow_logging_enabled =false;

    m_AAFW = new Async_audio_file_writer(m_overflow_logging_enabled);

    // start the AAFW before starting the stream, because the audio callback needs it
    Error_param error_AAFW;
    AAFW_started = m_AAFW->start(new_config, error_AAFW);
    on_error(!AAFW_started, new Record_error(error_location(),
                                             Record_error::open_input_stream,
                                             error_AAFW()));

    on_error(new_config.sound_file->channels <= 0,
             new Record_error(error_location(),
                              Record_error::startup));

    size_t number_channels = static_cast<size_t>(m_config->sound_file->channels);

    size_t frames_per_buffer = static_cast<size_t>(m_config->frames_per_buffer);
    // Standards Rule #163 deviation
    size_t muted_buffer_size = sizeof(Stream_float) * number_channels * frames_per_buffer;

    m_muted_buffer = new char[muted_buffer_size];
    memset(m_muted_buffer,
           0,
           muted_buffer_size);

    PA_error = Pa_StartStream(m_PA_stream);
    on_error(paNoError != PA_error, new Record_error(error_location(),
                                                     Record_error::open_input_stream,
                                                     new Portaudio_error(PA_error, "Pa_StartStream")));

    m_sound_initialized = true;

    m_frames_per_second = static_cast<Long_int>(m_config->sound_file->sample_rate);
    m_frames_per_minute = m_frames_per_second * 60;
    m_frames_per_hour = m_frames_per_minute * 60;

    m_RMS_level_calc.integration_period = m_config->RMS_level_integration_period;
    int integration_count = m_config->sound_file->sample_rate * (m_RMS_level_calc.integration_period / 1000.0);
    if (integration_count < m_config->frames_per_buffer)
    {
      // the average should have a count at least equal to frames per buffer
      integration_count = m_config->frames_per_buffer;
    }
    // the count should be a power of 2 for averaging algorithm
    if (!is_pow2(integration_count))
    {
      integration_count = round_up_to_pow2(integration_count);
    }
    assert(is_pow2(integration_count)
           && "error, RMS level integration count is not a power of 2");
    m_RMS_level_calc.square_sum = new double[number_channels];
    m_RMS_level_calc.square_cache = new double[integration_count * m_config->sound_file->channels];
    m_RMS_level_calc.integration_count = integration_count;
    // zero out square sum and square cache arrays
    for (size_t channel = 0; channel < number_channels; ++channel)
    {
      m_RMS_level_calc.square_sum[channel] = 0.0;
      for (int position = 0; position < integration_count; ++position)
      {
        m_RMS_level_calc.square_cache[integration_count * channel + position] = 0.0;
      }
    }
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (PA_stream_opened)
  {
    if (!Pa_IsStreamStopped(m_PA_stream))
    {
      Pa_AbortStream(m_PA_stream); // ignore return code
    }
    Pa_CloseStream(m_PA_stream); // ignore return code
    m_PA_stream = 0;
  }
  // shutdown the AAFW after shutting down the
  // Portaudio stream, since the audio callback needs
  // the AAFW
  if (AAFW_started)
  {
    Error_param e(false);
    m_AAFW->stop(e); // ignore error
  }
  if (m_AAFW)
  {
    delete m_AAFW;
    m_AAFW = 0;
  }
  if (PA_initialized)
  {
    Pa_Terminate(); // ignore return code
  }
  *m_config = old_config;

  // delete this after Portaudio stream shutdown
  // because the audio callback needs them
  delete [] m_muted_buffer;
  m_muted_buffer = 0;
  free(m_level_peak); // allocated with posix_memalign, so use free()
  m_level_peak = 0;
  free(m_level_RMS); // allocated with posix_memalign, so use free()
  m_level_RMS = 0;

  m_frames_per_second = 0;
  m_frames_per_minute = 0;
  m_frames_per_hour = 0;

  // delete these after Portaudio stream shutdown
  // because the audio callback needs them
  delete m_RMS_level_calc.square_sum;
  delete m_RMS_level_calc.square_cache;
  m_RMS_level_calc = RMS_level_calc_struct();

  m_sound_initialized = false;
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(( // function success
                 (return_value &&
                  ((m_PA_stream && Pa_IsStreamActive(m_PA_stream))
                   && (m_AAFW && m_AAFW->running())
                   && (*m_config == new_config)
                   && m_level_RMS
                   && m_level_peak
                   && m_muted_buffer
                   && m_sound_initialized))
                 // function failure
                 || (!return_value &&
                     (!m_PA_stream
                      && !m_AAFW
                      && (*m_config == old_config)
                      && !m_level_RMS
                      && !m_level_peak
                      && !m_muted_buffer
                      && !m_sound_initialized)))
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Sound_recorder::shutdown(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  m_frames_per_second = 0;
  m_frames_per_minute = 0;
  m_frames_per_hour = 0;

  if (!m_sound_initialized)
  {
    return_value = true;
    goto exit_point;
  }

  if (Recorder::state_stopped != state())
  {
    Error_param error;
    if (!stop(error))
    {
      on_error(true, new Record_error(error_location(),
                                      Record_error::shutdown,
                                      error()));
    }
  }

  if (m_PA_stream)
  {
    PaError PA_error_close = paNoError;
    const PaError PA_error_abort = Pa_AbortStream(m_PA_stream);
    if (paNoError != PA_error_abort)
    {
      PA_error_close = Pa_CloseStream(m_PA_stream);
    }
    m_PA_stream = 0;

    on_error(paNoError != PA_error_abort, new Record_error(error_location(),
                                                           Record_error::shutdown,
                                                           new Portaudio_error(PA_error_abort,
                                                                               "Pa_AbortStream")));

    on_error(paNoError != PA_error_close, new Record_error(error_location(),
                                                           Record_error::shutdown,
                                                           new Portaudio_error(PA_error_close,
                                                                               "Pa_CloseStream")));
  }

  // shutdown the AAFW after shutdown of Portaudio stream
  // because the audio call back needs it
  if (m_AAFW)
  {
    Error_param error_AAFW;
    if (m_AAFW->running())
    {
      const bool AAFW_stopped = m_AAFW->stop(error_AAFW);
      on_error(!AAFW_stopped, new Record_error(error_location(),
                                               Record_error::shutdown,
                                               error_AAFW()));
    }

    delete m_AAFW;
    m_AAFW = 0;
  }

  const PaError PA_error = Pa_Terminate();
  on_error(paNoError != PA_error, new Record_error(error_location(),
                                                   Record_error::shutdown,
                                                   new Portaudio_error(PA_error, "Pa_Terminate")));

  m_sound_initialized = false;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  // currently, we don't need to check for recorder stopped state,
  // because we tried at least once already if we reach this point
  if (m_PA_stream)
  {
    if (Pa_AbortStream(m_PA_stream) == paNoError)
    {
      Pa_CloseStream(m_PA_stream); // ignore error
    }
    m_PA_stream = 0;
  }
  if (m_AAFW)
  {
    if (m_AAFW->running())
    {
      Error_param e(false);
      m_AAFW->stop(e); // ignore error
    }
    delete m_AAFW;
    m_AAFW = 0;
  }

  m_sound_initialized = false;
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition((Recorder::state_stopped == state())
                && !m_PA_stream
                && !m_AAFW
                && !m_sound_initialized
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
//   ...basic Recorder functions
//

bool Sound_recorder::record(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  const bool have_error = !can_record();
  on_error(have_error, new Record_error(error_location(),
                                        Record_error::record,
                                        string("Error, recorder cannot start recording")));

  Recorder::record();

  __sync_and_and_fetch(&m_monitoring, 0x0);
  m_have_recording = true;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(( // function success
                 (return_value
                  && !is_monitoring()
                  && m_have_recording)
                 //function failure
                 || !return_value)
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Sound_recorder::pause(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  const bool have_error = (!m_sound_initialized
                           || (Recorder::state_recording != state())
                           || !m_AAFW
                           || !m_PA_stream);
  on_error(have_error, new Record_error(error_location(),
                                        Record_error::pause,
                                        string("Error, recorder cannot pause")));

  Error_param error;
  const bool command_completed = m_AAFW->pause(error);
  on_error(!command_completed,
           new Record_error(error_location(),
                            Record_error::pause,
                            error()));

  Recorder::pause();

  __sync_or_and_fetch(&m_monitoring, 0x1);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(( // function success
                 (return_value
                  && is_monitoring())
                 // function failure
                 || !return_value)
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Sound_recorder::stop(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  const bool overwrite = m_config->file_overwrite;
  bool AAFW_restarted = false;
  bool stream_restarted = false;

  start_error_block();

  const bool have_error = (!m_sound_initialized
                           || (Recorder::state_stopped == state())
                           || !m_AAFW
                           || !m_PA_stream);
  on_error(have_error, new Record_error(error_location(),
                                        Record_error::stop,
                                        string("Error, recorder cannot stop")));

  __sync_fetch_and_or(&m_monitoring, 0x1);

  // stop audio stream
  //
  // Note: Pa_StopStream has an occasional significant
  // delay (seconds) before returning
  const PaError PA_error_stop = Pa_StopStream(m_PA_stream);

  // stop AAFW (try even if there was an error stopping the stream above)
  Error_param error;
  bool AAFW_stopped = m_AAFW->stop(error); // blocks until writer thread stopped

  on_error(paNoError != PA_error_stop, new Record_error(error_location(),
                                                        Record_error::stop,
                                                        new Portaudio_error(PA_error_stop, "Pa_StopStream")));
  on_error(!AAFW_stopped, new Record_error(error_location(), Record_error::stop, error()));

  // restart AAFW if overwrite allowed
  // (resets sound file to beginning to prepare for a new recording)
  if (overwrite)
  {
    AAFW_restarted = m_AAFW->start(*m_config, error);
    on_error(!AAFW_restarted, new Record_error(error_location(),
                                               Record_error::stop,
                                               error()));

    const PaError PA_error_start = Pa_StartStream(m_PA_stream);
    on_error(paNoError != PA_error_start, new Record_error(error_location(),
                                                           Record_error::stop,
                                                           new Portaudio_error(PA_error_start,
                                                                               "Pa_StartStream")));
    stream_restarted = true;
  }
  else
  {
    // no overwrite, make sure we can't restart by
    // closing stream
    const PaError PA_error_close = Pa_CloseStream(m_PA_stream);
    m_PA_stream = 0;
    on_error(paNoError != PA_error_close, new Record_error(error_location(),
                                                           Record_error::stop,
                                                           new Portaudio_error(PA_error_close,
                                                                               "Pa_CloseStream")));
    // zero out levels
    const Stream_float zero = 0.0;
    for (int channel = 0; channel < m_config->sound_file->channels; ++ channel)
    {
      // RMS
      memcpy(&m_level_RMS[channel],
             &zero,
             sizeof(Stream_conv_int));
      // peak
      memcpy(&m_level_peak[channel],
             &zero,
             sizeof(Stream_conv_int));
    }
  }

  Recorder::stop();

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  {
    if (AAFW_restarted)
    {
      Error_param e(false);
      m_AAFW->stop(e); // ignore error
    }
    if (stream_restarted)
    {
      if (paNoError == Pa_AbortStream(m_PA_stream))
      {
        Pa_CloseStream(m_PA_stream); // ignore error
      }
    }
    m_PA_stream = 0;

    if (Recorder::state_stopped != Recorder::state())
    {
      Recorder::stop();
    }

    return_value = false;
  }
  goto exit_point;

 exit_point:
  postcondition(( // function success
                 (return_value &&
                  ((m_config->file_overwrite &&
                    // overwrite
                    ((m_PA_stream && Pa_IsStreamActive(m_PA_stream))
                     && (m_AAFW && m_AAFW->running())))
                   || (!m_config->file_overwrite &&
                       // no overwrite
                       (!m_PA_stream
                        && (m_AAFW && !m_AAFW->running())))))
                 // function failure
                 || (!return_value
                     && (!m_PA_stream
                         && (m_AAFW && !m_AAFW->running()))))
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
//   ...sound recorder functions
//

// toggles mute feature
bool Sound_recorder::mute(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  on_error(!m_sound_initialized, new Record_error(error_location(),
                                                  Record_error::mute,
                                                  string("Error, recorder cannot mute")));

  // toggle muted flag
  __sync_xor_and_fetch(&m_muted, 0x1);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// Accessors
//

void Sound_recorder::sound_levels_RMS(vector<double>& return_levels) const
{
  precondition(mf_invariant());

  if (!m_level_RMS)
  {
    goto exit_point;
  }
  else
  {
    size_t number_channels = min(static_cast<size_t>(m_config->sound_file->channels),
                                 return_levels.size());
    if (is_RMS_metering_enabled())
    {
      Stream_conv_int cur_level;
      Stream_float conv_level;
      for (size_t channel = 0; channel < number_channels; ++channel)
      {
        // get level (as int) atomically
        cur_level = __sync_fetch_and_add(const_cast<Stream_conv_int*>(&m_level_RMS[channel]), 0x0);
        memcpy(&conv_level,
               &cur_level,
               sizeof(Stream_float)); // copy back to a float

        return_levels[channel] = conv_level;
      }
    }
    else
    {
      for (size_t channel = 0; channel < number_channels; ++channel)
      {
        return_levels[channel] = 0;
      }
    }
  }

 exit_point:
  return;
}

void Sound_recorder::sound_levels_peak(vector<double>& return_levels, bool reset_peaks) const
{
  precondition(mf_invariant());

  if (!m_level_peak)
  {
    goto exit_point;
  }
  else
  {
    size_t number_channels = min(static_cast<size_t>(m_config->sound_file->channels),
                                 return_levels.size());
    if (is_peak_metering_enabled())
    {
      Stream_conv_int cur_level;
      Stream_float conv_level;
      for (size_t channel = 0; channel < number_channels; ++channel)
      {
        // get (as int) and clear peak (atomically)
        cur_level = __sync_fetch_and_add(const_cast<Stream_conv_int*>(&m_level_peak[channel]), 0x0);
        memcpy(&conv_level,
               &cur_level,
               sizeof(Stream_float)); // copy back to a float
        return_levels[channel] = conv_level;
        if (reset_peaks)
        {
          __sync_fetch_and_or(&m_reset_peaks, 0x1);
        }
      }
    }
    else
    {
      for (size_t channel = 0; channel < number_channels; ++channel)
      {
        // clear peak (atomically)
        __sync_and_and_fetch(const_cast<Stream_conv_int*>(&m_level_peak[channel]), 0x0);
        return_levels[channel] = 0;
      }
    }
  }

 exit_point:
  return;
}

Long_int Sound_recorder::frames_written_count() const
{
  precondition(mf_invariant());

  const Long_int return_value = (m_AAFW ? m_AAFW->frames_written_count() : 0);

  return return_value;
}

bool Sound_recorder::is_started() const
{
  precondition(mf_invariant());

  return (m_AAFW && m_AAFW->running());
}

bool Sound_recorder::can_record() const
{
  precondition(mf_invariant());

  return (m_sound_initialized
          && (Recorder::state_recording != state())
          && m_AAFW
          && m_PA_stream);
}

void Sound_recorder::frames_to_recorded_time(Long_int frames,
                                             char separator,
                                             string& return_recorded_time) const
{
  precondition((frames >= 0)
               && mf_invariant());

  const Long_int seconds = (frames / m_frames_per_second) % 60;
  const Long_int minutes = (frames / m_frames_per_minute) % 60;
  Long_int hours = frames / m_frames_per_hour;
  if (hours > 99)
  {
    hours = 0;
  }
  assert((seconds < 60)
         && (minutes < 60)
         && (hours < 100)
         && "error, invalid HMS");

  stringstream recorded_time;
  recorded_time << iv_time_digits_cache[hours] << separator;
  recorded_time << iv_time_digits_cache[minutes] << separator;
  recorded_time << iv_time_digits_cache[seconds];

  return_recorded_time = recorded_time.str();

  postcondition((return_recorded_time.length() == string("HH:MM:SS").length()));
}

void Sound_recorder::recorded_time(string& return_recorded_time, char separator) const
{
  precondition(mf_invariant());

  const Long_int frames_recorded = frames_written_count();

  frames_to_recorded_time(frames_recorded, separator, return_recorded_time);
}

const Error* Sound_recorder::record_error() const
{
  precondition(mf_invariant());

  Error *return_value = 0;

  // get error status atomically
  const bool have_record_error = __sync_fetch_and_add(const_cast<int32_t*>(&m_have_callback_error), 0x0);
  if (!have_record_error)
  {
    goto exit_point;
  }
  else
  {
    Error *base_error = 0;

    // if the error is related to the AAFW, get the base error
    if (Record_error::record_write == m_callback_record_error_code)
    {
      base_error = m_AAFW->write_error();
    }

    return_value = new Record_error(__FILE__,
                                    "Sound_recorder::mf_audio_callback",
                                    0,
                                    m_callback_record_error_code,
                                    base_error);
  }

 exit_point:
  return return_value;
}

string Sound_recorder::output_file() const
{
  precondition(mf_invariant());

  const string blank("");
  const string *return_value_ptr = &blank;
  string temporary_output_file;

  if (m_config->sound_file->file_name != "")
  {
    return_value_ptr = &m_config->sound_file->file_name;
    goto exit_point;
  }
  else if (!m_AAFW)
  {
    goto exit_point;
  }
  else
  {
    if (m_AAFW->temporary_output_file(temporary_output_file))
    {
      return_value_ptr = &temporary_output_file;
      goto exit_point;
    }
  }

 exit_point:
  return *return_value_ptr;
}

// returns false if available space/time is less than thresholds
// return_bytes_available, return_fraction_available, and return_seconds_remaining are set
// to 0 if the directory isn't found
bool Sound_recorder::space_available(Long_int& return_bytes_available,
                                     double& return_fraction_available,
                                     double& return_seconds_remaining) const
{
  precondition(mf_invariant());

  bool return_value = false;

  double fraction_available = 0.0;
  Long_int available_bytes = 0;
  double seconds_remaining = 0.0;
  const string directory = File_manager::parent_directory_for_path(output_file());
  Error_param e(false);
  if (!File_manager::file_system_available_for_path(directory,
                                                    fraction_available,
                                                    available_bytes,
                                                    e))
  {
    return_bytes_available = 0;
    return_fraction_available = 0.0;
    return_seconds_remaining = 0.0;
    return_value = true;
    goto exit_point;
  }
  else
  {
    const double rate = m_config->sound_file->sample_rate;
    const double channels = m_config->sound_file->channels;
    size_t bytes_per_sample;
    if (!Sound_file_config::sample_size_for_data_format(m_config->sound_file->data_format, bytes_per_sample))
    {
      assert(false && "invalid data format");
    }
    // TODO: the VBR estimate could be better...
    double VBR_quality;
    if (m_config->sound_file->has_VBR(VBR_quality))
    {
      bytes_per_sample = 1; // variable, guess-timate
    }
    if (0 == bytes_per_sample)
    {
      bytes_per_sample = 2; // variable, guess-timate
    }

    seconds_remaining = (static_cast<double>(available_bytes)
                         / (rate * static_cast<double>(bytes_per_sample) * channels))
      * ic_remaining_adjustment;

    if (m_config->sound_file->is_compressed())
    {
      seconds_remaining *= 2.0; // assuming 50% average compression
    }

    return_bytes_available = available_bytes;
    return_fraction_available = fraction_available;
    return_seconds_remaining = seconds_remaining;

    return_value = ((fraction_available > ic_fraction_available_warning_threshold)
                    && (seconds_remaining > ic_seconds_remaining_warning_threshold));
  }

 exit_point:
  return return_value;
}

// currently always returns true
// return_file_size and return_data_rate are set to 0 if there is no output file
bool Sound_recorder::output_file_size(Long_int& return_file_size,
                                      double& return_data_rate) const
{
  precondition(mf_invariant());

  Long_int file_size = 0;
  double data_rate = 0.0;

  string file_path = output_file();
  if ("" == file_path)
  {
    return_file_size = 0;
    return_data_rate = 0.0;

    goto exit_point;
  }
  else
  {
    Error_param e(false);
    if (!File_manager::file_size_for_path(file_path,
                                          file_size,
                                          e))
    {
      goto exit_point;
    }

    const double rate = m_config->sound_file->sample_rate;
    const double channels = m_config->sound_file->channels;
    size_t bytes_per_sample;
    if (!Sound_file_config::sample_size_for_data_format(m_config->sound_file->data_format, bytes_per_sample))
    {
      assert(false && "invalid data format");
    }
    // TODO: the VBR estimate could be better...
    double VBR_quality;
    if (m_config->sound_file->has_VBR(VBR_quality))
    {
      bytes_per_sample = 1; // variable, guess-timate
    }
    if (0 == bytes_per_sample)
    {
      bytes_per_sample = 2; // variable, guess-timate
    }

    data_rate = rate * channels * static_cast<double>(bytes_per_sample);

    if (m_config->sound_file->is_compressed())
    {
      data_rate /= 2;
    }

    return_file_size = file_size;
    return_data_rate = data_rate;
  }

 exit_point:
  return true;
}

//
// Mutators
//
void Sound_recorder::enable_RMS_metering(bool enable)
{
  precondition(mf_invariant());

  if (enable)
  {
    if (!is_RMS_metering_enabled() && m_level_RMS)
    {
      // if not already enabled, clear levels and enable
      const Stream_float zero = 0.0;
      for (int channel = 0; channel < config()->sound_file->channels; ++ channel)
      {
        memcpy(&m_level_RMS[channel],
               &zero,
               sizeof(Stream_conv_int));
      }
      __sync_or_and_fetch(&m_calc_levels_RMS, 0x1);
    }
  }
  else
  {
    __sync_and_and_fetch(&m_calc_levels_RMS, 0x0);
  }

  postcondition(mf_invariant());
}

void Sound_recorder::enable_peak_metering(bool enable)
{
  precondition(mf_invariant());

  if (enable)
  {
    if (!is_peak_metering_enabled() && m_level_peak)
    {
      // if not already enabled, clear levels and enable
      const Stream_float zero = 0.0;
      for (int channel = 0; channel < config()->sound_file->channels; ++ channel)
      {
        memcpy(&m_level_peak[channel],
               &zero,
               sizeof(Stream_conv_int));
      }
      __sync_or_and_fetch(&m_calc_levels_peak, 0x1);
    }
  }
  else
  {
    __sync_and_and_fetch(&m_calc_levels_peak, 0x0);
  }

  postcondition(mf_invariant());
}

void Sound_recorder::set_metering_channels(int channel_count)
{
  precondition(((channel_count >= 0)
                && (channel_count < integer_traits<int32_t>::max()))
               && mf_invariant());

  const int32_t new_channel_count = channel_count;
  const int32_t old_channel_count = __sync_fetch_and_add(&m_metering_channels, 0x0);
  // xor old and new channel count
  const int32_t update_channel_count = new_channel_count ^ old_channel_count;
  // xor old level with "update xor" atomically
  __sync_xor_and_fetch(&m_metering_channels, update_channel_count);

  postcondition((metering_channels() == channel_count)
                && mf_invariant());
}

void Sound_recorder::clear_record_error()
{
  precondition((Recorder::state_stopped == state())
               && mf_invariant());

  __sync_fetch_and_and(const_cast<int32_t*>(&m_have_callback_error), 0x0);
  m_callback_record_error_code = Record_error::none;

  precondition((Recorder::state_stopped == state())
               && mf_invariant());
}

//
// Misc
//

// removes 0 (audio frame) length temporary files,
// which are created during startup and before the record command
// to improve record startup time
//
// returns true if one is removed
bool Sound_recorder::cleanup_temporary_file()
{
  precondition(mf_invariant());

  bool return_value = false;
  string temporary_file = "";

  if (m_sound_initialized && !m_have_recording)
  {
    if (m_AAFW && (0 == m_AAFW->frames_written_count()))
    { // paranoia
      if (m_AAFW->temporary_output_file(temporary_file))
      {
        if ("" != temporary_file)
        { // paranoia
          return_value = !unlink(temporary_file.c_str());
          goto exit_point;
        }
      }
    }
  }

 exit_point:
  postcondition((("" == temporary_file) || !File_manager::path_exists(temporary_file))
                && mf_invariant());
  return return_value;
}

//
// Protected member functions
//

// invariant check
bool Sound_recorder::mf_invariant(bool check_base_class) const
{
  bool return_value = false;
  {
    // member objects allocated check
    if (!m_config)
    {
      goto exit_point;
    }

    // range checking
    if ((input_overflow_count() < 0)
        || (output_overflow_count() < 0)
        || (metering_channels() < 0))
    {
      goto exit_point;
    }
    if (m_RMS_level_calc.integration_period < 0)
    {
      goto exit_point;
    }
    if ((m_sound_initialized
         && (m_frames_per_second < 1)
         && (m_frames_per_minute < 1)
         && (m_frames_per_hour < 1)
         && (m_RMS_level_calc.integration_period < 1)
         && (m_RMS_level_calc.integration_count < 1))
        || (!m_sound_initialized
            && (m_frames_per_second != 0)
            && (m_frames_per_minute != 0)
            && (m_frames_per_hour != 0)
            && (m_RMS_level_calc.integration_period != 0)
            && (m_RMS_level_calc.integration_count != 0)
            && (m_RMS_level_calc.square_cache_position != 0)))
    {
      goto exit_point;
    }

    // object consistency checking
    const bool is_monitoring = Sound_recorder::is_monitoring();
    const int state = Recorder::state();
    if (((Recorder::state_recording == state) && is_monitoring)
        || (((Recorder::state_paused == state) || (Recorder::state_stopped == state))
            && !is_monitoring))
    {
      goto exit_point;
    }
    else if (((Recorder::state_recording == state) || (Recorder::state_paused == state))
             && (!m_sound_initialized
                 || !m_level_RMS
                 || !m_level_peak
                 || !m_AAFW
                 || !(m_AAFW->running()
                      // AAFW allowed to not be running if there was an error
                      || m_AAFW->have_write_error())
                 || !m_PA_stream
                 || Pa_IsStreamStopped(m_PA_stream)
                 || !m_have_recording
                 || !m_muted_buffer
                 || !m_RMS_level_calc.square_sum
                 || !m_RMS_level_calc.square_cache))
    {
      goto exit_point;
    }
    // no final else needed, we've looked at all the states

    const bool have_record_error = __sync_fetch_and_add(const_cast<int32_t*>(&m_have_callback_error), 0x0);
    if (have_record_error && !m_have_recording)
    {
      goto exit_point;
    }

    return_value = (!check_base_class || Recorder::mf_invariant(check_base_class));

  }

 exit_point:
  return return_value;
}
