// Sound_file_config.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef SOUND_FILE_CONFIG_HPP_
#define SOUND_FILE_CONFIG_HPP_

#include <map>
#include <string>

// helper struct to store sound file settings

namespace Roan_trail
{
  namespace Recorder
  {
    struct Sound_file_config
    {
    public:
      // constructor/destructor/copy
      Sound_file_config();
      // typedef
      typedef int File_type;
      typedef int Data_format;
      typedef int Endianness;
      typedef std::map<int, std::string> Constant_map;
      // structure data members
      int channels;
      double sample_rate;
      double VBR_quality; // < 0.0 default, range >= 0.0 (lowest) to <= 1.0 (highest)
      std::string file_name;
      File_type file_type;
      Data_format data_format;
      Endianness endianness;
      // operators
      bool operator==(const Sound_file_config& config) const;
      bool operator!=(const Sound_file_config& config) const;
      // helpers
      //   file type
      static std::string string_for_file_type(File_type file);
      static bool file_type_for_string(const std::string& file_type_string, File_type& return_file);
      //   data format
      static std::string string_for_data_format(Data_format format);
      static bool data_format_for_string(const std::string& data_format_string, Data_format& return_data_format);
      static bool sample_size_for_data_format(Data_format format, size_t& return_size);
      static bool is_supported_data_format(Data_format format);
      //   endianness
      static std::string string_for_endianness(Endianness endian);
      static bool endianness_for_string(const std::string& endianness_string, Endianness& return_endian);
      static Endianness endianness_for_CPU();
      static bool endianness_for_file_type(File_type file, Endianness& return_endian);
      //   other
      std::string format_string(const std::string& prefix) const;
      std::string format_string_brief() const;
      bool has_VBR(double& return_VBR) const;
      bool is_compressed() const;
      bool from_map(const std::map<std::string, std::string>& config_map);
      bool to_map(std::map<std::string, std::string>& return_config_map) const;
      // struct constants
      //   default constants
      static const int default_channels;
      static const double default_sample_rate;
      static const double default_VBR_quality;
      static const File_type default_file_type;
      static const Data_format default_data_format;
      static const Endianness default_endianness;
      //   miscellaneous constants
      static const double highest_VBR_quality;
      //   file type constants
      static const File_type file_type_unknown = -1;
      static const File_type file_type_wav   = 0;
      static const File_type file_type_wavex = 1;
      static const File_type file_type_aiff  = 2;
      static const File_type file_type_au    = 3;
      static const File_type file_type_flac  = 4;
      static const File_type file_type_ogg   = 5;
      //   data format constants
      static const Data_format data_format_PCMS8  = 0;
      static const Data_format data_format_PCM16  = 1;
      static const Data_format data_format_PCM24  = 2;
      static const Data_format data_format_PCM32  = 3;
      static const Data_format data_format_float  = 4;
      static const Data_format data_format_double = 5;
      static const Data_format data_format_vorbis = 6;
      //   endian constants
      static const Endianness endian_file   = 0;
      static const Endianness endian_little = 1;
      static const Endianness endian_big    = 2;
      static const Endianness endian_CPU    = 3;
    private:
      // maps
      static Constant_map m_file_type_map;
      static Constant_map m_data_format_map;
      static std::map<Data_format, size_t> m_data_format_bytes_map;
      static Constant_map m_endianness_map;
      static std::map<File_type, Endianness> m_file_type_endianness_map;
      //
      static void mf_setup_constant_maps();
    };
  }
}

#endif // SOUND_FILE_CONFIG_HPP_
