// Slide_collection.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// Abstract base, slide collection class

#ifndef SLIDE_COLLECTION_HPP_
#define SLIDE_COLLECTION_HPP_

#include "Slide.hpp"
#include "common.hpp"
#include "Image_types.hpp"
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <libxml++/exceptions/exception.h>
#include <libxml++/nodes/node.h>
#include <libxml2/libxml/tree.h>
#include <string>
#include <vector>
#include <sstream>
#include "error/Source_error.hpp"
#include "error/Error.hpp"

namespace Roan_trail
{
  namespace Source
  {
    template <class Image> class Slide_collection
    {
      BOOST_CONCEPT_ASSERT((boost::Assignable<Image>));
      BOOST_CONCEPT_ASSERT((boost::CopyConstructible<Image>));
    public:
      virtual ~Slide_collection();
      // accessors
      const std::string& source() const { return m_source; }
      virtual std::string source_type() const = 0;
      int current_slide_index() const { return m_current_slide_index; }
      virtual int slide_count() const { return static_cast<int>(m_slides.size()); }
      virtual bool image_for_slide_at(int slide_index, Image& return_image) const = 0;
      virtual void thumbnail_for_slide_at(int slide_index, Image& return_image) const = 0;
      virtual Rect_size max_bounds() const = 0;
      // operators
      Slide<Image>* operator[](size_t index) { return m_slides[index]; }
      const Slide<Image>* operator[](size_t index) const { return m_slides[index]; }
      // mutators
      void set_current_slide_index(int index) { m_current_slide_index = index; }
      void next_slide();
      void previous_slide();
      // load/save
      //   using source
      virtual bool load_from_source(const std::string& path, Error_param& return_error) = 0;
      //   import notes from XML
      virtual bool import_notes_from_XML(const std::string& file_path,
                                         const xmlpp::Node* XML_node,
                                         Error_param& return_error);
      //   load whole collection from XML
      virtual bool load_from_XML(const std::string& file_path,
                                 const xmlpp::Node* XML_node,
                                 const std::string& custom_source_path,
                                 Error_param& return_error) = 0;
      //   save whole collection to XML
      virtual bool save_as_XML(const std::string& file_path,
                               xmlpp::Node* XML_node,
                               Error_param& return_error) = 0;
    protected:
      Slide_collection();
      //
      std::vector<Slide<Image>*> m_slides;
      bool m_source_loaded;
      std::string m_source;
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      int m_current_slide_index;
      // prevent compiler from generating
      Slide_collection(const Slide_collection& collection);
      Slide_collection& operator=(const Slide_collection& collection);
    };

    //
    // Template implementation
    //

    namespace
    {
      std::string ih_format_XML_file_line(int line)
      {
        std::stringstream line_stream;
        line_stream << ":" << line;

        return line_stream.str();
      }
    }

    template <class Image> Slide_collection<Image>::~Slide_collection()
    {
      precondition(mf_invariant(false));

      for (size_t i = 0; i < m_slides.size(); ++i)
      {
        delete m_slides[i];
      }
    }

    //
    // Mutators
    //

    template <class Image> void Slide_collection<Image>::next_slide()
    {
      precondition(m_source_loaded
                   && mf_invariant());

      ++m_current_slide_index;
      if (m_current_slide_index >= slide_count())
      {
        // wrap
        m_current_slide_index = 0;
      }

      postcondition(mf_invariant());
    }

    template <class Image> void Slide_collection<Image>::previous_slide()
    {
      precondition(m_source_loaded
                   && mf_invariant());

      if (m_current_slide_index > 0)
      {
        --m_current_slide_index;
      }
      else
      {
        // wrap back
        m_current_slide_index = slide_count() - 1;
      }

      postcondition(mf_invariant());
    }

    //
    // Load save
    //

    template <class Image> bool Slide_collection<Image>::import_notes_from_XML(const std::string& file_path,
                                                                               const xmlpp::Node* XML_node,
                                                                               Error_param& return_error)
    {
      precondition(XML_node
                   && !return_error());

      bool return_value = false;
      start_error_block();

      Error_param error;

      try
      {
        const std::vector<xmlpp::Node*> slide_nodes = XML_node->find("//slide");
        for (std::vector<xmlpp::Node*>::const_iterator current_slide_node = slide_nodes.begin();
             current_slide_node != slide_nodes.end();
             ++current_slide_node)
        {
          const std::vector<xmlpp::Node*> slide_index_nodes = (*current_slide_node)->find("index");
          on_error(!slide_index_nodes.size(),
                   new Source_error(error_location(),
                                    Source_error::load,
                                    std::string("error importing XML file, node: ")
                                    + (*current_slide_node)->get_path() + std::string(" index not found"),
                                    file_path
                                    + ih_format_XML_file_line((*current_slide_node)->get_line())));
          const std::vector<xmlpp::Node*> slide_index_text_nodes = slide_index_nodes[0]->find("text()");
          on_error(!slide_index_text_nodes.size(),
                   new Source_error(error_location(),
                                    Source_error::load,
                                    std::string("error importing XML file, node: ")
                                    + slide_index_nodes[0]->get_path() + std::string(" text node not found"),
                                    file_path + ih_format_XML_file_line(slide_index_nodes[0]->get_line())));
          // the reinterpret cast is used to convert unsigned char to char
          // unfortunately, there seems to be no obvious way around this
          const char *slide_index_text_content =
            reinterpret_cast<const char *>(slide_index_text_nodes[0]->cobj()->content);
          on_error(!slide_index_text_content,
                   new Source_error(error_location(),
                                    Source_error::load,
                                    std::string("error importing XML file, node: ")
                                    + slide_index_text_nodes[0]->get_path()
                                    + std::string(" index text content not found"),
                                    file_path
                                    + ih_format_XML_file_line(slide_index_text_nodes[0]->get_line())));
          int current_slide_index;
          std::stringstream current_slide_index_stream(slide_index_text_content);
          current_slide_index_stream >> current_slide_index;
          const std::vector<xmlpp::Node*> notes_nodes = (*current_slide_node)->find("notes");
          on_error(!notes_nodes.size(),
                   new Source_error(error_location(),
                                    Source_error::load,
                                    std::string("error importing XML file, node: ")
                                    + (*current_slide_node)->get_path() + std::string(" notes not found"),
                                    file_path +
                                    ih_format_XML_file_line((*current_slide_node)->get_line())));
          const std::vector<xmlpp::Node*> notes_text_nodes = notes_nodes[0]->find("text()");
          if (notes_text_nodes.size())
          {
            // the reinterpret cast is used to convert unsigned char to char
            // unfortunately, there seems to be no obvious way around this
            const char *notes_content = reinterpret_cast<const char*>(notes_text_nodes[0]->cobj()->content);
            if (notes_content && ((current_slide_index >= 0) && (current_slide_index < slide_count())))
            {
              m_slides[current_slide_index]->set_notes(std::string(notes_content));
            }
          }
          // else if notes are empty, there might be no notes text node, so just go on
        }
      }
      catch (const xmlpp::exception& e)
      {
        const std::string diagnostic = std::string("error importing XML file: ") + e.what();
        on_error(true, new Source_error(error_location(),
                                        Source_error::load,
                                        diagnostic,
                                        file_path));
      }

      return_value = true;

      goto exit_point;
      end_error_block();

      default_error_handler_and_cleanup(return_error,
                                        return_value,
                                        false);
      goto exit_point;

    exit_point:
      return return_value;
    }

    //
    // Protected member functions
    //

    template <class Image> Slide_collection<Image>::Slide_collection()
      : m_source_loaded(false),
        m_source(),
        m_current_slide_index(0)
    {
      postcondition(mf_invariant(false));
    }

    template <class Image> bool Slide_collection<Image>::mf_invariant(bool check_base_class) const
    {
      static_cast<void>(check_base_class); // avoid unused warning

      bool return_value = false;

      if (m_current_slide_index < 0)
      {
        goto exit_point;
      }

      if (m_source_loaded
          && ((slide_count() < 1)
              || (m_current_slide_index >= slide_count())))
      {
        goto exit_point;
      }

      return_value = true;

    exit_point:
      return return_value;
    }
  }
}

#endif // SLIDE_COLLECTION_HPP_
