// Ring_buffer.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

//
// This code was based on the PortAudio file pa_ringbuffer.c:
//
//    This program uses the PortAudio Portable Audio Library.
//    For more information see: http://www.portaudio.com
//    Copyright (c) 1999-2000 Ross Bencina and Phil Burk
//
//    Permission is hereby granted, free of charge, to any person obtaining
//    a copy of this software and associated documentation files
//    (the "Software"), to deal in the Software without restriction,
//    including without limitation the rights to use, copy, modify, merge,
//    publish, distribute, sublicense, and/or sell copies of the Software,
//    and to permit persons to whom the Software is furnished to do so,
//    subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be
//    included in all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
//    IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
//    ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
//    CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

// Note that this is safe only for a single-thread reader and a
// single-thread writer.

#include "Ring_buffer.hpp"
#include "error/Error.hpp"
#include <cstring>

using namespace Roan_trail;

//
// Constructor/destructor/copy
//

Ring_buffer::Ring_buffer(Long_int num_bytes)
  : m_buffer_size(0),
    m_write_index(0),
    m_read_index(0),
    m_big_mask(0),
    m_small_mask(0),
    m_buffer(0)
{
  precondition(num_bytes > 0);

  if (((num_bytes - 1) & num_bytes) != 0)
  {
    *const_cast<Long_int*>(&m_buffer_size) = round_up_to_pow2(num_bytes); // const cast OK, in constructor
    assert(is_pow2(m_buffer_size)
           && "error ring buffer size is not a power of 2");
  }
  else
  {
    *const_cast<Long_int*>(&m_buffer_size) = num_bytes; // const cast OK, in constructor
  }

  m_buffer = new char[m_buffer_size];

  // equivalent to clear()
  memset(m_buffer,
         0,
         static_cast<size_t>(m_buffer_size));

  *const_cast<Long_int*>(&m_big_mask) = (m_buffer_size * 2) - 1; // const cast OK, in constructor
  *const_cast<Long_int*>(&m_small_mask) = m_buffer_size - 1; // const cast OK, in constructor

  postcondition(mf_invariant(false));
}

Ring_buffer::~Ring_buffer()
{
  precondition(mf_invariant(false));

  delete [] m_buffer;
}

//
// Read/write
//

Long_int Ring_buffer::read_ahead_data(void *data, Long_int num_bytes) const
{
  precondition(data
               && (num_bytes > 0)
               && mf_invariant());

  Long_int size_1;
  Long_int size_2;
  void *data_1;
  void *data_2;
  const Long_int num_read = mf_read_regions(num_bytes,
                                            &data_1,
                                            &size_1,
                                            &data_2,
                                            &size_2);

  if (size_2 > 0)
  {
    memcpy(data,
           data_1,
           static_cast<size_t>(size_1));
    data = reinterpret_cast<char*>(data) + size_1;
    memcpy(data,
           data_2,
           static_cast<size_t>(size_2));
  }
  else
  {
    memcpy(data,
           data_1,
           static_cast<size_t>(size_1));
  }

  postcondition(num_read >= 0);
  return num_read;
}


Long_int Ring_buffer::write_data(const void* data, Long_int num_bytes)
{
  precondition(data
               && (num_bytes > 0)
               && mf_invariant());

  Long_int size_1;
  Long_int size_2;
  void* data_1;
  void* data_2;
  const Long_int num_written = mf_write_regions(num_bytes,
                                                &data_1,
                                                &size_1,
                                                &data_2,
                                                &size_2);

  if (size_2 > 0)
  {
    memcpy(data_1,
           data,
           static_cast<size_t>(size_1));
    data = reinterpret_cast<const char*>(data) + size_1;
    memcpy(data_2,
           data,
           static_cast<size_t>(size_2));
  }
  else
  {
    memcpy(data_1,
           data,
           static_cast<size_t>(size_1));
  }
  mf_advance_write_index(num_written);

  postcondition((num_written >= 0)
                && mf_invariant());
  return num_written;
}


Long_int Ring_buffer::read_data(void* data, Long_int num_bytes)
{
  precondition(data
               && (num_bytes > 0)
               && mf_invariant());

  Long_int size_1;
  Long_int size_2;
  void* data_1;
  void* data_2;
  const Long_int num_read = mf_read_regions(num_bytes,
                                            &data_1,
                                            &size_1,
                                            &data_2,
                                            &size_2);

  if (size_2 > 0)
  {
    memcpy(data,
           data_1,
           static_cast<size_t>(size_1));
    data = reinterpret_cast<char*>(data) + size_1;
    memcpy(data,
           data_2,
           static_cast<size_t>(size_2));
  }
  else
  {
    memcpy(data,
           data_1,
           static_cast<size_t>(size_1));
  }

  mf_advance_read_index(num_read);

  postcondition((num_read >= 0)
                && mf_invariant());
  return num_read;
}

//
// Mutators
//

void Ring_buffer::clear()
{
  precondition(mf_invariant());

  memset(m_buffer,
         0,
         static_cast<size_t>(m_buffer_size));

  postcondition(mf_invariant());
}

//
// Protected member functions
//

// (Note: mf_invariant() has a benign race condition due to lock free algorithm)
bool Ring_buffer::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class);

  bool return_value = false;

  // range checking
  if (m_buffer_size <= 0)
  {
    goto exit_point;
  }
  else if ((m_write_index < 0) || (m_read_index < 0))
  {
    goto exit_point;
  }
  // object allocation checking
  else if (!m_buffer)
  {
    goto exit_point;
  }
  else
  {
    return_value = true;
  }

 exit_point:

  return return_value;
}

//
// Private member functions
//

// Get address of region(s) to which we can write data.
// If the region is contiguous, size_2 will be zero.
// If non-contiguous, size_2 will be the size of second region.
// Returns room available to be written or num_bytes, whichever is smaller.
Long_int Ring_buffer::mf_write_regions(Long_int num_bytes,
                                       void** data_ptr_1, // (check_code_ignore)
                                       Long_int* size_ptr_1,
                                       void** data_ptr_2, // (check_code_ignore)
                                       Long_int* size_ptr_2)
{
  precondition((num_bytes > 0)
               && data_ptr_1
               && size_ptr_1
               && data_ptr_2
               && size_ptr_2);

  const Long_int available = write_available();
  if (num_bytes > available)
  {
    num_bytes = available;
  }
  // Check to see if write is not contiguous
  const Long_int index = m_write_index & m_small_mask;
  if ((index + num_bytes) > m_buffer_size)
  {
    // Write data in two blocks that wrap the buffer
    const Long_int first_half = m_buffer_size - index;
    *data_ptr_1 = &m_buffer[index];
    *size_ptr_1 = first_half;
    *data_ptr_2 = &m_buffer[0];
    *size_ptr_2 = num_bytes - first_half;
  }
  else
  {
    *data_ptr_1 = &m_buffer[index];
    *size_ptr_1 = num_bytes;
    *data_ptr_2 = 0;
    *size_ptr_2 = 0;
  }

  return num_bytes;
}

// Get address of region(s) from which we can read data.
// If the region is contiguous, size_2 will be zero.
// If non-contiguous, size_2 will be the size of second region.
// Returns room available to be written or num_bytes, whichever is smaller.
Long_int Ring_buffer::mf_read_regions(Long_int num_bytes,
                                     void** data_ptr_1, // (check_code_ignore)
                                     Long_int* size_ptr_1,
                                     void** data_ptr_2, // (check_code_ignore)
                                     Long_int* size_ptr_2) const
{
  precondition((num_bytes > 0)
               && data_ptr_1
               && size_ptr_1
               && data_ptr_2
               && size_ptr_2);

  const Long_int available = read_available();
  if (num_bytes > available)
  {
    num_bytes = available;
  }
  // Check to see if read is not contiguous.
  const Long_int index = m_read_index & m_small_mask;
  if ((index + num_bytes) > m_buffer_size)
  {
    // Write data in two blocks that wrap the buffer.
    const Long_int first_half = m_buffer_size - index;
    *data_ptr_1 = &m_buffer[index];
    *size_ptr_1 = first_half;
    *data_ptr_2 = &m_buffer[0];
    *size_ptr_2 = num_bytes - first_half;
  }
  else
  {
    *data_ptr_1 = &m_buffer[index];
    *size_ptr_1 = num_bytes;
    *data_ptr_2 = 0;
    *size_ptr_2 = 0;
  }

  return num_bytes;
}

// (Note: mf_advance_write_index() has a benign race condition due to lock free algorithm)
Long_int Ring_buffer::mf_advance_write_index(Long_int num_bytes)
{
  precondition((num_bytes > 0));

  // we need to ensure that previous writes are seen before we update the write index
  __sync_synchronize();

  return (m_write_index = (m_write_index + num_bytes) & m_big_mask);
}

// (Note: mf_advance_read_index() has a benign race condition due to lock free algorithm)
Long_int Ring_buffer::mf_advance_read_index(Long_int num_bytes)
{
  precondition((num_bytes > 0));

  // we need to ensure that previous writes are always
  // seen before updating the index.
  __sync_synchronize();

  return (m_read_index = (m_read_index + num_bytes) & m_big_mask);
}
