// Narration_recorder.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Narration_recorder.hpp"
#include "Action_recorder.hpp"
#include "Segment.hpp"
#include "Sound_recorder.hpp"
#include "Sound_recorder_config.hpp"
#include "Sound_file_config.hpp"
#include "error/Record_error.hpp"
#include <libxml++/exceptions/exception.h>
#include <libxml++/nodes/element.h>
#include <libxml++/nodes/node.h>
#include <libxml2/libxml/tree.h>
#include <string>
#include <sstream>
#include <vector>

using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using xmlpp::Element;
using xmlpp::Node;
using Roan_trail::Long_int;
using Roan_trail::Error;
using Roan_trail::Error_param;
using namespace Roan_trail::Recorder;

//
// Internal helpers
//

namespace
{
  string ih_format_XML_file_line(int line)
  {
    stringstream line_stream;
    line_stream << ":" << line;

    return line_stream.str();
  }
}

//
// Constructor/destructor
//

Narration_recorder::Narration_recorder()
  : m_action_recorder(new Action_recorder()),
    m_sound_recorder(new Sound_recorder()),
    m_segment_cache(),
    m_temporary_directory(),
    m_index_frames_written_count(0)
{
  postcondition(mf_invariant(false));
}

Narration_recorder::~Narration_recorder()
{
  precondition(mf_invariant(false));

  // stop if not already
  if (Recorder::state_stopped != Recorder::state())
  {
    Error_param error(false);
    m_sound_recorder->stop(error); // ignore error
    const Long_int frame = m_sound_recorder->frames_written_count();
    m_action_recorder->stop(frame);
    Recorder::stop();
  }

  delete m_sound_recorder;
  delete m_action_recorder;
}

//
// Control
//

//
//   ...startup/shutdown functions
//

bool Narration_recorder::startup(Sound_recorder_config& config, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  // start sound recorder
  Error_param error;
  const bool started = m_sound_recorder->startup(config, error);
  on_error(!started, new Record_error(error_location(),
                                      Record_error::general,
                                      error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Narration_recorder::shutdown(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;

  // shutdown sound recorder
  const bool ended = m_sound_recorder->shutdown(error);
  on_error(!ended, new Record_error(error_location(),
                                    Record_error::general,
                                    error()));

  // stop if not already
  if (Recorder::state_stopped != state())
  {
    const Long_int frame = m_sound_recorder->frames_written_count();
    m_index_frames_written_count = frame;
    m_action_recorder->stop(frame);
    Recorder::stop();
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (Recorder::state_stopped != state())
  {
    const Long_int frame = m_sound_recorder->frames_written_count();
    m_index_frames_written_count = frame;
    m_action_recorder->stop(frame);
    Recorder::stop();
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition((Recorder::state_stopped == state())
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
//   ...basic recorder functions
//

bool Narration_recorder::record(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  return record(0, return_error);
}

bool Narration_recorder::pause(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  // update sound recorder
  Error_param error;
  const bool paused = m_sound_recorder->pause(error);
  on_error(!paused, new Record_error(error_location(),
                                     Record_error::general,
                                     error()));

  // update action recorder
  m_action_recorder->pause();

  Recorder::pause();

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Narration_recorder::stop(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  // update sound recorder
  Error_param error;
  const bool stopped = m_sound_recorder->stop(error);
  on_error(!stopped, new Record_error(error_location(),
                                      Record_error::general,
                                      error()));

  // update action recorder
  const Long_int frame = m_sound_recorder->frames_written_count();
  m_index_frames_written_count = frame;
  m_action_recorder->stop(frame);

  Recorder::stop();

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(((return_value && (Recorder::state_stopped == state()))
                 || !return_value)
                && return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
//   ...narration functions
//

bool Narration_recorder::record(Long_int index, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  // update sound recorder
  Error_param error;
  const bool recording = m_sound_recorder->record(error);
  on_error(!recording, new Record_error(error_location(),
                                        Record_error::general,
                                        error()));

  // update action recorder
  if (Recorder::state_paused == state())
  {
    // resume recording
    const Long_int frame = m_sound_recorder->frames_written_count();
    m_action_recorder->record(index, frame);
    m_index_frames_written_count = frame;
  }
  else if (Recorder::state_stopped == state())
  {
    // record from stop
    const Long_int audio_frame_rate = static_cast<Long_int>(m_sound_recorder->config()->sound_file->sample_rate);
    m_action_recorder->set_audio_frame_rate(audio_frame_rate);
    m_action_recorder->record(index);
    m_index_frames_written_count = 0;
    m_segment_cache.clear();
  }
  else
  {
    assert(false && "invalid state for record in narration recorder");
  }

  Recorder::record();

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

bool Narration_recorder::new_index(Long_int index, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  // error if not recording
  on_error((Recorder::state_recording != state()), new Record_error(error_location(),
                                                                    Record_error::invalid_state,
                                                                    "Not recording for new index"));

  // update action recorder
  const Long_int frame = m_sound_recorder->frames_written_count();
  m_index_frames_written_count = frame;
  m_action_recorder->new_index(frame, index);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition((Recorder::state_recording == state())
                && mf_invariant());
  return return_value;
}

bool Narration_recorder::retake(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  on_error((Recorder::state_recording != state()), new Record_error(error_location(),
                                                                    Record_error::invalid_state,
                                                                    "Not recording for retake"));

  // update action recorder
  const Long_int frame = m_sound_recorder->frames_written_count();
  m_index_frames_written_count = frame;
  m_action_recorder->retake(frame);

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition((Recorder::state_recording == state())
                && mf_invariant());
  return return_value;
}

bool Narration_recorder::mute(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  // update sound recorder
  Error_param error;
  const bool muted = m_sound_recorder->mute(error);
  on_error(!muted, new Record_error(error_location(),
                                    Record_error::general,
                                    error()));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// Accessors
//

bool Narration_recorder::is_muted() const
{
  precondition(mf_invariant());

  return m_sound_recorder->is_muted();
}

bool Narration_recorder::is_started() const
{
  precondition(mf_invariant());

  return m_sound_recorder->is_started();
}

bool Narration_recorder::can_record() const
{
  precondition(mf_invariant());

  return m_sound_recorder->can_record();
}

void Narration_recorder::recorded_time(string& return_recorded_time, char separator) const
{
  precondition(mf_invariant());

  m_sound_recorder->recorded_time(return_recorded_time, separator);
}

void Narration_recorder::recorded_time_for_current_index(string& return_recorded_time, char separator) const
{
  precondition(mf_invariant());

  Long_int frames = 0;
  if (Recorder::state_stopped != Recorder::state())
  {
    const Long_int current_audio_frame = m_sound_recorder->frames_written_count();
    frames = current_audio_frame - m_index_frames_written_count;
    assert(frames >= 0 && "error, invalid frame count for current index");
  }

  m_sound_recorder->frames_to_recorded_time(frames, separator, return_recorded_time);
}

void Narration_recorder::sound_levels_RMS(vector<double>& return_sound_levels) const
{
  precondition(mf_invariant());

  m_sound_recorder->sound_levels_RMS(return_sound_levels);
}

void Narration_recorder::sound_levels_peak(vector<double>& return_sound_levels) const
{
  precondition(mf_invariant());

  m_sound_recorder->sound_levels_peak(return_sound_levels);
}

bool Narration_recorder::frames(Long_int& return_total_frames,
                                Long_int& return_index_frames,
                                Error_param& return_error) const
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  if (Recorder::state_stopped == Recorder::state())
  {
    if (have_recording())
    {
      if (!m_segment_cache.size())
      {
        Error_param error;
        const bool got_segments = m_action_recorder->segments(m_segment_cache, error);
        on_error(!got_segments, new Record_error(error_location(),
                                                 Record_error::general,
                                                 error()));
      }
      assert(m_segment_cache.size() >= 1
             && "error, must be at least 1 segment in action cache");
      const Segment start = m_segment_cache[0];
      const Segment end = m_segment_cache[m_segment_cache.size() - 1];
      assert(end.end_frame() >= start.start_frame()
             && "error, end frame must be >= start frame");
      const Long_int frames = end.end_frame() - start.start_frame() + 1;
      return_index_frames = 0;
      return_total_frames = frames;
    }
    else
    {
      return_index_frames = 0;
      return_total_frames = 0;
    }
  }
  else
  {
    const Long_int current_audio_frame = m_sound_recorder->frames_written_count();
    return_total_frames = current_audio_frame;
    const Long_int index_audio_frame = current_audio_frame - m_index_frames_written_count;
    return_index_frames = index_audio_frame;
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

const Sound_recorder_config* Narration_recorder::config() const
{
  precondition(mf_invariant());

  return m_sound_recorder->config();
}

string Narration_recorder::output_file() const
{
  precondition(mf_invariant());

  return m_sound_recorder->output_file();
}

bool Narration_recorder::have_recording() const
{
  precondition(mf_invariant());

  return m_sound_recorder->have_recording();
}

const Error *Narration_recorder::record_error() const
{
  precondition(mf_invariant());

  return m_sound_recorder->record_error();
}

bool Narration_recorder::segments(vector<Segment>& return_segments, Error_param& return_error) const
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  Error_param error;
  const bool got_segments = m_action_recorder->segments(return_segments, error);
  if (!got_segments)
  {
    string sound_file = m_sound_recorder->output_file();
    if ("" == sound_file)
    {
      sound_file = "unknown file";
    }
    Error *record_error = new Record_error(error_location(),
                                           Record_error::general,
                                           error());
    record_error->error_dictionary()[Error::file_path_error_key] = sound_file;
    on_error(true, record_error);
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

Long_int Narration_recorder::overflow_count() const
{
  precondition(mf_invariant());

  return m_sound_recorder->input_overflow_count() +
    m_sound_recorder->output_overflow_count();
}

Long_int Narration_recorder::input_overflow_count() const
{
  precondition(mf_invariant());

  return m_sound_recorder->input_overflow_count();
}

Long_int Narration_recorder::output_overflow_count() const
{
  precondition(mf_invariant());

  return m_sound_recorder->output_overflow_count();
}

bool Narration_recorder::space_available(Long_int& return_bytes_available,
                                         double& return_fraction_available,
                                         double& return_seconds_remaining) const
{
  precondition(mf_invariant());

  return m_sound_recorder->space_available(return_bytes_available,
                                           return_fraction_available,
                                           return_seconds_remaining);
}

bool Narration_recorder::output_file_size(Long_int& return_file_size, double& return_data_rate) const
{
  precondition(mf_invariant());

  return m_sound_recorder->output_file_size(return_file_size, return_data_rate);
}

bool Narration_recorder::is_RMS_metering_enabled() const
{
  precondition(mf_invariant());

  return m_sound_recorder->is_RMS_metering_enabled();
}

bool Narration_recorder::is_peak_metering_enabled() const
{
  precondition(mf_invariant());

  return m_sound_recorder->is_peak_metering_enabled();
}

int Narration_recorder::metering_channels() const
{
  precondition(mf_invariant());

  return m_sound_recorder->metering_channels();
}


//
// Mutators
//

void Narration_recorder::reset_overflow_count()
{
  precondition(mf_invariant());

  m_sound_recorder->reset_input_overflow_count();
  m_sound_recorder->reset_output_overflow_count();

  postcondition(mf_invariant());
}

void Narration_recorder::enable_RMS_metering(bool enable)
{
  precondition(mf_invariant());

  m_sound_recorder->enable_RMS_metering(enable);

  postcondition(mf_invariant());
}

void Narration_recorder::enable_peak_metering(bool enable)
{
  precondition(mf_invariant());

  m_sound_recorder->enable_peak_metering(enable);

  postcondition(mf_invariant());
}

void Narration_recorder::set_metering_channels(int channel_count)
{
  precondition((channel_count >= 0)
               && mf_invariant());

  m_sound_recorder->set_metering_channels(channel_count);

  postcondition(mf_invariant());
}

void Narration_recorder::cleanup_temporary_file()
{
  precondition(mf_invariant());

  m_sound_recorder->cleanup_temporary_file();

  postcondition(mf_invariant());
}

void Narration_recorder::clear_record_error()
{
  precondition(mf_invariant());

  m_sound_recorder->clear_record_error();

  postcondition(mf_invariant());
}

//
// Export
//

bool Narration_recorder::export_as_XML(const string& file_path,
                                       Node* XML_node,
                                       Error_param& return_error)
{
  precondition(XML_node
               && !return_error());

  bool return_value = false;
  start_error_block();

  Error_param error;

  try
  {
    Node* recorder_node = XML_node->add_child("recorder");

    // audio recording path
    Element* audio_recording_path_element = recorder_node->add_child("audio_recording_path");
    audio_recording_path_element->add_child_text(output_file());

    // audio recording frame rate
    Element* audio_recording_frame_rate_element = recorder_node->add_child("audio_recording_frame_rate");
    stringstream frame_rate_stream;
    frame_rate_stream << config()->sound_file->sample_rate;
    audio_recording_frame_rate_element->add_child_text(frame_rate_stream.str());

    if (have_recording())
    {
      vector<Segment> recording_segments;
      bool got_segments = segments(recording_segments, error);
      on_error(!got_segments, new Record_error(error_location(),
                                               Record_error::general,
                                               error()));
      // segments
      Node* segments_node = recorder_node->add_child("segments");;
      for (size_t segment_number = 0; segment_number < recording_segments.size(); ++segment_number)
      {
        const Segment &segment = recording_segments[segment_number];
        Node* segment_node = segments_node->add_child("segment");
        // index
        Element* index_element = segment_node->add_child("index");
        stringstream segment_index_stream;
        segment_index_stream << segment.index();
        index_element->add_child_text(segment_index_stream.str());
        // start frame
        Element* start_frame_element = segment_node->add_child("start_frame");
        stringstream start_frame_stream;
        start_frame_stream << segment.start_frame();
        start_frame_element->add_child_text(start_frame_stream.str());
        // end frame
        Element* end_frame_element = segment_node->add_child("end_frame");
        stringstream end_frame_stream;
        end_frame_stream << segment.end_frame();
        end_frame_element->add_child_text(end_frame_stream.str());
        // ordinality
        Element* ordinality_element = segment_node->add_child("ordinality");
        stringstream ordinality_stream;
        ordinality_stream << segment.ordinality();
        ordinality_element->add_child_text(ordinality_stream.str());
        Element* retake_element = segment_node->add_child("retake");
        retake_element->add_child_text(segment.is_retake() ? "1" : "0");
      }
    }
  }
  catch (const xmlpp::exception& e)
  {
    string diagnostic = string("error exporting XML to file: ") + e.what();
    on_error(true, new Record_error(error_location(),
                                    Record_error::save,
                                    diagnostic,
                                    file_path));
  }

  return_value = true;

  goto exit_point;
  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Protected member functions
//

// invariant check
bool Narration_recorder::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (m_index_frames_written_count < 0)
  {
    goto exit_point;
  }

  // states of recorders must be synchronized
  if ((state() != m_action_recorder->state()) || (state() != m_sound_recorder->state()))
  {
    goto exit_point;
  }

  // member objects which must exist during this object's lifetime
  if (!m_sound_recorder || !m_action_recorder)
  {
    goto exit_point;
  }

  return_value = (!check_base_class || Recorder::mf_invariant(check_base_class));
  goto exit_point;

 exit_point:
  return return_value;
}
