// Async_audio_file_writer.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef ASYNC_AUDIO_FILE_WRITER_HPP_
#define ASYNC_AUDIO_FILE_WRITER_HPP_

#include "common.hpp"
#include <pthread.h>
#include <string>

typedef struct SNDFILE_tag SNDFILE;

namespace Roan_trail
{
  class Error_param;
  class Ring_buffer;

  namespace Recorder
  {
    class AAFW_error;
    struct Sound_recorder_config;

    class Async_audio_file_writer
    {
    public:
      // constructor/destructor
      Async_audio_file_writer(bool enable_overflow_logging);
      virtual ~Async_audio_file_writer();
      // accessors
      Long_int frames_written_count() { return __sync_fetch_and_add(&m_frames_written_count, 0x0); }
      bool temporary_output_file(std::string& return_file)
      {
        return_file = m_temporary_output_file; return ("" != return_file);
      }
      bool have_write_error() const
      { return __sync_fetch_and_add(const_cast<int32_t *>(&m_have_write_error), 0); }

      AAFW_error* write_error() const;
      bool running() const { return __sync_fetch_and_add(const_cast<int32_t *>(&m_writer_thread_running), 0); }

      // control
      bool start(const Sound_recorder_config& recorder_config, Error_param& return_error);
      bool write(int number_frames,
                 const void* input_buffer,
                 bool& return_have_overflow);
      bool pause(Error_param& return_error);
      bool stop(Error_param& return_error);
      // other
      bool file_exists(const std::string& file_path);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      // aligned data members used for synchronized access
      int32_t __attribute__ ((aligned (4))) m_write_loop_finish;
      int32_t __attribute__ ((aligned (4))) m_writer_thread_running;
      int32_t __attribute__ ((aligned (4))) m_have_write_error;
      int64_t __attribute__ ((aligned (8))) m_frames_written_count;
      //
      Sound_recorder_config *m_recorder_config;
      SNDFILE* m_output_file;
      std::string m_output_file_path;
      char* m_output_buffer;
      Ring_buffer* m_ring_buffer;
      int m_max_frames;
      int m_write_chunk_size;
      int m_write_buffer_size;
      int m_sample_size;
      pthread_t m_writer_thread;
      pthread_mutex_t m_input_mutex;
      pthread_cond_t m_input_cond;
      pthread_mutex_t m_thread_mutex;
      pthread_cond_t m_thread_cond;
      bool m_overflow_logging_enabled;
      std::string m_temporary_output_file;
      AAFW_error* m_write_loop_error;
      int m_write_error_code;
      bool m_clear_frames_written_count;
      // private member functions
      SNDFILE* mf_open_sound_file(std::string& return_output_file_path,
                                  bool& return_created_temp_file,
                                  Error_param& return_error);
      void mf_set_write_error_code(int write_error_code);
      void mf_set_write_loop_error(AAFW_error* write_loop_error);
      bool mf_wait_for_min_read_available(Long_int min_read_available,
                                          Long_int& return_available,
                                          bool& return_finished,
                                          Error_param& return_error);
      // worker thread function
      static void* mf_write_loop(void* arg);
      // prevent compiler from generating
      Async_audio_file_writer(const Async_audio_file_writer& writer);
      Async_audio_file_writer& operator=(const Async_audio_file_writer& writer);
    };
  }
}

#endif // ASYNC_AUDIO_FILE_WRITER_HPP_
