// AV_compat.hpp
//
// Copyright 2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// Compatibility class for ffmpeg

#ifndef AV_COMPAT_HPP_
#define AV_COMPAT_HPP_

#include "common.hpp"

extern "C"
{
#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
}

namespace Roan_trail
{
  namespace Recorder
  {
    struct Sound_file_config;
  }

  typedef int Endianness;

  namespace Builder
  {
    class AV_compat
    {
    public:
      // constructor/destructor
      AV_compat();
      ~AV_compat();

      // class member functions

      static bool codec_ID_for_sound_file_config(const Roan_trail::Recorder::Sound_file_config& config,
                                                 enum CodecID& return_codec_ID);
      static bool endianness_for_output_format(const AVOutputFormat* output_format,
                                               const Roan_trail::Recorder::Sound_file_config& config,
                                               Endianness& return_endian);

#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(52, 123, 0) // (libavcodec 52.123.0 ffmpeg release 0.7.11)
      //
      static bool sample_fmt_for_sound_file_config(const Roan_trail::Recorder::Sound_file_config& config,
                                                   enum SampleFormat& return_sample_format);
      //
#else
      //
      static bool sample_fmt_for_sound_file_config(const Roan_trail::Recorder::Sound_file_config& config,
                                                   enum AVSampleFormat& return_sample_format);
      //
#endif

#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(52, 64, 2) // (libavformat 52.64.2 in ffmpeg release 0.6.5)
      //
      static AVOutputFormat* av_guess_format(const char* short_name,
                                             const char* file_name,
                                             const char* mime_type)
      {
        return ::guess_format(short_name,
                              file_name,
                              mime_type);
      }
      //
#else
      //
      static AVOutputFormat *av_guess_format(const char* short_name,
                                             const char* file_name,
                                             const char* mime_type)
      {
        return ::av_guess_format(short_name,
                                 file_name,
                                 mime_type);
      }
      //
#endif

#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(52, 123, 0) // (libavcodec 52.123.0 ffmpeg release 0.7.11)
      //
      static int avcodec_open(AVCodecContext *context, AVCodec *codec)
      {
        return ::avcodec_open(context, codec);
      }
      //
#else
      //
      static int avcodec_open(AVCodecContext *context, AVCodec *codec)
      {
        return ::avcodec_open2(context,
                               codec,
                               0);
      }
      //
#endif

#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(52, 111, 0) // (libavformat 52.111.0 in ffmpeg release 0.7.11)
      //
      static int avio_open(ByteIOContext** context, const char* URL, int flags)
      {
        return ::url_fopen(context,
                           URL,
                           flags);
      }
      static int avio_close(ByteIOContext* context)
      {
        return ::url_fclose(context);
      }
      static int avformat_write_header(AVFormatContext* context)
      {
        return ::av_write_header(context);
      }
      //
#else
      //
      static int avio_open(AVIOContext** context, const char* file_name, int flags)
      {
        return ::avio_open(context,
                           file_name,
                           flags);
      }
      static int avio_close(AVIOContext* context)
      {
        return ::avio_close(context);
      }
      static int avformat_write_header(AVFormatContext* s)
      {
        return ::avformat_write_header(s, 0);
      }
      //
#endif

#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(53, 24, 2) // (libavformat 53.24.2 in ffmpeg release 0.9.1)
      //
      static void av_zero_dts(AVFormatContext* context, AVStream* ref_stream)
      {
        return ::av_update_cur_dts(context,
                                   ref_stream,
                                   0);
      }
      static AVStream* avformat_new_stream(AVFormatContext *context, int track_ID)
      {
        return ::av_new_stream(context, track_ID);
      }
      //
#else
      //
      static void av_zero_dts(AVFormatContext* context, AVStream* ref_stream);
      static AVStream* avformat_new_stream(AVFormatContext* context, int track_ID)
      {
        AVStream* new_stream = ::avformat_new_stream(context, 0);
        new_stream->id = track_ID;
        return new_stream;
      }
      //
#endif

      // class constants
#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(52, 72, 2) // (libavcodec 52.72.2 ffmpeg release 0.6.5)
      //
      static const enum CodecType codec_type_video = CODEC_TYPE_VIDEO;
      static const enum CodecType codec_type_audio = CODEC_TYPE_AUDIO;
      static const int pkt_flag_key = PKT_FLAG_KEY;
      //
#else
      //
      static const enum AVMediaType codec_type_video = AVMEDIA_TYPE_VIDEO;
      static const enum AVMediaType codec_type_audio = AVMEDIA_TYPE_AUDIO;
      static const int pkt_flag_key = AV_PKT_FLAG_KEY;
      //
#endif

#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(52, 123, 0) // (libavcodec 52.123.0 ffmpeg release 0.7.11)
      //
      static const enum SampleFormat sample_fmt_U8 = SAMPLE_FMT_U8;
      static const enum SampleFormat sample_fmt_S16 = SAMPLE_FMT_S16;
      static const enum SampleFormat sample_fmt_S32 = SAMPLE_FMT_S32;
      static const enum SampleFormat sample_fmt_FLT = SAMPLE_FMT_FLT;
      static const enum SampleFormat sample_fmt_DBL = SAMPLE_FMT_DBL;
      //
      static const int av_picture_type_I = FF_I_TYPE; // intra frame
      static const int av_picture_type_P = FF_P_TYPE; // predicted frame
      //
#else
      //
      static const enum AVSampleFormat sample_fmt_U8 = AV_SAMPLE_FMT_U8;
      static const enum AVSampleFormat sample_fmt_S16 = AV_SAMPLE_FMT_S16;
      static const enum AVSampleFormat sample_fmt_S32 = AV_SAMPLE_FMT_S32;
      static const enum AVSampleFormat sample_fmt_FLT = AV_SAMPLE_FMT_FLT;
      static const enum AVSampleFormat sample_fmt_DBL = AV_SAMPLE_FMT_DBL;
      //
      static const enum AVPictureType av_picture_type_I = AV_PICTURE_TYPE_I; // intra frame
      static const enum AVPictureType av_picture_type_P = AV_PICTURE_TYPE_P; // predicted frame
      //
#endif
    private:
      // prevent compiler from generating
      AV_compat(const AV_compat& c);
      AV_compat& operator=(const AV_compat& c);
    };
  }
}

#endif // AV_COMPAT_HPP_
