// AV_compat.cpp
//
// Copyright 2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "AV_compat.hpp"
#include "Sound_file_config.hpp"
#include <string>

using std::string;
using Roan_trail::Recorder::Sound_file_config;
using namespace Roan_trail::Builder;

// Note:
// The ffmpeg file: libavformat/isom.c lists the supported codecs for some containers, including "mov"

//
// Constructor/destructor
//

AV_compat::AV_compat()
{
#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(53, 24, 2) // (libavformat 53.24.2 in ffmpeg release 0.9.1)
  //
#else
  //
  avformat_network_init();
  //
#endif
}

AV_compat::~AV_compat()
{
#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(53, 24, 2) // (libavformat 53.24.2 in ffmpeg release 0.9.1)
  //
#else
  //
  avformat_network_deinit();
  //
#endif
}

//
// Class member functions
//

bool AV_compat::codec_ID_for_sound_file_config(const Sound_file_config& config, enum CodecID& return_codec_ID)
#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(52, 64, 2) // (libformat 52.64.2 ffmpeg release 0.6.5)
{
  bool return_value = true;

  switch (config.data_format)
  {
  case Sound_file_config::data_format_PCMS8:
    return_codec_ID = CODEC_ID_PCM_S8;
    break;
  case Sound_file_config::data_format_PCM16:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_S16LE;
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_S16BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  case Sound_file_config::data_format_PCM24:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_S24LE;
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_S24BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  case Sound_file_config::data_format_PCM32:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_S32LE;
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_S32BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  case Sound_file_config::data_format_float:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_F32BE; // MOV container only supports F32BE
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_F32BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  case Sound_file_config::data_format_double:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_F64BE; // MOV container only supports F64BE
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_F64BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  default:
    return_value = false;
    break;
  }
  goto exit_point;

 exit_point:
  return return_value;
}
#else
{
  bool return_value = true;

  switch (config.data_format)
  {
  case Sound_file_config::data_format_PCMS8:
    return_codec_ID = CODEC_ID_PCM_S8;
    break;
  case Sound_file_config::data_format_PCM16:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_S16LE;
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_S16BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  case Sound_file_config::data_format_PCM24:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_S24LE;
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_S24BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  case Sound_file_config::data_format_PCM32:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_S32LE;
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_S32BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  case Sound_file_config::data_format_float:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_F32LE;
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_F32BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  case Sound_file_config::data_format_double:
    switch (config.endianness)
    {
    case Sound_file_config::endian_little:
      return_codec_ID = CODEC_ID_PCM_F64LE;
      break;
    case Sound_file_config::endian_big:
      return_codec_ID = CODEC_ID_PCM_F64BE;
      break;
    default:
      return_value = false;
      break;
    }
    break;
  default:
    return_value = false;
    break;
  }
  goto exit_point;

 exit_point:
  return return_value;
}
#endif

bool AV_compat::endianness_for_output_format(const AVOutputFormat* output_format,
                                             const Sound_file_config& config,
                                             Sound_file_config::Endianness& return_endian)
#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(52, 64, 2) // (libformat 52.64.2 ffmpeg release 0.6.5)
{
  bool return_value = true;

  if (string("mov") == string(output_format->name))
  {
    switch (config.data_format)
    {
    case Sound_file_config::data_format_PCMS8:
    case Sound_file_config::data_format_PCM16:
    case Sound_file_config::data_format_PCM24:
    case Sound_file_config::data_format_PCM32:
      // both big- and little-endian are supported for these
      return_endian = config.endianness;
      break;
    case Sound_file_config::data_format_float:
    case Sound_file_config::data_format_double:
      // only big-endian supported for these
      return_endian = Sound_file_config::endian_big;
      break;
    default:
      return_value = false;
      break;
    }
  }
  else
  {
    return_value = false;
  }

  return return_value;
}
#else
{
  bool return_value = true;

  if (string("mov") == string(output_format->name))
  {
    switch (config.data_format)
    {
    case Sound_file_config::data_format_PCMS8:
    case Sound_file_config::data_format_PCM16:
    case Sound_file_config::data_format_PCM24:
    case Sound_file_config::data_format_PCM32:
    case Sound_file_config::data_format_float:
    case Sound_file_config::data_format_double:
      // both big- and little-endian are supported for these
      return_endian = config.endianness;
      break;
    default:
      return_value = false;
      break;
    }
  }
  else
  {
    return_value = false;
  }

  return return_value;
}
#endif

#if LIBAVCODEC_VERSION_INT < AV_VERSION_INT(52, 123, 0) // (libavcodec 52.123.0 ffmpeg release 0.7.11)
//
bool AV_compat::sample_fmt_for_sound_file_config(const Sound_file_config& config,
                                                 enum SampleFormat& return_sample_format)
//
#else
//
bool AV_compat::sample_fmt_for_sound_file_config(const Sound_file_config& config,
                                                 enum AVSampleFormat& return_sample_format)
//
#endif
{
  bool return_value = true;

  switch (config.data_format)
  {
  case Sound_file_config::data_format_PCMS8:
    return_sample_format = AV_compat::sample_fmt_S16;
    break;
  case Sound_file_config::data_format_PCM16:
    return_sample_format = AV_compat::sample_fmt_S16;
    break;
  case Sound_file_config::data_format_PCM24:
    return_sample_format = AV_compat::sample_fmt_S32;
    break;
  case Sound_file_config::data_format_PCM32:
    return_sample_format = AV_compat::sample_fmt_S32;
    break;
  case Sound_file_config::data_format_float:
    return_sample_format = AV_compat::sample_fmt_FLT;
    break;
  case Sound_file_config::data_format_double:
    return_sample_format = AV_compat::sample_fmt_DBL;
    break;
  default:
    return_value = false;
    break;
  }
  goto exit_point;

 exit_point:
  return return_value;
}

#if LIBAVFORMAT_VERSION_INT < AV_VERSION_INT(53, 24, 2) // (libavformat 53.24.2 in ffmpeg release 0.9.1)
//
#else
//
void AV_compat::av_zero_dts(AVFormatContext* context, AVStream* ref_stream)
{
  precondition(context
               && ref_stream);

  for (unsigned int i = 0; i < context->nb_streams; ++i)
  {
    AVStream* stream = context->streams[i];
    stream->cur_dts = 0;
  }
}
//
#endif
