// Scaled_image_widget.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// include *mm first to avoid conflicts
#include <gdkmm/drawable.h>
#include <gdkmm/general.h>
#include <gdkmm/pixbuf.h>

#include "Scaled_image_widget.hpp"

#include "../base/common.hpp"
#include "../base/error/Error.hpp"
#include <iostream>
#include <cstring>

using Gdk::Color;
using Gdk::Pixbuf;
using Gdk::Window;
using Glib::RefPtr;
using Gtk::Allocation;
using Gtk::Requisition;
using Gtk::Widget;
using Roan_trail::Error_param;
using namespace Gdk::Cairo;
using namespace Roan_trail::Kinetophone;

namespace
{
  Error_param iv_ignore_error(false);
  const Color ic_too_small_color = Color("dark grey");
  bool ih_is_valid_rect(const Rectangle& rect)
  {
    return ((rect.get_x() >= 0)
            && (rect.get_y() >= 0)
            && (rect.get_width() >= 0)
            && (rect.get_height() >= 0));
  }
}

// Note: The GType name will be gtkmm__CustomObject_Scaled_image_widget

// GType name:  std::cout << "GType name: " << G_OBJECT_TYPE_NAME(gobj()) << std::endl;
//
// Note: This shows that the GType still derives from GtkWidget:
// std::cout << "Gtype is a GtkWidget?:" << GTK_IS_WIDGET(gobj()) << std::endl;

Scaled_image_widget::Scaled_image_widget()
  : ObjectBase("Scaled_image_widget"),
    Widget(),
    m_ref_window(0),
    m_image_rect(),
    m_image(),
    m_scaled_image(),
    m_widget_sized(false)
{
  set_has_window(false);

  postcondition(mf_invariant(false));
}

Scaled_image_widget::~Scaled_image_widget()
{
  precondition(mf_invariant(false));
}

//
// View update
//

bool Scaled_image_widget::update(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  static_cast<void>(return_error); // avoid unused warning

  queue_draw(); // invalidate entire area of widget, triggering an expose event to redraw

  postcondition(!return_error()
                && mf_invariant());
  return true;
}

//
// Mutators
//

void Scaled_image_widget::set_image(const RefPtr<Pixbuf>& image)
{
  precondition(image
               && mf_invariant());

  m_image = image;
  m_scaled_image = image;

  update(iv_ignore_error); // ignore error

  postcondition((m_image == image)
                && (m_scaled_image == image)
                && mf_invariant());
}

//
// Protected member functions
//

bool Scaled_image_widget::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (get_realized() && !m_ref_window)
  {
    goto exit_point;
  }

  if ((!m_image && m_scaled_image)
      || (!m_scaled_image && m_image))
  {
    goto exit_point;
  }

  if (!ih_is_valid_rect(m_image_rect))
  {
    goto exit_point;
  }

  return_value = true;

  goto exit_point;

 exit_point:
  return return_value;
}

//
// Signal handlers
//

void Scaled_image_widget::on_size_request(Requisition* return_requisition)
{
  *return_requisition = Requisition();

  return_requisition->width = 0;
  return_requisition->height = 0;
}

void Scaled_image_widget::on_size_allocate(Allocation& allocation)
{
  // use what is given, (need to use an AspectFrame to maintain aspect)
  set_allocation(allocation);

  Rectangle window_rect(allocation.get_x(),
                        allocation.get_y(),
                        allocation.get_width(),
                        allocation.get_height());
  if (m_ref_window)
  {
    m_ref_window->move_resize(window_rect.get_x(),
                              window_rect.get_y(),
                              window_rect.get_width(),
                              window_rect.get_height());
  }

  m_image_rect = window_rect;

  m_widget_sized = true;
}

void Scaled_image_widget::on_map()
{
  Widget::on_map();
}

void Scaled_image_widget::on_unmap()
{
  Widget::on_unmap();
}

void Scaled_image_widget::on_realize()
{
  Widget::on_realize();

  ensure_style();

  if(!m_ref_window)
  {
    GdkWindowAttr attributes;
    memset(&attributes, 0, sizeof(attributes));

    Allocation allocation = get_allocation();

    // set initial position and size of the Gdk::Window:
    attributes.x = allocation.get_x();
    attributes.y = allocation.get_y();
    attributes.width = allocation.get_width();
    attributes.height = allocation.get_height();

    attributes.event_mask = get_events () | Gdk::EXPOSURE_MASK;
    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.wclass = GDK_INPUT_OUTPUT;

    RefPtr<Window> parent_window = get_window();
    m_ref_window = Window::create(parent_window,
                                  &attributes,
                                  GDK_WA_X | GDK_WA_Y);
    set_has_window();
    set_window(m_ref_window);

    m_ref_window->reference(); // Note: causes a leak, but can be ignored

    // make the widget receive expose events
    m_ref_window->set_user_data(gobj());
  }

}

void Scaled_image_widget::on_unrealize()
{
  m_ref_window.clear();

  Widget::on_unrealize();
}

bool Scaled_image_widget::on_expose_event(GdkEventExpose* event)
{
  static_cast<void>(event); // avoid unused warning

  // TODO: optimize more: change scaling interpolation if dynamically resizing (?), partial rendering
  if (m_image && m_ref_window)
  {
    const int src_x = 0;
    const int src_y = 0;
    const int dest_x = 0;
    const int dest_y = 0;
    const int dest_width = m_image_rect.get_width();
    const int dest_height = m_image_rect.get_height();
    if ((dest_width >= 16) && (dest_height >= 16))
    {
      // scale the image to the destination size, if necessary,
      // and draw image in destination
      if ((m_scaled_image->get_width() != dest_width)
          || (m_scaled_image->get_height() != dest_height))
      {
        m_scaled_image = m_image->scale_simple(dest_width,
                                               dest_height,
                                               Gdk::INTERP_BILINEAR);
      }
      m_scaled_image->render_to_drawable(m_ref_window,
                                         get_style()->get_black_gc(),
                                         src_x,
                                         src_y,
                                         dest_x,
                                         dest_y,
                                         dest_width,
                                         dest_height,
                                         Gdk::RGB_DITHER_NONE,
                                         0,
                                         0);
    }
    else
    {
      // draw a grey background
      if ((dest_width > 0) && (dest_height > 0))
      {
        Cairo::RefPtr<Cairo::Context> context = m_ref_window->create_cairo_context();
        set_source_color(context, ic_too_small_color);
        context->paint();
      }
    }
  }

  return true;
}
