// Kinetophone_dbus_test_client.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_dbus_client.hpp"
#include "Kinetophone_dbus_client_recorder.hpp"
#include "../base/common.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include "../base/error/DBus_error.hpp"
#include "../base/error/Posix_error.hpp"
#include <portaudio.h>
#include <iostream>
#include <string>
#include <map>
#include <cstdlib>
#include <csignal> // Standards Rule 21 deviation (check_code_ignore)

using DBus::BusDispatcher;
using DBus::Connection;
using std::cout;
using std::cerr;
using std::endl;
using std::string;
using std::map;
using Roan_trail::Long_int;
using Roan_trail::Error;
using Roan_trail::DBus_error;
using Roan_trail::Posix_error;
using namespace Roan_trail::Kinetophone;

//
// Internal variables, constants, and helper functions
//

namespace
{
  const string ic_kinetophone_dbus_server_name = "com.roantrail.dbus.kinetophone";
  const string ic_kinetophone_dbus_server_path = "/com/roantrail/dbus/kinetophone";

  pthread_t iv_thread;
  BusDispatcher* iv_dispatcher = 0;
  int32_t iv_exit_thread = 0;

  void ih_signal_handler(int s)
  {
    __sync_fetch_and_or(&iv_exit_thread, 0x1);
    pthread_join(iv_thread, 0); // wait for thread to exit, ignore return value

    assert(iv_dispatcher && "error, dispatcher object not created");
    iv_dispatcher->leave();
  }
}

void* client_thread(void *arg)
{
  precondition(arg);

  Kinetophone_dbus_client_recorder* recorder = static_cast<Kinetophone_dbus_client_recorder*>(arg);

  bool have_error;
  Error::Error_dictionary error;

  start_error_block();

  try
  {
    handler_error = 0;

    recorder->enable_RMS_metering(true);
    recorder->set_metering_channels(1);

    // start up the recorder with default settings
    map<string, string> settings_map;
    // change settings here before recording
    // e.g. settings_map["input_device"] = "3";
    recorder->startup(settings_map,
                      have_error,
                      error);
    if (have_error)
    {
      goto error_handler;
    }
    cout << "Started recorder." << endl;

    map<string, string> current_settings_map;
    recorder->settings(current_settings_map, have_error, error);
    if (have_error)
    {
      goto error_handler;
    }
    cout << endl << "Settings for recording: " << endl;
    for (map<string, string>::const_iterator i = current_settings_map.begin();
         i != current_settings_map.end();
         ++i)
    {
      cout << "  " << i->first << ": " << i->second << endl;
    }
    cout << endl;

    int index = 0;
    // record for ~5 seconds, with new indexes at 1 second intervals
    recorder->record_with_index(index,
                                have_error,
                                error);
    if (have_error)
    {
      goto error_handler;
    }
    cout << "Recording at index 0." << endl;;

    const int sleep_msec = 1000 / 30; // update 30 times/second
    for (int i = 0; i < (5 * 30); ++i)
    {
      if (__sync_fetch_and_add(&iv_exit_thread, 0x0))
      {
        break;
      }
      string recorded_time;
      string index_recorded_time;
      vector<double> levels;
      Long_int overflow_count;

      levels.clear();
      // get update info
      recorder->values_for_update(recorded_time,
                                  index_recorded_time,
                                  levels,
                                  overflow_count,
                                  have_error,
                                  error);
      if (have_error)
      {
        goto error_handler;
      }

      cout << "Time: " << recorded_time << " Ch. 0 level: " << levels[0];
      cout << " Overflow: " << overflow_count <<  endl;
      if (0 == (i % 30))
      {
        ++index;
        cout << "Changing index to: " << index << endl;
        recorder->new_index(index, have_error, error); // ignore error
      }
      Pa_Sleep(sleep_msec);
    }

    // stop recording
    recorder->stop(have_error, error);
    if (have_error)
    {
      goto error_handler;
    }
    cout << "Recorder stopped." << endl;;

    // "provoke" a recoverable error
    recorder->stop(have_error, error);
    if (have_error)
    {
      if ("kinetophone_error_record" == error[Error::code_string_error_key])
      {
        cout << endl << "RECOVERABLE ERROR:" << endl;
        cout << "Error: " << error[Error::brief_description_error_key] << endl;
        cout << "Code: " << error[Error::code_number_error_key];
        cout << " (" << error[Error::code_string_error_key] << ")" << endl;
        cout << "Details: " << error[Error::detailed_description_error_key] << endl;
        cout << "Recovery suggestion: " << error[Error::recovery_description_error_key] << endl;
        cout << "CONTINUING..." << endl << endl;
      }
      else
      {
        goto error_handler;
      }
    }

    // get final recorded time
    string recorded_time = recorder->recorded_time(':');
    cout << "Recorded time: " << recorded_time << endl;;

    // see where the temporary output sound file is
    string output_file = recorder->output_file();
    cout << "Output file: " << output_file << endl;;

    string segments_xml;
    recorder->segments(segments_xml,
                       have_error,
                       error);
    if (have_error)
    {
      goto error_handler;
    }
    cout << "Segments XML: " << endl << segments_xml << endl;;
  }
  catch (const DBus::Error& e)
  {
    const Kinetophone_error* client_error = new Kinetophone_error(error_location(),
                                                                  Kinetophone_error::client,
                                                                  new DBus_error(error_location(), e));
    client_error->format_chain_as_dictionary(error);
    delete client_error;
    goto error_handler;
  }

  goto exit_point;

  end_error_block();

 error_handler:
  cout << "Error: " << error[Error::brief_description_error_key] << endl;
  cout << "Code: " << error[Error::code_number_error_key];
  cout << " (" << error[Error::code_string_error_key] << ")" << endl;
  cout << "Details: " << error[Error::detailed_description_error_key] << endl;
  cout << "Recovery suggestion: " << error[Error::recovery_description_error_key] << endl;
  goto error_cleanup;

 error_cleanup:
  goto exit_point;

 exit_point:
  iv_dispatcher->leave();
  pthread_exit(0);
}

//////////
// main //
//////////

int main()
{
  bool have_recorder = false;
  bool have_error;
  Error::Error_dictionary error;
  string recorder_path;
  Kinetophone_dbus_client* client = 0;
  int return_value = 1;

  start_error_block();

  try
  {
    iv_dispatcher = new BusDispatcher;
  }
  catch (const DBus::Error& e)
  {
    cerr << "Exception encountered creating bus dispatcher: " << e.what() << endl;;
    goto exit_point;
  }

  // where error descriptions are found
  Error::set_error_directory("../../data");

  signal(SIGTERM, ih_signal_handler);
  signal(SIGINT, ih_signal_handler);

  try
  {
    DBus::default_dispatcher = iv_dispatcher;
    DBus::_init_threading();

    // connect to server
    Connection connection = Connection::SessionBus();
    client = new Kinetophone_dbus_client(connection,
                                         ic_kinetophone_dbus_server_path.c_str(),
                                         ic_kinetophone_dbus_server_name.c_str());
    // create a new recorder
    client->create_recorder(recorder_path,
                            have_error,
                            error);
    if (have_error)
    {
      goto error_handler;
    }
    have_recorder = true;
    cout << "Created recorder, path: " << recorder_path << endl;

    vector<string> recorders;
    client->list_recorders(recorders,
                           have_error,
                           error);
    if (have_error)
    {
      goto error_handler;
    }
    cout << "Currently available recorders:" << endl;
    for (vector<string>::const_iterator i = recorders.begin(); i != recorders.end(); ++i)
    {
      cout << *i << endl;
    }

    Kinetophone_dbus_client_recorder recorder(connection,
                                              recorder_path.c_str(),
                                              ic_kinetophone_dbus_server_name.c_str());
    // get a list of available devices
    string device_list;
    recorder.list_devices(device_list,
                          have_error,
                          error);
    if (have_error)
    {
      goto error_handler;
    }
    cout << "Available devices:" << endl;
    cout << device_list << endl;

    int posix_error_code = pthread_create(&iv_thread, 0, client_thread, &recorder);
    on_error_with_label(posix_error_code,
                        posix_error_handler,
                        new Posix_error(posix_error_code,
                                        "pthread_create",
                                        0));

    iv_dispatcher->enter();

    // tell the server we are done with the recorder
    client->remove_recorder(recorder_path,
                            have_error,
                            error);
    if (have_error)
    {
      goto error_handler;
    }
    cout << "Recorder removed." << endl;;

  }
  catch (const DBus::Error& dbus_error)
  {
    const Kinetophone_error* client_error = new Kinetophone_error(error_location(),
                                                                  Kinetophone_error::client,
                                                                  new DBus_error(error_location(),
                                                                                 dbus_error));
    client_error->format_chain_as_dictionary(error);
    delete client_error;
    goto error_handler;
  }

  goto exit_point;

  end_error_block();

 posix_error_handler:
  cerr << "Error creating thread: ";
  cerr << handler_error->error_dictionary()[Error::diagnostic_error_key] << endl;
  goto error_cleanup;

 error_handler:
  cout << "Error: " << error[Error::brief_description_error_key] << endl;
  cout << "Code: " << error[Error::code_number_error_key];
  cout << " (" << error[Error::code_string_error_key] << ")" << endl;
  cout << "Details: " << error[Error::detailed_description_error_key] << endl;
  cout << "Recovery suggestion: " << error[Error::recovery_description_error_key] << endl;
  goto error_cleanup;

 error_cleanup:
  if (client && have_recorder)
  {
    try
    {
      client->remove_recorder(recorder_path,
                              have_error,
                              error);
    }
    catch (const DBus::Error& dbus_error)
    {
      // do nothing
    }
  }
  goto exit_point;

 exit_point:
  delete client;

  return return_value;
}
