// Kinetophone_console_controller.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Kinetophone_console_controller.hpp"
#include "../base/Kinetophone_model.hpp"
#include "Kinetophone_console_view.hpp"
#include "../base/Sound_recorder.hpp"
#include "../base/Sound_recorder_config.hpp"
#include "../base/Sound_file_config.hpp"
#include "../base/error/Kinetophone_error.hpp"
#include <ncurses.h>

using namespace Roan_trail::Kinetophone;

//
// Constructor/destructor/copy
//

Kinetophone_console_controller::Kinetophone_console_controller(Kinetophone_model& model,
                                                               Kinetophone_console_view& view)
  : Console_controller(),
    m_model(model),
    m_view(view),
    m_quit_on_confirm(false)
{
  // make a singleton by convention
  precondition(!m_instance);

  m_instance = this;

  Sound_recorder* recorder = m_model.sound_recorder();
  if (m_view.has_level_meter())
  {
    recorder->set_metering_channels(recorder->config()->sound_file->channels);
    recorder->enable_RMS_metering(true);
  }

  postcondition(mf_invariant(false));
}

//
// Public member functions
//

bool Kinetophone_console_controller::process_input(bool& return_done, Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  return_done = false;
  bool command_completed = true;
  Error_param error;

  start_error_block();

  const int input = getch(); // returns ERR if no input available
  if (ERR == input)
  {
    return_value = true;
    goto exit_point;
  }

  Sound_recorder* sound_recorder = m_model.sound_recorder();
  const int state = sound_recorder->state();

  if (m_view.confirm_overwrite())
  {
    m_view.set_confirm_overwrite(false);
    if (('Y' == input) || ('y' == input))
    { // yes, confirm
      command_completed = sound_recorder->record(error);
      on_error(!command_completed, new Kinetophone_error(error_location(),
                                                         Kinetophone_error::record,
                                                         error()));
    }
    m_quit_on_confirm = false;
  }
  else if (m_view.confirm_stop())
  {
    m_view.set_confirm_stop(false);
    if (('Y' == input) || ('y' == input))
    { // yes, confirm
      command_completed = sound_recorder->stop(error);
      on_error(!command_completed, new Kinetophone_error(error_location(),
                                                         Kinetophone_error::record,
                                                         error()));
      if (m_quit_on_confirm)
      {
        return_done = true;
      }
    }
    m_quit_on_confirm = false;
  }
  else
  {
    switch (input)
    {
    case 'q': // Quit
    case 'Q':
      // confirm stop while recording or paused
      if (Roan_trail::Recorder::Recorder::state_stopped != state)
      {
        m_view.set_confirm_stop(true);
        m_quit_on_confirm = true;
      }
      else
      {
        return_done = true;
      }
      break;
    case 's': // Stop
    case 'S':
      if (Roan_trail::Recorder::Recorder::state_stopped == state)
      {
        break;
      }
      // confirm stop
      m_view.set_confirm_stop(true);
      // see if we need to quit on stop (if no overwrite allowed)
      m_quit_on_confirm = (!sound_recorder->config()->file_overwrite
                           && sound_recorder->have_recording());
      break;
    case 'p': // Pause
    case 'P':
      if (Roan_trail::Recorder::Recorder::state_recording != state)
      {
        break;
      }
      command_completed = sound_recorder->pause(error);
      on_error(!command_completed, new Kinetophone_error(error_location(),
                                                         Kinetophone_error::record,
                                                         error()));
      break;
    case 'm': // Mute
    case 'M':
      command_completed = sound_recorder->mute(error);
      on_error(!command_completed, new Kinetophone_error(error_location(),
                                                         Kinetophone_error::record,
                                                         error()));
      break;
    case 'l': // Toggle level (RMS or peak)
    case 'L':
      if (m_view.has_level_meter())
      {
        if (m_view.level_meter_peak_mode())
        {
          sound_recorder->enable_peak_metering(false);
          sound_recorder->enable_RMS_metering(true);
          m_view.set_level_meter_peak_mode(false);
        }
        else
        {
          sound_recorder->enable_RMS_metering(false);
          sound_recorder->enable_peak_metering(true);
          m_view.set_level_meter_peak_mode(true);
        }
        m_view.resize();
      }
      break;
    case 'r': // Record
    case 'R':
      if (Roan_trail::Recorder::Recorder::state_paused == state)
      {
        command_completed = sound_recorder->record(error);
        on_error(!command_completed, new Kinetophone_error(error_location(),
                                                           Kinetophone_error::record,
                                                           error()));
      }
      else if (Roan_trail::Recorder::Recorder::state_stopped == state)
      {
        // see if we need to confirm overwrite (after first recording)
        if (sound_recorder->have_recording())
        {
          m_view.set_confirm_overwrite(true);
        }
        else
        {
          command_completed = sound_recorder->record(error);
          on_error(!command_completed, new Kinetophone_error(error_location(),
                                                             Kinetophone_error::record,
                                                             error()));
        }
      }
      // no final else needed, ignore if state is recording
      break;
    case 'C': // Clear overflow counts, clipping
    case 'c':
      m_view.clear();
      m_model.clear();
      break;
    default:
      break;
    }
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition((return_value || !return_error.need_error() || return_error())
                && (!return_value || command_completed)
                && mf_invariant());
  return return_value;
}

//
// Protected member functions
//

bool Kinetophone_console_controller::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  return true;
}

//
// Private data members

Kinetophone_console_controller* Kinetophone_console_controller::m_instance = 0;
