// Console_level_meter_view.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

//
// This code was based on jackmeter.c:
//
//   jackmeter.c
//   Simple console based Digital Peak Meter for JACK
//   Copyright (C) 2005  Nicholas J. Humfrey
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "Console_level_meter_view.hpp"
#include "../base/common.hpp"
#include "../base/error/Error.hpp"
#include <cstring>
#include <cstdio> // Standards Rule 22 deviation (check_code_ignore)
#include <iostream>
#include <sstream>
#include <iomanip>

using std::endl;
using std::setfill;
using std::setw;
using std::stringstream;
using namespace Roan_trail;

//
// Internal helper
//

namespace
{
  const char ic_bar_element = '=';
  const char ic_peak_indicator = 'I';
  const char ic_clipped_indicator = '+';
  //
  const int ic_width_adjustment = 7;
  const int ic_print_level_adjustment = 9;
  const int ic_level_width = 4;
  const int ic_mark_width = 4;

  int ih_adjusted_width(int width, bool print_level)
  {
    int min_width = ic_width_adjustment + 1;
    if (print_level)
    {
      min_width += ic_print_level_adjustment;
    }

    return max(min_width, width);
  }
}

//
// Constructor/destructor/copy

Console_level_meter_view::Console_level_meter_view(int channels,
                                                   int rate,
                                                   int width,
                                                   bool print_level,
                                                   bool model_ballistics,
                                                   int attack_period,
                                                   int decay_period,
                                                   int peak_hold_period)
  : Level_meter_view(channels,
                     rate,
                     false,
                     false,
                     model_ballistics,
                     (attack_period < 1 ? Level_meter_view::default_attack_period : attack_period),
                     (decay_period < 1 ? Level_meter_view::default_decay_period : decay_period),
                     (peak_hold_period < 1 ? Level_meter_view::default_peak_hold_period : peak_hold_period)),
    m_width(width),
    m_print_level(print_level),
    m_meter(),
    m_scale(),
    m_scale_label()
{
  precondition((channels >=1)
               && (rate >= 1)
               && (width >= 1));

  m_width = ih_adjusted_width(width, m_print_level);

  postcondition(mf_invariant(false));
}

Console_level_meter_view::~Console_level_meter_view()
{
  precondition(mf_invariant(false));
}

//
// View update
//

bool Console_level_meter_view::update(Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  static_cast<void>(return_error); // avoid unused warning

  mf_set_meter();

  postcondition(!return_error()
                && mf_invariant());

  return true;
}

//
// Mutators
//

void Console_level_meter_view::set_width(int width)
{
  precondition((width > 0)
               && mf_invariant());

  m_width = ih_adjusted_width(width, m_print_level);

  mf_set_scale();

  postcondition(mf_invariant());
}

//
// Protected member functions
//

bool Console_level_meter_view::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (m_width < ih_adjusted_width(0, m_print_level))
  {
    goto exit_point;
  }
  else
  {
    return_value = true;
  }

 exit_point:

  return return_value;
}

//
// Private member functions
//

void Console_level_meter_view::mf_set_meter()
{
  int width = m_width - ic_width_adjustment;
  if (m_print_level)
  {
    width -= ic_print_level_adjustment;
  }

  Level_meter_view::update();

  stringstream meter_stream;
  const size_t number_channels = static_cast<size_t>(Level_meter_view::channels());
  for (size_t channel = 0; channel < number_channels; ++channel)
  {
    meter_stream << setw(2) << setfill('0') << channel % 100 << " ";
    const double scaled_level = Level_meter_view::scaled_level(channel);
    const int size = scaled_level * static_cast<double>(width);
    const int current_peak = Level_meter_view::peak(channel) * static_cast<double>(width);
    int i;
    for (i = 1; i < size; ++i)
    {
      meter_stream << ic_bar_element;
    }
    if (i == current_peak)
    {
      meter_stream << ic_peak_indicator;
    }
    else
    {
      meter_stream << ic_bar_element;
    }
    i++;
    while (i < current_peak)
    {
      meter_stream << " ";
      ++i;
    }
    if (i == current_peak)
    {
      meter_stream << ic_peak_indicator;
      ++i;
    }
    while (i <= width)
    {
      meter_stream << " ";
      ++i;
    }
    if (Level_meter_view::clipped(channel))
    {
      meter_stream << ic_clipped_indicator;
    }
    else
    {
      meter_stream << " ";
    }
    if (m_print_level)
    {
      const double dB = Level_meter_view::convert_level_to_dB(Level_meter_view::level(channel));
      meter_stream.setf(std::ios_base::fixed, std::ios_base::floatfield);
      meter_stream.precision(0);
      meter_stream.fill(' ');
      meter_stream << '[' << std::setw(ic_level_width) << dB << ']';
    }

    meter_stream << endl;
  }

  m_meter = meter_stream.str();
}

void Console_level_meter_view::mf_set_scale()
{
  const int marks[] = { 0, -5, -10, -15, -20, -25, -30, -35, -40, -50, -60 };
  int width = m_width - ic_width_adjustment;
  if (m_print_level)
  {
    width -= ic_print_level_adjustment;
  }
  string scale(static_cast<string::size_type>(width), ' ');
  string line(static_cast<string::size_type>(width), '_');

  // 'draw' on each of the db marks
  for (int i = 0; i < static_cast<int>(sizeof(marks) / sizeof(int)); i++)
  {
    int pos = scale_IEC(marks[i]) * width;
    if (pos > 0)
    {
      --pos;
    }
    // create string of the db value
    stringstream mark;
    mark << std::setw(ic_mark_width) << marks[i];
    // position the label string
    const string& mark_str = mark.str();
    int slen = static_cast<int>(mark_str.length());
    int spos = (pos < (slen / 2)) ? 0 : pos - (slen / 2);
    if (width >= slen)
    {
      if ((spos + slen) > width)
      {
        spos = width - slen;
      }
      scale.replace(static_cast<string::size_type>(spos),
                    static_cast<string::size_type>(slen), mark_str);
    }

    // add scale "ticks"
    line[static_cast<string::size_type>(pos)] = '|';
  }

  // update data members
  stringstream scale_label_stream;
  scale_label_stream << "   " << scale << " dB  ";
  m_scale_label = scale_label_stream.str();

  stringstream scale_stream;
  scale_stream << "CH " << line << (peak_meter_mode() ? " Peak" : " RMS ");
  m_scale = scale_stream.str();
}
