// Error.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef ERROR_HPP_
#define ERROR_HPP_

#include <string>
#include <iostream>
#include <map>
#include <vector>

using std::string;
using std::ostream;
using std::vector;

namespace Roan_trail
{
  // Error is an abstract class, intended to be used as the root of
  // an error hierarchy.  Each subclass represents the errors for a subsystem.
  //
  // Subclasses of Error which are intended to be user level (presentation)
  // should override mf_is_user_error() to return true when appropriate.
  //
  class Error
  {
  public:
    // constructor/destructor
    Error(const char* file,
          const char* function,
          int line,
          int code,
          const Error* base_error = 0);
    Error(int code, const Error* base_error = 0);
    Error(const Error& error);
    Error& operator=(const Error& error);
    virtual Error *clone() const = 0;
    virtual ~Error();
    // operators
    bool operator==(const Error& error) const;
    bool operator!=(const Error& error) const;
    // typedefs
    typedef std::map<int, string> Error_code_string_map;
    typedef std::map<string, string> Error_dictionary;
    // friends
    friend ostream& operator<<(ostream& s, const Error& e);
    // accessors
    const string& file() const { return m_file; }
    int line() const { return m_line; }
    int code() const { return m_code; }
    const Error *base_error() const { return m_base_error; }
    Error_dictionary& error_dictionary() { return m_error_dictionary; }
    const Error_dictionary& error_dictionary() const { return m_error_dictionary; }
    // mutators
    void set_base_error(const Error* base_error);
    static void set_error_directory(const string& dir_path) { m_error_directory = dir_path; }
    // error keys
    const static string code_string_error_key;
    const static string code_number_error_key;
    const static string diagnostic_error_key;
    const static string file_path_error_key;
    const static string to_path_error_key;
    const static string brief_description_error_key;
    const static string detailed_description_error_key;
    const static string recovery_description_error_key;
    const static string error_dump_error_key;
    // other
    virtual string error_class() const = 0;
    // error chain (list of current error and base errors)
    void format_chain_as_dictionary(Error_dictionary& return_dictionary) const;
    const string& dictionary_entry_for_chain(const string& key) const;
    const Error* user_error_for_chain() const;
    int format_message_for_chain(bool use_details, string& return_message); // returns error code
  protected:
    // invariant check
    bool mf_invariant(bool check_base_class = true) const;
    virtual bool mf_is_user_error() const { return false; }
    void mf_set_descriptions_from_file(const string& file);
  private:
    Error_dictionary m_error_dictionary;
    const string m_file;
    const string m_function;
    const int m_line;
    const int m_code;
    // the error object "owns" its base error
    // and is responsible for deleting it
    const Error* m_base_error;
    //
    static string m_error_directory;
    //
    void mf_set_error_code_number(int code);
  };

  // Error_param is a class that wraps an Error object for passing as a return (reference) parameter,
  // to avoid passing a pointer to a pointer.  The Error_param class does not manage the pointer's
  // memory.
  //
  // Setting the need_error parameter to false in the constructor allows the caller to specify
  // that the error will be ignored.
  //
  // The () operator retrieves the underlying error.

  class Error_param
  {
  public:
    // constructor/destructor
    explicit Error_param(bool need_error = true);
    Error_param& operator=(Error* error);
    // operators
    operator bool() const { return (m_error != 0); }
    Error* operator()() { return m_error; }
    const Error* operator()() const { return m_error; }
    // accessors
    bool need_error() const { return m_need_error; }
  protected:
    // invariant check
    bool mf_invariant(bool check_base_class = true) const;
  private:
    bool m_need_error;
    Error* m_error;
  };

  // operators
  ostream& operator<<(ostream& s, const Error& e);


  // Error macros (for convenience and to shrink visible code for error checking

  // introduces an anonymous block for error handling
#define start_error_block()                     \
  Roan_trail::Error* handler_error = 0;         \
  {

#define end_error_block()                       \
  }

#define on_error(cond, error)                   \
  if (cond)                                     \
  {                                             \
    handler_error = (error);                    \
    goto error_handler;                         \
  }

#define on_error_with_label(cond, label, error) \
  if (cond)                                     \
  {                                             \
    handler_error = (error);                    \
    goto label;                                 \
  }

#define default_error_handler(return_error) \
  error_handler:                            \
  if (return_error.need_error())            \
  {                                         \
    return_error = handler_error;           \
  }                                         \
  goto error_cleanup;

#define default_error_handler_and_cleanup(return_error, return_var, error_return_value) \
  error_handler:                                                                        \
  if (return_error.need_error())                                                        \
  {                                                                                     \
    return_error = handler_error;                                                       \
  }                                                                                     \
  goto error_cleanup;                                                                   \
                                                                                        \
 error_cleanup:                                                                         \
  if (!return_error.need_error())                                                       \
  {                                                                                     \
    delete handler_error;                                                               \
  }                                                                                     \
  return_var = (error_return_value);

#define default_error_cleanup(return_error, return_var, error_return_value) \
 error_cleanup:                                                             \
  if (!return_error.need_error())                                           \
  {                                                                         \
    delete handler_error;                                                   \
  }                                                                         \
  return_var = (error_return_value);

#define error_location() __FILE__, __PRETTY_FUNCTION__, __LINE__
}

#endif // ERROR_HPP_
