// Build_error.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Build_error.hpp"
#include "../common.hpp"
#include <sstream>

using std::stringstream;
using std::endl;
using namespace Roan_trail::Builder;

//
// Constructor/destructor/copy
//

Build_error::Build_error(const char* file,
                         const char* function,
                         int line,
                         int code,
                         const Error* base_error)
  : Error(file,
          function,
          line,
          code,
          base_error)
{
  mf_set_diagnostic("");

  postcondition(mf_invariant(false));
}

Build_error::Build_error(const char* file,
                         const char* function,
                         int line,
                         int code,
                         const string& file_path,
                         const string& to_path,
                         const string& diagnostic,
                         const Error* base_error)
  : Error(file,
          function,
          line,
          code,
          base_error)
{
  if ("" != file_path)
  {
    error_dictionary()[Error::file_path_error_key] = file_path;
  }

  if ("" != to_path)
  {
    error_dictionary()[Error::to_path_error_key] = to_path;
  }

  mf_set_diagnostic(diagnostic);

  postcondition(mf_invariant(false));
}

//
// Error codes
//

const int Build_error::none;
const int Build_error::general;
// pre-build checks
const int Build_error::prebuild_output_dir_access;
const int Build_error::prebuild_insuff_output_space;
const int Build_error::prebuild_movie_exists_overwrite;
const int Build_error::prebuild_movie_exists_no_overwrite;
const int Build_error::prebuild_insuff_temp_space;
// sound
//   source
const int Build_error::source_sound_file_read_alignment;
const int Build_error::source_sound_file_open;
const int Build_error::source_sound_file_read;
const int Build_error::source_sound_file_convert;
const int Build_error::source_sound_file_convert_format;
const int Build_error::source_sound_file_close;
//   temporary
const int Build_error::temporary_sound_file_create;
const int Build_error::temporary_sound_file_initialize;
const int Build_error::temporary_sound_file_write;
const int Build_error::temporary_sound_file_close;
const int Build_error::temporary_sound_file_remove;
// movie
//   temporary
const int Build_error::temporary_movie_create;
const int Build_error::temporary_movie_write;
const int Build_error::temporary_movie_close;
const int Build_error::temporary_movie_remove;
//   destination
const int Build_error::destination_movie_remove;
const int Build_error::move_temporary_to_destination_movie;
const int Build_error::destination_movie_change_permissions;
// image source
const int Build_error::image_source_not_available;
// misc
const int Build_error::setup;

//
// Protected member functions
//

bool Build_error::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if ((code() < Build_error::none) || (code() > Build_error::setup))
  {
    goto exit_point;
  }

  if (!m_default_diagnostic_map.size())
  {
    goto exit_point;
  }

  return_value = (!check_base_class || Error::mf_invariant(check_base_class));
  goto exit_point;

 exit_point:

  return return_value;
}

//
// Private class data members
//

map<int,string> Build_error::m_default_diagnostic_map;

//
// Private class member functions
//

void Build_error::mf_setup_default_diagnostic_map()
{
  m_default_diagnostic_map[general] = "general build error";
  // pre-build checks
  //   output directory
  m_default_diagnostic_map[prebuild_output_dir_access] =
    "the output movie directory is not accessible";
  m_default_diagnostic_map[prebuild_insuff_output_space] =
    "there is insufficient space in the output directory";
  //   output movies
  m_default_diagnostic_map[prebuild_movie_exists_overwrite] =
    "one or more output movies exist and WILL be overwritten";
  m_default_diagnostic_map[prebuild_movie_exists_no_overwrite] =
    "one or more output movies exist and will NOT be overwritten";
  //   temporary directory
  m_default_diagnostic_map[prebuild_insuff_temp_space] =
    "there is insufficent space in the temporary directory";
  // sound
  //   source
  m_default_diagnostic_map[source_sound_file_read_alignment] =
    "there was a problem with the alignment of the source sound file";
  m_default_diagnostic_map[source_sound_file_open] =
    "error opening the source sound file";
  m_default_diagnostic_map[source_sound_file_read] =
    "error reading the source sound file";
  m_default_diagnostic_map[source_sound_file_convert] =
    "error converting the source sound file";
  m_default_diagnostic_map[source_sound_file_convert_format] =
    "error converting the format of the source sound file";
  m_default_diagnostic_map[source_sound_file_close] =
    "error closing the source sound file";
  //   temporary
  m_default_diagnostic_map[temporary_sound_file_create] =
    "error opening the temporary sound file";
  m_default_diagnostic_map[temporary_sound_file_initialize] =
    "error initializing the temporary sound file";
  m_default_diagnostic_map[temporary_sound_file_write] =
    "error writing to the temporary sound file";
  m_default_diagnostic_map[temporary_sound_file_close] =
    "error closing the temporary sound file";
  m_default_diagnostic_map[temporary_sound_file_remove] =
    "error removing the temporary sound file";
  // movie
  //   temporary
  m_default_diagnostic_map[temporary_movie_create] =
    "error creating the temporary movie";
  m_default_diagnostic_map[temporary_movie_write] =
    "error writing to the temporary movie";
  m_default_diagnostic_map[temporary_movie_close] =
    "error closing the temporary movie";
  m_default_diagnostic_map[temporary_movie_remove] =
    "error removing the temporary movie";
  //   destination
  m_default_diagnostic_map[destination_movie_remove] =
    "error removing the destination movie";
  m_default_diagnostic_map[move_temporary_to_destination_movie] =
    "error moving the temporary movie to its destination";
  m_default_diagnostic_map[destination_movie_change_permissions] =
    "error changing permissions on the destination movie";
  // image source
  m_default_diagnostic_map[image_source_not_available] =
    "the source is not available";
  // misc
  m_default_diagnostic_map[setup] =
    "error setting up for build";
}

void Build_error::mf_set_diagnostic(const string& diagnostic)
{
  if (!m_default_diagnostic_map.size())
  {
    mf_setup_default_diagnostic_map();
  }

  string current_diagnostic;
  if (Build_error::general != code())
  {
    string default_diagnostic;
    map<int, string>::const_iterator p = m_default_diagnostic_map.find(code());
    if (p != m_default_diagnostic_map.end())
    {
      default_diagnostic = p->second;
    }

    string base_diagnostic;
    const Error* base = base_error();
    if (base)
    {
      Error_dictionary::const_iterator p = base->error_dictionary().find(Error::diagnostic_error_key);
      if (p != base->error_dictionary().end())
      {
        base_diagnostic = p->second;
      }
    }

    if (("" != default_diagnostic) && ("" != base_diagnostic))
    {
      current_diagnostic = default_diagnostic + string(":  ") + base_diagnostic;
    }
    else if ("" != default_diagnostic)
    {
      current_diagnostic = default_diagnostic;
    }
    else if ("" != base_diagnostic)
    {
      current_diagnostic = base_diagnostic;
    }
    // no final else needed
  }

  if ("" != diagnostic)
  {
    if ("" != current_diagnostic)
    {
      stringstream error_stream;
      error_stream << current_diagnostic << endl << string("(") + diagnostic + string(")");
      error_dictionary()[Error::diagnostic_error_key] = error_stream.str();
    }
    else
    {
      error_dictionary()[Error::diagnostic_error_key] = diagnostic;
    }
  }
  else
  {
    if ("" != current_diagnostic)
    {
      error_dictionary()[Error::diagnostic_error_key] = current_diagnostic;
    }
  }
}
