// Sound_file_config.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

// PORTING: uses "memcpy" to alias
// PORTING: look at assembler output to make sure memcpy is not
// PORTING: called as a function, but is optimized to a move
// PORTING: (currently gcc with -O2 correctly optimizes)
// PORTING: (use "objdump -S <obj_file>" where obj_file has been
// PORTING:  compiled with -g)

#include "Sound_file_config.hpp"
#include "common.hpp"
#include <sndfile.h>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>
#include <sstream>
#include <cmath>
#include <cstring>

using std::stringstream;
using std::endl;
using boost::lexical_cast;
using boost::algorithm::to_lower;
using boost::algorithm::to_upper;
using namespace Roan_trail::Recorder;

//
// Internal helpers
//

namespace
{
  bool ih_config_value_for_string(const string& config_string,
                                  int &return_config_value,
                                  const Sound_file_config::Constant_map& config_map)
  {
    bool return_value = false;

    string to_find = config_string;
    to_lower(to_find);
    Sound_file_config::Constant_map::const_iterator p = config_map.begin();
    while (p != config_map.end())
    {
      if (p->second == to_find)
      {
        return_config_value = p->first;

        return_value = true;
        goto exit_point;
      }
      p++;
    }

    return_value = false;

  exit_point:

    return return_value;
  }

  string ih_string_for_config_value(int config_value,
                                    const Sound_file_config::Constant_map& config_map)
  {
    string return_value = "";

    Sound_file_config::Constant_map::const_iterator p = config_map.find(config_value);
    if (p != config_map.end())
    {
      return_value = p->second;
    }

    return return_value;
  }
}

//
// Constructor/destructor
//

Sound_file_config::Sound_file_config()
  // setup some reasonable defaults
  : channels(default_channels),
    sample_rate(default_sample_rate),
    VBR_quality(default_VBR_quality),
    file_name(),
    file_type(default_file_type),
    data_format(default_data_format),
    endianness(default_endianness)
{
  if (!m_file_type_map.size() || !m_data_format_map.size() || !m_endianness_map.size())
  {
    mf_setup_constant_maps();
  }
}

//
// Operators
//

bool Sound_file_config::operator==(const Sound_file_config& config) const
{
  // assumption that sample rate is an integral value, despite having a floating point type
  assert(equal_fp(sample_rate, round(sample_rate))
         && equal_fp(config.sample_rate, round(config.sample_rate))
         && "error, sample rate does not have integral value");

  bool eq = (config.channels == channels)
    && (static_cast<int>(config.sample_rate) == static_cast<int>(sample_rate))
    && (Roan_trail::equal_fp(config.VBR_quality, VBR_quality))
    && (config.file_name == file_name)
    && (config.file_type == file_type)
    && (config.data_format == data_format)
    && (config.endianness == endianness);

  return eq;
}

bool Sound_file_config::operator!=(const Sound_file_config& config) const
{
  // operator== checks precondition
  return !operator==(config);
}

//
// Helpers
//

//
// ... file type
//

string Sound_file_config::string_for_file_type(File_type file)
{
  return ih_string_for_config_value(file, m_file_type_map);
}

bool Sound_file_config::file_type_for_string(const string& file_type_string, File_type& return_file_type)
{
  return ih_config_value_for_string(file_type_string,
                                    return_file_type,
                                    m_file_type_map);
}

//
//
// ... data format
//

string Sound_file_config::string_for_data_format(Data_format format)
{
  return ih_string_for_config_value(format, m_data_format_map);
}

bool Sound_file_config::data_format_for_string(const string& data_format_string, Data_format& return_data_format)
{
  return ih_config_value_for_string(data_format_string,
                                    return_data_format,
                                    m_data_format_map);
}

bool Sound_file_config::sample_size_for_data_format(Data_format format, size_t& return_size)
{
  bool return_value = false;

  map<Data_format, size_t>::const_iterator p = m_data_format_bytes_map.find(format);
  if (p != m_data_format_bytes_map.end())
  {
    return_size = p->second;
    return_value = true;
  }

  return return_value;
}

//
// ... endianness
//

string Sound_file_config::string_for_endianness(Endianness endian)
{
  return ih_string_for_config_value(endian, m_endianness_map);
}

bool Sound_file_config::endianness_for_string(const string& endianness_string, Endianness& return_endian)
{
  return ih_config_value_for_string(endianness_string,
                                    return_endian,
                                    m_endianness_map);
}

Sound_file_config::Endianness Sound_file_config::endianness_for_CPU()
{
  Endianness return_value = endian_little;

  uint32_t value = 0x1BADD00D;
  uint8_t convert_value[4];
  memcpy(&convert_value,
         &value,
         sizeof(uint32_t));

  if (0x1B == convert_value[0])
  {
    return_value = endian_big;
  }
  else if (0x0D == convert_value[0])
  {
    return_value = endian_little;
  }
  else
  {
    assert(false && "unknown CPU endianness");
  }

  return return_value;
}

bool Sound_file_config::endianness_for_file_type(File_type file, Endianness& return_endian)
{
  bool return_value = false;
  map<File_type, Endianness>::const_iterator p = m_file_type_endianness_map.find(file);
  if (p != m_file_type_endianness_map.end())
  {
    return_endian = p->second;
    return_value = true;
  }

  return return_value;
}

//
// ... other
//

string Sound_file_config::format_string(const string& prefix) const
{
  const string file_type_string = string_for_file_type(file_type);
  const string data_format_string = string_for_data_format(data_format);
  const string endianness_string = string_for_endianness(endianness);
  stringstream s;
  s << prefix << "File type: " << file_type_string << endl;
  s << prefix << "Data format: " << data_format_string << endl;
  s << prefix << "Endianness: " << endianness_string << endl;
  s << prefix << "Channels: " << channels << endl;
  s << prefix << "Sample rate: " << sample_rate << endl;

  return s.str();
}

string Sound_file_config::format_string_brief() const
{
  string file_type_string = string_for_file_type(file_type);
  to_upper(file_type_string);
  string data_format_string = string_for_data_format(data_format);
  to_upper(data_format_string);
  string endianness_string = string_for_endianness(endianness);
  to_upper(endianness_string);
  stringstream s;
  s << sample_rate << " S/S | ";
  s << channels << " CH | ";
  s << file_type_string << " | ";
  s << data_format_string << " | ";
  s << endianness_string << " END";

  return s.str();
}

bool Sound_file_config::has_VBR(double& return_VBR) const
{
  bool return_value = false;

  if (data_format_vorbis == data_format) // || ...
  {
    // VBR
    if (VBR_quality > 1.0)
    {
      return_VBR = 1.0;
    }
    else
    {
      return_VBR = VBR_quality;
    }

    return_value = true;
    goto exit_point;
  }
  else
  {
    // not VBR
    return_value = false;
    goto exit_point;
  }

 exit_point:
  return return_value;
}

bool Sound_file_config::is_compressed() const
{
  bool return_value = false;

  if ((file_type_ogg == file_type)
      && (data_format_vorbis == data_format))
  {
    return_value = true;
  }

  return return_value;
}

bool Sound_file_config::from_map(const map<string, string>& config_map)
{
  bool found_error = false;

  for (map<string, string>::const_iterator i = config_map.begin(); i != config_map.end(); ++i)
  {
    try
    {
      string key = i->first;
      to_lower(key);
      string entry = i->second;
      if ("channels" == key)
      {
        channels = lexical_cast<int>(entry);
      }
      else if ("sample_rate" == key)
      {
        sample_rate = lexical_cast<double>(entry);
      }
      else if ("vbr_quality" == key)
      {
        VBR_quality = lexical_cast<double>(entry);
      }
      else if ("file_name" == key)
      {
        file_name = entry;
      }
      else if ("file_type" == key)
      {
        if (!file_type_for_string(entry, file_type))
        {
          found_error = true;
        }
      }
      else if ("data_format" == key)
      {
        if (!data_format_for_string(entry, data_format))
        {
          found_error = true;
        }
      }
      else if ("endianness" == key)
      {
        if (!endianness_for_string(entry, endianness))
        {
          found_error = true;
        }
      }
      else
      {
        found_error = true;
      }
    }
    catch (...)
    {
      found_error = true;
    }
  }

  return !found_error;
}

bool Sound_file_config::to_map(map<string, string>& return_config_map) const
{
  bool found_error = false;

  try
  {
    return_config_map["channels"] = lexical_cast<string>(channels);
  }
  catch (...)
  {
    found_error = true;
  }
  try
  {
    return_config_map["sample_rate"] = lexical_cast<string>(sample_rate);
  }
  catch (...)
  {
    found_error = true;
  }
  try
  {
    return_config_map["vbr_quality"] = lexical_cast<string>(VBR_quality);
  }
  catch (...)
  {
    found_error = true;
  }

  return_config_map["file_name"] = file_name;

  return_config_map["file_type"] = string_for_file_type(file_type);

  return_config_map["data_format"] = string_for_data_format(data_format);

  return_config_map["endianness"] = string_for_endianness(endianness);

  return !found_error;
}

//
// Class constants
//

// default constants
const int Sound_file_config::default_channels = -1;
const double Sound_file_config::default_sample_rate = 48000.0;
const double Sound_file_config::default_VBR_quality = -1.0;
const Sound_file_config::File_type Sound_file_config::default_file_type =
                             Sound_file_config::file_type_wav;
const Sound_file_config::Data_format Sound_file_config::default_data_format =
                             Sound_file_config::data_format_PCM16;
const Sound_file_config::Endianness Sound_file_config::default_endianness =
                             Sound_file_config::endian_file;

// miscellaneous constants
const double Sound_file_config::highest_VBR_quality = 1.0;

// file type constants
const int Sound_file_config::file_type_wav;
const int Sound_file_config::file_type_wavex;
const int Sound_file_config::file_type_aiff;
const int Sound_file_config::file_type_au;
const int Sound_file_config::file_type_flac;
const int Sound_file_config::file_type_ogg;

// data format constants
const int Sound_file_config::data_format_PCMS8;
const int Sound_file_config::data_format_PCM16;
const int Sound_file_config::data_format_PCM24;
const int Sound_file_config::data_format_PCM32;
const int Sound_file_config::data_format_float;
const int Sound_file_config::data_format_double;
const int Sound_file_config::data_format_vorbis;

// endianness format constants
const int Sound_file_config::endian_file;
const int Sound_file_config::endian_little;
const int Sound_file_config::endian_big;
const int Sound_file_config::endian_CPU;

//
// Private static class members
//

Sound_file_config::Constant_map Sound_file_config::m_file_type_map;
Sound_file_config::Constant_map Sound_file_config::m_data_format_map;
map<Sound_file_config::Data_format, size_t> Sound_file_config::m_data_format_bytes_map;
Sound_file_config::Constant_map Sound_file_config::m_endianness_map;
map<Sound_file_config::File_type, Sound_file_config::Endianness> Sound_file_config::m_file_type_endianness_map;

//
// Private member functions
//

void Sound_file_config::mf_setup_constant_maps()
{
  // file type
  m_file_type_map[file_type_wav] = string("wav");
  m_file_type_map[file_type_wavex] = string("wavex");
  m_file_type_map[file_type_aiff] = string("aiff");
  m_file_type_map[file_type_au] = string("au");
  m_file_type_map[file_type_flac] = string("flac");
  m_file_type_map[file_type_ogg] = string("ogg");
  // data format constants
  m_data_format_map[data_format_PCMS8] = string("pcm8");
  m_data_format_map[data_format_PCM16] = string("pcm16");
  m_data_format_map[data_format_PCM24] = string("pcm24");
  m_data_format_map[data_format_PCM32] = string("pcm32");
  m_data_format_map[data_format_float] = string("float");
  m_data_format_map[data_format_double] = string("double");
  m_data_format_map[data_format_vorbis] = string("vorbis");
  // data format constants
  m_data_format_bytes_map[data_format_PCMS8] =  1;
  m_data_format_bytes_map[data_format_PCM16] =  2;
  m_data_format_bytes_map[data_format_PCM24] =  3;
  m_data_format_bytes_map[data_format_PCM32] =  4;
  m_data_format_bytes_map[data_format_float] =  4;
  m_data_format_bytes_map[data_format_double] = 8;
  m_data_format_bytes_map[data_format_vorbis] = 0;
  // endianness constants
  m_endianness_map[endian_file] = string("file");
  m_endianness_map[endian_little] = string("little");
  m_endianness_map[endian_big] = string("big");
  m_endianness_map[endian_CPU] = string("cpu");
  // file type to endian constants (should assigned endian_big or endian_little only)
  m_file_type_endianness_map[file_type_wav] = endian_little;
  m_file_type_endianness_map[file_type_wavex] = endian_little;
  m_file_type_endianness_map[file_type_aiff] = endian_big;
  m_file_type_endianness_map[file_type_au] = endian_big;
  m_file_type_endianness_map[file_type_flac] = endian_big;
  m_file_type_endianness_map[file_type_ogg] = endian_little;
}
