// Sound_builder.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Sound_builder.hpp"
#include "Sound_file_config.hpp"
#include "Movie.hpp"
#include "Sndfile_compat.hpp"
#include "error/Build_error.hpp"
#include "error/Posix_error.hpp"
#include "error/Sndfile_error.hpp"
#include <sndfile.h>
#include <strings.h>

using Roan_trail::Long_int;
using Roan_trail::Recorder::Sndfile_error;
using Roan_trail::Recorder::Sound_file_config;
using Roan_trail::Recorder::Sndfile_compat;
using namespace Roan_trail::Builder;

//
// Internal helpers
//

namespace
{
  const Long_int ic_silence_buffer_size = 32768;
  const Long_int ic_buffer_size = 32768;
}

Sound_builder::Sound_builder()
  : m_config(new Sound_file_config),
    m_started(false),
    m_src_sound_file(0),
    m_frame_size(0),
    m_buffer(new uint8_t[ic_buffer_size]),
    m_buffer_size_in_frames(0),
    m_silence_buffer(new uint8_t[ic_silence_buffer_size]),
    m_silence_buffer_size_in_frames(0)
{
  postcondition(mf_invariant(false));
}

Sound_builder::~Sound_builder()
{
  precondition(mf_invariant(false));

  delete m_config;
  delete [] m_buffer;
  delete [] m_silence_buffer;
}

//
// Build functions
//

bool Sound_builder::start(const string& sound_file_path, Error_param& return_error)
{
  precondition(!return_error()
               && !m_started
               && mf_invariant());

  SNDFILE* source_sound_file = 0;
  bool return_value = false;

  start_error_block();

  // open the source audio file
  SF_INFO sound_file_info;
  sound_file_info.format = 0;
  source_sound_file = sf_open(sound_file_path.c_str(), SFM_READ, &sound_file_info);
  on_error(!source_sound_file, new Build_error(error_location(),
                                               Build_error::source_sound_file_open,
                                               new Sndfile_error(sf_error(0),
                                                                 "sf_open",
                                                                 sound_file_path.c_str())));
  Sound_file_config::File_type file_type;
  const bool got_file_type = Sndfile_compat::file_type_for_sf_format(sound_file_info.format, file_type);
  on_error(!got_file_type, new Build_error(error_location(),
                                           Build_error::source_sound_file_open,
                                           sound_file_path.c_str(),
                                           "",
                                           "sound file format (file type) is not supported"));
  Sound_file_config::Data_format data_format;
  const bool got_data_format = Sndfile_compat::data_format_for_sf_format(sound_file_info.format, data_format);
  on_error(!got_data_format, new Build_error(error_location(),
                                             Build_error::source_sound_file_open,
                                             sound_file_path.c_str(),
                                             "",
                                             "sound file format (data format) is not supported"));
  Sound_file_config::Endianness endian;
  const bool got_endianness = Sndfile_compat::endianness_for_sf_format(sound_file_info.format,
                                                                       true,
                                                                       endian);
  on_error(!got_endianness, new Build_error(error_location(),
                                             Build_error::source_sound_file_open,
                                             sound_file_path.c_str(),
                                             "",
                                             "sound file format (endianness) is not supported"));
  Sound_file_config config;
  config.channels = sound_file_info.channels;
  config.sample_rate = sound_file_info.samplerate;
  config.file_name = sound_file_path;
  config.file_type = file_type;
  config.data_format = data_format;
  config.endianness = endian;

  Read_function sound_read_function;
  // set the sound file read function and update the configuration
  mf_set_read_function(sound_read_function, config);

  size_t sample_size;
  const bool got_sample_size = Sound_file_config::sample_size_for_data_format(config.data_format, sample_size);
  on_error(!got_sample_size, new Build_error(error_location(),
                                             Build_error::source_sound_file_open,
                                             config.file_name.c_str(),
                                             "",
                                             "sound file format (data format) is not supported"));
  int frame_size = sample_size * config.channels;

  // success, update data members
  m_src_sound_file = source_sound_file;
  *m_config = config;
  m_frame_size = frame_size;
  m_read_function = sound_read_function;

  m_buffer_size_in_frames = ic_buffer_size / m_frame_size;
  bzero(m_silence_buffer, ic_silence_buffer_size);   // assumes signed samples
  m_silence_buffer_size_in_frames = ic_silence_buffer_size / m_frame_size;

  m_started = true;

  return_value = true;

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (source_sound_file)
  {
    sf_close(source_sound_file);
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(((return_value && m_started)
                 || (!return_value && !m_started))
                && (return_value || !return_error.need_error() || return_error())
                && mf_invariant());
  return return_value;
}

bool Sound_builder::end(Error_param& return_error)
{
  precondition(!return_error()
               && m_started
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  m_started = false;

  const int close_return = sf_close(m_src_sound_file);
  on_error(close_return, new Build_error(error_location(),
                                         Build_error::source_sound_file_close,
                                               new Sndfile_error(sf_error(0),
                                                                 "sf_close",
                                                                 m_config->file_name.c_str())));

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error, return_value, false);
  goto exit_point;

 exit_point:
  postcondition(!m_started
                && (return_value || !return_error.need_error() || return_error())
                && mf_invariant());
  return return_value;
}

//
// Sound source functions
//

const uint8_t* Sound_builder::sound(Long_int start_frame,
                                    Long_int request_frame_count,
                                    Long_int& return_frame_count,
                                    Error_param& return_error)
{
  precondition((start_frame >=0)
               && (request_frame_count >= 0)
               && !return_error()
               && m_started
               && mf_invariant());

  const uint8_t* return_value = 0;
  return_frame_count = 0;

  {
    const sf_count_t seek_count = sf_seek(m_src_sound_file, start_frame, SEEK_SET);
    if (seek_count < 0)
    {
      return_value = m_buffer;
      goto exit_point;
    }

    const Long_int frame_count = min(request_frame_count, m_buffer_size_in_frames);
    return_frame_count = m_read_function(m_src_sound_file, m_buffer, m_frame_size, frame_count);

    return_value = m_buffer;
    goto exit_point;
  }

 exit_point:
  postcondition(m_started
                && (return_frame_count >= 0)
                && (return_value || !return_error.need_error() || return_error())
                && mf_invariant());
  return return_value;
}

const uint8_t* Sound_builder::silence(Long_int request_frame_count,
                                      Long_int& return_frame_count,
                                      Error_param& return_error)
{
  precondition((request_frame_count >= 0)
               && !return_error()
               && m_started
               && mf_invariant());

  const uint8_t* return_value = 0;

  {
    return_frame_count = min(request_frame_count, m_silence_buffer_size_in_frames);

    return_value = m_silence_buffer;
    goto exit_point;
  }

 exit_point:
  postcondition(m_started
                && (return_frame_count >= 0)
                && (return_value || !return_error.need_error() || return_error())
                && mf_invariant());
  return return_value;
}

//
// Protected member functions
//

bool Sound_builder::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_config
      || !m_buffer
      || !m_silence_buffer)
  {
    goto exit_point;
  }

  if (m_started
      && (!m_read_function))
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions

void Sound_builder::mf_set_read_function(Read_function& return_function, Sound_file_config& return_config)
{
  switch (return_config.file_type)
  {
  case Sound_file_config::file_type_flac:
    switch (return_config.data_format)
    {
    case Sound_file_config::data_format_PCMS8:
    case Sound_file_config::data_format_PCM16:
      return_function = mf_read_short;
      return_config.data_format = Sound_file_config::data_format_PCM16; // tweak data format for now-raw reads
      break;
    case Sound_file_config::data_format_PCM24:
    case Sound_file_config::data_format_PCM32:
      return_function = mf_read_int;
      return_config.data_format = Sound_file_config::data_format_PCM32; // tweak data format for now-raw reads
      break;
    case Sound_file_config::data_format_float:
      return_function = mf_read_float;
      return_config.data_format = Sound_file_config::data_format_float; // tweak data format for now-raw reads
      break;
    case Sound_file_config::data_format_double:
      return_function = mf_read_double;
      return_config.data_format = Sound_file_config::data_format_double; // tweak data format for now-raw reads
      break;
    }
    // tweak endianness to CPU for non-raw reads
    return_config.endianness = Sound_file_config::endianness_for_CPU();
    break;
  case Sound_file_config::file_type_wav:
  case Sound_file_config::file_type_wavex:
  case Sound_file_config::file_type_aiff:
  case Sound_file_config::file_type_au:
    return_function = mf_read_raw;
    break;
  default:
    assert(false && "error, unsupported file type setting read function");
    break;
  }
  goto exit_point;

 exit_point:
  return;
}

Long_int Sound_builder::mf_read_raw(SNDFILE* sound_file,
                                    uint8_t* buffer,
                                    int frame_size,
                                    Long_int frames_to_read)
{
  const sf_count_t bytes_to_read = frames_to_read * frame_size;
  const sf_count_t bytes_read = sf_read_raw(sound_file,
                                            buffer,
                                            bytes_to_read);
  return bytes_read / frame_size;
}

Long_int Sound_builder::mf_read_short(SNDFILE* sound_file,
                                      uint8_t* buffer,
                                      int frame_size,
                                      Long_int frames_to_read)
{
  const sf_count_t frames_read = sf_read_short(sound_file,
                                               reinterpret_cast<short int*>(buffer),
                                               frames_to_read);
  return frames_read;
}

Long_int Sound_builder::mf_read_int(SNDFILE* sound_file,
                                    uint8_t* buffer,
                                    int frame_size,
                                    Long_int frames_to_read)
{
  const sf_count_t frames_read = sf_read_int(sound_file,
                                             reinterpret_cast<int*>(buffer),
                                             frames_to_read);

  return frames_read;
}

Long_int Sound_builder::mf_read_float(SNDFILE* sound_file,
                                      uint8_t* buffer,
                                      int frame_size,
                                      Long_int frames_to_read)
{
  const sf_count_t frames_read = sf_read_float(sound_file,
                                               reinterpret_cast<float*>(buffer),
                                               frames_to_read);
  return frames_read;
}

Long_int Sound_builder::mf_read_double(SNDFILE* sound_file,
                                       uint8_t* buffer,
                                       int frame_size,
                                       Long_int frames_to_read)
{
  const sf_count_t frames_read = sf_read_double(sound_file,
                                                reinterpret_cast<double*>(buffer),
                                                frames_to_read);
  return frames_read;
}
