// Sound_file_compat.cpp
//
// Copyright 2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Sndfile_compat.hpp"
#include "Sound_file_config.hpp"
#include <sndfile.h>

using namespace Roan_trail::Recorder;

bool Sndfile_compat::file_type_for_sf_format(int format, Sound_file_config::File_type& return_file_type)
{
  bool return_value = true;

  const int masked_format = (format & SF_FORMAT_TYPEMASK);
  switch (masked_format)
  {
  case SF_FORMAT_WAV:
    return_file_type = Sound_file_config::file_type_wav;
    break;
  case SF_FORMAT_WAVEX:
    return_file_type = Sound_file_config::file_type_wavex;
    break;
  case SF_FORMAT_AIFF:
    return_file_type = Sound_file_config::file_type_aiff;
    break;
  case SF_FORMAT_AU:
    return_file_type = Sound_file_config::file_type_au;
    break;
  case SF_FORMAT_FLAC:
    return_file_type = Sound_file_config::file_type_flac;
    break;
  case SF_FORMAT_OGG:
    return_file_type = Sound_file_config::file_type_ogg;
    break;
  default:
    return_value = false;
    break;
  }

  return return_value;
}

bool Sndfile_compat::data_format_for_sf_format(int format, Sound_file_config::Data_format& return_data_format)
{
  bool return_value = true;

  const int masked_format = (format & SF_FORMAT_SUBMASK);
  switch (masked_format)
  {
  case SF_FORMAT_PCM_S8:
    return_data_format = Sound_file_config::data_format_PCMS8;
    break;
  case SF_FORMAT_PCM_16:
    return_data_format = Sound_file_config::data_format_PCM16;
    break;
  case SF_FORMAT_PCM_24:
    return_data_format = Sound_file_config::data_format_PCM24;
    break;
  case SF_FORMAT_PCM_32:
    return_data_format = Sound_file_config::data_format_PCM32;
    break;
  case SF_FORMAT_FLOAT:
    return_data_format = Sound_file_config::data_format_float;
    break;
  case SF_FORMAT_DOUBLE:
    return_data_format = Sound_file_config::data_format_double;
    break;
  case SF_FORMAT_VORBIS:
    return_data_format = Sound_file_config::data_format_vorbis;
    break;
  default:
    return_value = false;
    break;
  }

  return return_value;
}

bool Sndfile_compat::endianness_for_sf_format(int format,
                                              bool convert,
                                              Sound_file_config::Endianness& return_endianness)
{
  bool return_value = true;

  const int masked_format = (format & SF_FORMAT_ENDMASK);
  switch (masked_format)
  {
  case SF_ENDIAN_FILE:
    return_endianness = Sound_file_config::endian_file;
    break;
  case SF_ENDIAN_LITTLE:
    return_endianness = Sound_file_config::endian_little;
    break;
  case SF_ENDIAN_BIG:
    return_endianness = Sound_file_config::endian_big;
    break;
  case SF_ENDIAN_CPU:
    return_endianness = Sound_file_config::endian_CPU;
    break;
  default:
    return_value = false;
    break;
  }

  if (return_value && convert)
  {
    // convert file/CPU endianness values to a specific big/little endianness
    switch (return_endianness)
    {
    case Sound_file_config::endian_file:
      Sound_file_config::Endianness endian;
      Sound_file_config::File_type file;
      if (!file_type_for_sf_format(format, file))
      {
        return_value = false;
      }
      else if (!Sound_file_config::endianness_for_file_type(file, endian))
      {
        return_value = false;
      }
      else
      {
        return_endianness = endian;
      }
      break;
    case Sound_file_config::endian_CPU:
      return_endianness = Sound_file_config::endianness_for_CPU();
      break;
    default:
      // no need to convert
      break;
    }
  }

  return return_value;
}

bool Sndfile_compat::sf_format_for_file_type(Sound_file_config::File_type file_type, int& return_format)
{
  bool return_value = true;

  switch (file_type)
  {
  case Sound_file_config::file_type_wav:
    return_format = SF_FORMAT_WAV;
    break;
  case Sound_file_config::file_type_wavex:
    return_format = SF_FORMAT_WAVEX;
    break;
  case Sound_file_config::file_type_aiff:
    return_format = SF_FORMAT_AIFF;
    break;
  case Sound_file_config::file_type_au:
    return_format = SF_FORMAT_AU;
    break;
  case Sound_file_config::file_type_flac:
    return_format = SF_FORMAT_FLAC;
    break;
  case Sound_file_config::file_type_ogg:
    return_format = SF_FORMAT_OGG;
    break;
  default:
    return_value = false;
    break;
  }

  return return_value;
}

bool Sndfile_compat::sf_format_for_data_format(Sound_file_config::Data_format data_format, int& return_format)
{
  bool return_value = true;

  switch (data_format)
  {
  case Sound_file_config::data_format_PCMS8:
    return_format = SF_FORMAT_PCM_S8;
    break;
  case Sound_file_config::data_format_PCM16:
    return_format = SF_FORMAT_PCM_16;
    break;
  case Sound_file_config::data_format_PCM24:
    return_format = SF_FORMAT_PCM_24;
    break;
  case Sound_file_config::data_format_PCM32:
    return_format = SF_FORMAT_PCM_32;
    break;
  case Sound_file_config::data_format_float:
    return_format = SF_FORMAT_FLOAT;
    break;
  case Sound_file_config::data_format_double:
    return_format = SF_FORMAT_DOUBLE;
    break;
  case Sound_file_config::data_format_vorbis:
    return_format = SF_FORMAT_VORBIS;
    break;
  default:
    return_value = false;
    break;
  }

  return return_value;
}

bool Sndfile_compat::sf_format_for_endianness(Sound_file_config::Endianness endian, int& return_format)
{
  bool return_value = true;

  switch (endian)
  {
  case Sound_file_config::endian_file:
    return_format = SF_ENDIAN_FILE;
    break;
  case Sound_file_config::endian_little:
    return_format = SF_ENDIAN_LITTLE;
    break;
  case Sound_file_config::endian_big:
    return_format = SF_ENDIAN_BIG;
    break;
  case Sound_file_config::endian_CPU:
    return_format = SF_ENDIAN_CPU;
    break;
  default:
    return_value = false;
    break;
  }

  return return_value;
}

bool Sndfile_compat::sf_format_for_config(const Sound_file_config& config, int& return_format)
{
  bool return_value = false;

  int format_file_type;
  int format_data_format;
  int format_endianness;

  if (!sf_format_for_file_type(config.file_type, format_file_type))
  {
    goto exit_point;
  }
  if (!sf_format_for_data_format(config.data_format, format_data_format))
  {
    goto exit_point;
  }
  if (!sf_format_for_file_type(config.endianness, format_endianness))
  {
    goto exit_point;
  }

  return_format = format_file_type | format_data_format | format_endianness;
  return_value = true;

 exit_point:
  return return_value;
}
