// Movie_config.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef MOVIE_CONFIG_HPP_
#define MOVIE_CONFIG_HPP_

#include <map>
#include <string>
#include <vector>

#include "Image_types.hpp"
#include "Movie_types.hpp"

using std::map;
using std::string;
using std::vector;

namespace Roan_trail
{
  namespace Recorder
  {
    struct Sound_file_config;
  }

  using Roan_trail::Recorder::Sound_file_config;

  namespace Builder
  {
    struct Movie_config
    {
      Movie_config();
      ~Movie_config();
      // typedef
      typedef int Movie_preset;
      typedef int Frame_rate_type;
      typedef int Frame_format;
      typedef map<int, string> Constant_map;
      // data members
      Movie_preset movie_preset;
      Frame_rate_type frame_rate;
      Frame_format frame_format;
      Rect_size original_size;
      int keyframe_period;
      Sound_file_config* sound;
      // helpers
      //   movie preset
      static string string_for_movie_preset(Movie_preset preset)
      { return m_movie_preset_map[preset]; }
      static bool movie_preset_for_string(const string& movie_preset_string,
                                          Movie_preset& return_movie_preset);
      static void movie_preset_string_for_file_extension(const string& extension,
                                                         string& return_movie_preset_string);
      static string extension_for_movie_preset(Movie_preset preset)
      { return m_movie_preset_extension_map[preset]; }
      //   frame rate
      static string string_for_frame_rate_type(Frame_rate_type rate)
      { return m_frame_rate_type_map[rate]; }
      static bool frame_rate_type_for_string(const string& frame_rate_type_string,
                                             Frame_rate_type& return_frame_rate);
      static void frame_rate_for_frame_rate_type(Frame_rate_type rate, Frame_rate& return_frame_rate)
      { return_frame_rate = m_frame_rate_map[rate]; }
      static bool audio_sample_rate_compatible_with_frame_rate_type(int audio_sample_rate,
                                                                    Frame_rate_type rate);
      static void compatible_frame_rates_for_audio_sample_rate(int audio_sample_rate,
                                                               vector<Frame_rate_type>& return_frame_rates);
      //   frame format
      static string string_for_frame_format(Frame_format format)
      { return m_frame_format_map[format]; }
      static bool frame_format_for_string(const string& frame_format_string,
                                          Frame_format& return_frame_format);
      // input size is the original size of a graphic for a given frame format (pixels are always square)
      static bool input_size_for_movie_config(const Movie_config& config, Rect_size& return_size);
      // output size is the destination frame size for a given frame format (pixels can be non-square)
      static bool output_size_for_movie_config(const Movie_config& config, Rect_size& return_size);
      // struct constants
      //   movie
      static const Movie_preset movie_preset_qtrle = 0; // Apple Animation (RLE) codec, MOV file format
      //   frame rate
      static const Frame_rate_type frame_rate_5_Hz            = 0;
      static const Frame_rate_type frame_rate_10_Hz           = 1;
      static const Frame_rate_type frame_rate_20_Hz           = 2;
      static const Frame_rate_type frame_rate_24_Hz           = 3;
      static const Frame_rate_type frame_rate_25_Hz           = 4;
      static const Frame_rate_type frame_rate_29_97_Hz_exact  = 5;
      static const Frame_rate_type frame_rate_29_97_Hz_approx = 6;
      static const Frame_rate_type frame_rate_30_Hz           = 7;
      static const Frame_rate_type frame_rate_50_Hz           = 8;
      static const Frame_rate_type frame_rate_60_Hz           = 9;
      //   frame size
      static const Frame_format frame_format_601_ntsc_4_to_3               = 0;
      static const Frame_format frame_format_601_ntsc_16_to_9_anamorphic   = 1;
      static const Frame_format frame_format_dv_ntsc_4_to_3                = 2;
      static const Frame_format frame_format_dv_ntsc_16_to_9_anamorphic    = 3;
      static const Frame_format frame_format_601_dv_pal_4_to_3             = 4;
      static const Frame_format frame_format_601_dv_pal_16_to_9_anamorphic = 5;
      static const Frame_format frame_format_hd_720p                       = 6;
      static const Frame_format frame_format_hd_1080i_p                    = 7;
      static const Frame_format frame_format_original                      = 8;
      //   default constants
      static const Movie_preset default_movie_preset;
      static const Frame_rate_type default_frame_rate_type;
      static const Frame_format default_frame_format;
    private:
      // maps
      static Constant_map m_movie_preset_map;
      static Constant_map m_frame_rate_type_map;
      static Constant_map m_frame_format_map;
      static map<Movie_preset, string> m_movie_preset_extension_map;
      static map<Frame_rate_type, Frame_rate> m_frame_rate_map;
      static map<Frame_format, Rect_size> m_frame_format_input_map;
      static map<Frame_format, Rect_size> m_frame_format_output_map;

      //
      static bool m_maps_setup;
      static void mf_setup_maps();
      // prevent compiler from generating
      Movie_config(const Movie_config& c);
      Movie_config& operator=(const Movie_config& c);
    };
  }
}

#endif // MOVIE_CONFIG_HPP_
