// Logger.cpp
//
// Copyright 2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#include "Logger.hpp"
#include "common.hpp"

using namespace Roan_trail;

//
// Constructor/destructor
//

Logger::Logger()
  : m_output_stream(&std::cout),
    m_output_level(normal),
    m_entry_type(message),
    m_entry_threshold(message),
    m_pass_output(true)
{
  set_output_level(normal);

  postcondition(mf_invariant(false));
}

//
// Mutators
//

void Logger::set_output_level(Output_level level)
{
  precondition(mf_invariant());

  m_output_level = level;

  switch (m_output_level)
  {
  case quiet:
    m_entry_threshold = error;
    break;
  case normal:
    m_entry_threshold = message;
    break;
  case verbose:
    m_entry_threshold = info;
    break;
  default:
    assert(false && "invalid output level");
    break;
  }

  m_pass_output = mf_check_pass_output();

  postcondition(mf_invariant());
}

//
// Stream operations
//

Logger& Logger::operator<<(Entry_type entry)
{
  precondition(mf_invariant());

  m_entry_type = entry;
  m_pass_output = mf_check_pass_output();

  postcondition(mf_invariant());
  return *this;
}

//
// Protected member functions
//

bool Logger::mf_invariant(bool check_base_class) const
{
  bool return_value = false;

  if (!m_output_stream)
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

bool Logger::mf_check_pass_output()
{
  return (static_cast<int>(m_entry_type) >= static_cast<int>(m_entry_threshold));
}

//
// Class constants
//

Logger Logger::m_default_logger;

//
// Stream functions
//

Logger& ::Roan_trail::operator<<(Logger& log, ostream&(*pf)(ostream&))
{
  if (log.m_pass_output)
  {
    pf(*log.m_output_stream);
  }

  return log;
}

Logger& ::Roan_trail::operator<<(Logger& log, ios_base&(*pf)(ostream&))
{
  if (log.m_pass_output)
  {
    pf(*log.m_output_stream);
  }

  return log;
}
