// Dir_slide_collection.hpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

#ifndef DIR_SLIDE_COLLECTION_HPP_
#define DIR_SLIDE_COLLECTION_HPP_

#include "Slide_collection.hpp"
#include "File_manager.hpp"
#include "error/Source_error.hpp"
#include <libxml++/nodes/element.h>
#include <string>
#include <vector>
#include <algorithm>

using std::string;
using std::vector;
using xmlpp::Element;
using Roan_trail::Error;

namespace Roan_trail
{
  namespace Source
  {
    template <class Image> class Dir_slide_collection : public Slide_collection<Image>
    {
    public:
      // constructor/destructor
      Dir_slide_collection()
        : Slide_collection<Image>()
      {}
      ~Dir_slide_collection() {}
      // accessors
      string source_type() const { return "directory"; }
      // load/save
      //   using source
      virtual bool load_from_source(const string& path, Error_param& return_error);
      //   load whole collection from XML
      virtual bool load_from_XML(const string& file_path,
                                 const Node* XML_node,
                                 Error_param& return_error);
      //   save whole collection to XML
      virtual bool save_as_XML(const string& file_path,
                               Node* XML_node,
                               Error_param& return_error);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
      //
      virtual bool mf_process_source_images_with_notes(const vector<string>& image_paths,
                                                       const string& source_path,
                                                       Error_param& return_error) = 0;
    private:
      // prevent compiler from generating
      Dir_slide_collection(const Dir_slide_collection& slide_collection);
      Dir_slide_collection& operator= (const Dir_slide_collection& slide_collection);
    };

    //
    // Template implementation
    //

    //
    // Helpers
    //

    namespace
    {
      const char* ih_text_for_field(const char *node_name,
                                    const Node* start_node,
                                    const string& diagnostic_prefix,
                                    const string& file_path,
                                    Error_param& return_error)
      {
        precondition(node_name
                     && start_node
                     && !return_error());

        const char* return_value = 0;

        start_error_block();

        const vector<Node*> field_nodes = start_node->find(node_name);
        on_error(!field_nodes.size(),
                 new Source_error(error_location(),
                                  Source_error::load,
                                  diagnostic_prefix + string(", node ")
                                  + start_node->get_path()
                                  + string(" ")
                                  + string(node_name)
                                  + string(" node not found"),
                                  file_path + ih_format_XML_file_line(start_node->get_line())));
        const vector<Node*> field_text_nodes = field_nodes[0]->find("text()");
        on_error(!field_text_nodes.size(),
                 new Source_error(error_location(),
                                  Source_error::load,
                                  diagnostic_prefix + string(", node ")
                                  + field_nodes[0]->get_path()
                                  + string(" text node not found"),
                                  file_path + ih_format_XML_file_line(field_nodes[0]->get_line())));
        // the reinterpret cast is used to convert unsigned char to char
        // unfortunately, there seems to be no obvious way around this
        const char *field_text_content =
          reinterpret_cast<const char *>(field_text_nodes[0]->cobj()->content);
        on_error(!field_text_content,
                 new Source_error(error_location(),
                                  Source_error::load,
                                  diagnostic_prefix + string(", node ")
                                  + field_text_nodes[0]->get_path()
                                  + string(" text content not found"),
                                  file_path
                                  + ih_format_XML_file_line(field_text_nodes[0]->get_line())));

        return_value = field_text_content;
        goto exit_point;

        end_error_block();

        default_error_handler_and_cleanup(return_error, return_value, 0);
        goto exit_point;

      exit_point:
        postcondition(return_value || !return_error.need_error() || return_error());
        return return_value;
      }
    }

    //
    // Load/save
    //

    template <class Image> bool Dir_slide_collection<Image>::load_from_source(const string& path,
                                                                              Error_param& return_error)
    {
      precondition(!return_error()
                   && mf_invariant());

      bool return_value = true;

      start_error_block();

      const string& source_path = path;

      const bool source_exists = File_manager::path_exists(source_path);
      on_error(!source_exists, new Source_error(error_location(),
                                                Source_error::invalid_source,
                                                string("Source directory not found: ") + source_path));
      const bool is_directory = File_manager::path_is_directory(source_path);
      on_error(!is_directory, new Source_error(error_location(),
                                               Source_error::invalid_source,
                                               string("Source path is not a directory: ")
                                               + source_path));

      vector<string> dir_contents;
      const bool have_contents = File_manager::contents_of_directory_at_path(source_path, dir_contents);
      on_error(!have_contents, new Source_error(error_location(),
                                                Source_error::invalid_source,
                                                string("Could not get contents of directory: ")
                                                + source_path));

      sort(dir_contents.begin(), dir_contents.end());

      Error_param error;
      const bool images_processed = mf_process_source_images_with_notes(dir_contents,
                                                                        source_path,
                                                                        error);
      on_error(!images_processed, new Source_error(error_location(),
                                                   Source_error::general,
                                                   error()));

      Slide_collection<Image>::m_source = source_path;
      Slide_collection<Image>::m_source_loaded = true;

      goto exit_point;

      end_error_block();

      default_error_handler_and_cleanup(return_error, return_value, false);
      goto exit_point;

    exit_point:
      postcondition(mf_invariant());
      return return_value;
    }

    template <class Image> bool Dir_slide_collection<Image>::load_from_XML(const string& file_path,
                                                                           const Node* XML_node,
                                                                           Error_param& return_error)
    {
      precondition(!Slide_collection<Image>::m_source_loaded
                   && XML_node
                   && !return_error()
                   && mf_invariant());

      bool return_value = false;

      start_error_block();

      Error_param error;
      const string diagnostic_prefix = "error loading XML file";

      try
      {
        const char *directory_text = ih_text_for_field("//directory",
                                                       XML_node,
                                                       diagnostic_prefix,
                                                       file_path,
                                                       error);
        on_error(!directory_text, new Source_error(error_location(),
                                                   Source_error::general,
                                                   error()));
        string directory = directory_text;
        const vector<Node*> slide_nodes = XML_node->find("//slide");
        on_error(!slide_nodes.size(), new Source_error(error_location(),
                                                       Source_error::load,
                                                       diagnostic_prefix + string(", node: ")
                                                       + XML_node->get_path()
                                                       + string(" no slides found"),
                                                       file_path
                                                       + ih_format_XML_file_line(XML_node->get_line())));
        for (vector<Node*>::const_iterator current_slide_node = slide_nodes.begin();
             current_slide_node != slide_nodes.end();
             ++current_slide_node)
        {
          // index
          const char *slide_index_text = ih_text_for_field("index",
                                                           (*current_slide_node),
                                                           diagnostic_prefix,
                                                           file_path,
                                                           error);
          on_error(!slide_index_text, new Source_error(error_location(),
                                                       Source_error::general,
                                                       error()));
          int current_slide_index;
          stringstream current_slide_index_stream(slide_index_text);
          current_slide_index_stream >> current_slide_index;
          // file image name
          const char *file_image_name_text = ih_text_for_field("file_image_name",
                                                               (*current_slide_node),
                                                               diagnostic_prefix,
                                                               file_path,
                                                               error);
          on_error(!file_image_name_text, new Source_error(error_location(),
                                                           Source_error::general,
                                                           error()));
          string file_image_name = file_image_name_text;
          // notes
          Error_param notes_error(false);
          const char *notes_text = ih_text_for_field("notes",
                                                     (*current_slide_node),
                                                     diagnostic_prefix,
                                                     file_path,
                                                     notes_error); // ignore error
          string slide_notes;
          if (notes_text)
          {
            slide_notes = notes_text;
          }
          // else if notes_text is empty, there might be no notes text node, so ignore any error and go on

          const string file_image_path = directory + string("/") + file_image_name;
          Slide_collection<Image>::m_slides.push_back(new Slide<Image>(current_slide_index,
                                                                       slide_notes,
                                                                       file_image_path));
          if (Slide_collection<Image>::slide_count() != (current_slide_index + 1))
          {
            stringstream diagnostic_stream;
            diagnostic_stream << diagnostic_prefix;
            diagnostic_stream << ", invalid slide index " << current_slide_index;
            diagnostic_stream << " found at actual index " << Slide_collection<Image>::slide_count() - 1;
            on_error(true, new Source_error(error_location(),
                                            Source_error::load,
                                            file_path));
          }
        }

        Slide_collection<Image>::m_source = directory;
        Slide_collection<Image>::m_source_loaded = true;

        return_value = true;

      }
      catch (const xmlpp::exception& e)
      {
        const string diagnostic = diagnostic_prefix + string(": ") + e.what();
        on_error(true, new Source_error(error_location(),
                                        Source_error::load,
                                        diagnostic,
                                        file_path));
      }

      goto exit_point;
      end_error_block();

      default_error_handler_and_cleanup(return_error, return_value, false);
      goto exit_point;

    exit_point:
      postcondition((return_value || !return_error.need_error() || return_error())
                    && mf_invariant());
      return return_value;
    }

    template <class Image> bool Dir_slide_collection<Image>::save_as_XML(const string& file_path,
                                                                         Node* XML_node,
                                                                         Error_param& return_error)
    {
      precondition(XML_node
                   && !return_error()
                   && mf_invariant());

      bool return_value = false;
      start_error_block();

      Error_param error;

      try
      {
        Node* slides_node = XML_node->add_child("slides");
        Element* source_type_element = slides_node->add_child("source_type");
        source_type_element->add_child_text(source_type());
        Element* source_directory_element = slides_node->add_child("directory");
        source_directory_element->add_child_text(Slide_collection<Image>::source());
        for (size_t slide_index = 0; slide_index < Slide_collection<Image>::m_slides.size(); ++slide_index)
        {
          const Slide<Image> &current_slide = *(Slide_collection<Image>::m_slides[slide_index]);
          assert((static_cast<int>(slide_index) == current_slide.index())
                 && "error, slide numbers do not match when saving to XML");
          Node* current_slide_node = slides_node->add_child("slide");
          stringstream slide_index_stream;
          slide_index_stream << slide_index;
          Element* index_element = current_slide_node->add_child("index");
          index_element->add_child_text(slide_index_stream.str());
          Element* file_name_element = current_slide_node->add_child("file_image_name");
          file_name_element->add_child_text(File_manager::file_name_for_file_path(current_slide.path()));
          Element* notes_element = current_slide_node->add_child("notes");
          notes_element->add_child_text(current_slide.notes());
        }
      }
      catch (const xmlpp::exception& e)
      {
        string diagnostic = string("error saving XML to file: ") + e.what();
        on_error(true, new Source_error(error_location(),
                                        Source_error::save,
                                        diagnostic,
                                        file_path));
      }

      return_value = true;

      goto exit_point;
      end_error_block();

      default_error_handler_and_cleanup(return_error, return_value, false);
      goto exit_point;

    exit_point:
      postcondition((return_value || !return_error.need_error() || return_error())
                    && mf_invariant());
      return return_value;
    }

    //
    // Protected member functions
    //

    template <class Image> bool Dir_slide_collection<Image>::mf_invariant(bool check_base_class) const
    {
      bool return_value = false;

      return_value = !check_base_class || Slide_collection<Image>::mf_invariant(check_base_class);

      goto exit_point;

    exit_point:
      return return_value;
    }
  }
}

#endif // DIR_SLIDE_COLLECTION_HPP_
