// Async_audio_file_writer.cpp
//
// Copyright 2011-2012 Roan Trail, Inc.
//
// This file is part of Kinetophone.
//
// Kinetophone is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published
// by the Free Software Foundation, either version 2 of the License,
// or (at your option) any later version.
//
// Kinetophone is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.  You should have received
// a copy of the GNU General Public License along with Kinetophone. If
// not, see <http://www.gnu.org/licenses/>.

//
// This code was based on the PortAudio file pa_mac_core_blocking.c:
//
//   Based on the Open Source API proposed by Ross Bencina
//   Copyright (c) 1999-2002 Ross Bencina, Phil Burk
//
//   Permission is hereby granted, free of charge, to any person obtaining
//   a copy of this software and associated documentation files
//   (the "Software"), to deal in the Software without restriction,
//   including without limitation the rights to use, copy, modify, merge,
//   publish, distribute, sublicense, and/or sell copies of the Software,
//   and to permit persons to whom the Software is furnished to do so,
//   subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be
//   included in all copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
//   IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
//   ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
//   CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//   WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE

// TODO: add checksum to buffer?

#include "Async_audio_file_writer.hpp"
#include "Ring_buffer.hpp"
#include "Sound_recorder_config.hpp"
#include "Sound_file_config.hpp"
#include "File_manager.hpp"
#include "common.hpp"
#include "Sndfile_compat.hpp"
#include <sndfile.h>
#include <portaudio.h>
#include "error/AAFW_error.hpp"
#include "error/Posix_error.hpp"
#include "error/Sndfile_error.hpp"
#include <iostream>
#include <iomanip>
#include <sstream>
#include <ctime> // Standards Rule 25 deviation (check_code_ignore)
#include <sys/time.h> // Standards Rule 25 deviation (check_code_ignore)
#include <cerrno> // (check_code_ignore)
#include <fcntl.h>

using std::cerr;
using std::endl;
using std::stringstream;
using Roan_trail::Long_int;
using Roan_trail::Error_param;
using namespace Roan_trail::Recorder;

//
// Internal constant/helper
//

namespace
{
  using Roan_trail::Error;

  static const uint32_t ic_start_marker = static_cast<uint32_t>(~0xE4524FF4);
  static const uint32_t ic_end_marker   = static_cast<uint32_t>(~0x4FF42152);

  bool ih_set_VBR(SNDFILE *file,
                  const string& error_file_path,
                  const Sound_recorder_config* recorder_config,
                  Error_param& return_error)
  {
    precondition(file
                 && !return_error());

    bool return_value = false;

    start_error_block();

    double VBR_quality = Sound_file_config::default_VBR_quality;
    bool have_VBR_quality = recorder_config->sound_file->has_VBR(VBR_quality);

    if (have_VBR_quality &&
        !Roan_trail::equal_fp(VBR_quality, Sound_file_config::default_VBR_quality))
    { // set VBR only if not default, because libsndfile already sets a default
      stringstream diagnostic;
      int ret_code = sf_command(file, SFC_SET_VBR_ENCODING_QUALITY, &VBR_quality, sizeof(double));
      if (ret_code)
      {
        stringstream diagnostic;
        diagnostic << "error setting VBR (variable bit rate) quality to " << VBR_quality;
        on_error(ret_code, new AAFW_error(error_location(),
                                          AAFW_error::general,
                                          diagnostic.str(),
                                          new Sndfile_error(0,
                                                            "sf_command",
                                                            error_file_path.c_str())));
      }
    }

    return_value = true;
    goto exit_point;

    end_error_block();

    default_error_handler_and_cleanup(return_error, return_value, false);
    goto exit_point;

  exit_point:
    return return_value;
  }
}

SNDFILE* Async_audio_file_writer::mf_open_sound_file(string& return_output_file_path, Error_param& return_error)
{
  precondition(!return_error());

  SNDFILE* return_value = 0;

  Error_param error;

  // variable declarations for error cleanup
  bool created_temp_file = false;
  string temporary_file_name = "";
  SNDFILE* file = 0;

  start_error_block();

  // use libsndfile to open the file

  SF_INFO sf_info;
  sf_info.samplerate = static_cast<int>(m_recorder_config->sound_file->sample_rate);
  sf_info.frames = m_recorder_config->frames_per_buffer;
  sf_info.channels = m_recorder_config->sound_file->channels;
  if (!Sndfile_compat::sf_format_for_config(*m_recorder_config->sound_file, sf_info.format)
      || !sf_format_check(&sf_info))
  {
    // invalid sf_info structure
    string prefix("  ");
    stringstream diagnostic;
    diagnostic << "The combination:" << endl <<  m_recorder_config->sound_file->format_string(prefix);
    diagnostic << "is invalid.  Not all combinations of these options are allowed.";
    AAFW_error *format_error = new AAFW_error(error_location(),
                                              AAFW_error::open_output_file,
                                              diagnostic.str());
    on_error(true, format_error);
  }

  string output_file = m_recorder_config->sound_file->file_name;
  bool overwrite = m_recorder_config->file_overwrite;
  // use existing temporary file if overwrite is allowed
  // and no output file has been specified
  if ((output_file == "") && overwrite)
  {
    output_file = m_temporary_output_file;
  }

  int close_on_sf_close = 1;
  string error_file_path;
  if (output_file == "")
  {
    // create temporary output file
    int fd;
    string file_extension = string(".")
      + Sound_file_config::string_for_file_type(m_recorder_config->sound_file->file_type);
    created_temp_file = File_manager::create_temporary_file(file_extension,
                                                            File_manager::temporary_sound_file,
                                                            temporary_file_name,
                                                            fd,
                                                            error);
    on_error(!created_temp_file, new AAFW_error(error_location(),
                                                AAFW_error::create_temp_file,
                                                error()));
    file = sf_open_fd(fd, SFM_WRITE, &sf_info, close_on_sf_close);
    error_file_path = string(temporary_file_name.c_str());
    on_error_with_label(!file,
                        sf_open_fd_error_handler,
                        new Sndfile_error(sf_error(0),
                                          "sf_open_fd",
                                          error_file_path.c_str()));

    m_temporary_output_file = temporary_file_name;
    output_file = temporary_file_name;
  }
  else
  {
    // output file specified
    const char* file_path = output_file.c_str();
    error_file_path = string(file_path);
    int oflag = O_CREAT;
    int mode = S_IRUSR | S_IWUSR;
    if (!overwrite)
    { // no overwrite:
      oflag |= O_WRONLY;
      // using O_CREAT and O_EXCL flags together will cause an error to be returned
      // if the file exists
      oflag |= O_EXCL;
      int fd = open(file_path,
                    oflag,
                    mode);
      if (Posix_error::error_return == fd)
      {
        if (EEXIST == errno) // (check_code_ignore)
        {
          Error *overwrite_error = new AAFW_error(error_location(),
                                                  AAFW_error::overwrite,
                                                  string("sound file exists and overwrite not requested"));
          overwrite_error->error_dictionary()[Error::file_path_error_key] = output_file;
          on_error(true, overwrite_error);
        }
        else
        {
          on_error(true, new AAFW_error(error_location(),
                                        AAFW_error::open_output_file,
                                        new Posix_error(errno, // (check_code_ignore)
                                                        "open",
                                                        error_file_path.c_str())));
        }
      }
      file = sf_open_fd(fd,
                        SFM_WRITE,
                        &sf_info,
                        close_on_sf_close);
      on_error_with_label(!file,
                          sf_open_fd_error_handler,
                          new Sndfile_error(sf_error(0),
                                            "sf_open_fd",
                                            error_file_path.c_str()));
    }
    else
    {
      // overwrite
      oflag |= O_RDWR;
      int fd = open(file_path, oflag, mode);
      on_error(Posix_error::error_return == fd, new AAFW_error(error_location(),
                                                               AAFW_error::open_output_file,
                                                               new Posix_error(errno, // (check_code_ignore)
                                                                               "open",
                                                                               error_file_path.c_str())));
      // TODO: FLAC files do not like to be opened read/write, is there a work-around? (see callback)
      file = sf_open_fd(fd,
                        SFM_RDWR,
                        &sf_info,
                        close_on_sf_close);
      on_error_with_label(!file,
                          sf_open_fd_error_handler,
                          new Sndfile_error(sf_error(0),
                                            "sf_open_fd",
                                            error_file_path.c_str()));
      sf_count_t count = sf_seek(file,
                                 0,
                                 SEEK_SET);
      on_error(-1 == count, new AAFW_error(error_location(),
                                           AAFW_error::open_output_file,
                                           new Sndfile_error(sf_error(file),
                                                             "sf_seek",
                                                             error_file_path.c_str())));
    }
  }

  bool set_VBR = ih_set_VBR(file,
                            error_file_path,
                            m_recorder_config,
                            error);
  on_error(!set_VBR, new AAFW_error(error_location(),
                                    AAFW_error::open_output_file,
                                    error()));

  return_output_file_path = output_file;
  return_value = file;
  goto exit_point;

  end_error_block();

 sf_open_fd_error_handler:
  if (return_error.need_error())
  {
    if (SF_ERR_SYSTEM == handler_error->code())
    {
      const string &output_file = handler_error->error_dictionary()[Error::file_path_error_key];
      if ("" != output_file)
      {
        const string parent_directory = File_manager::parent_directory_for_path(output_file);
        double dummy_fraction = 0.0;
        Long_int available_bytes = 0L;
        if (File_manager::file_system_available_for_path(parent_directory,
                                                         dummy_fraction,
                                                         available_bytes,
                                                         error) && (0 == available_bytes))
        {
          stringstream diagnostic;
          diagnostic << "The file system is full for file: " << output_file;
          return_error = new AAFW_error(error_location(),
                                        AAFW_error::open_output_file,
                                        diagnostic.str(),
                                        handler_error);
          goto error_cleanup;
        }
      }
    }

    return_error = new AAFW_error(error_location(),
                                  AAFW_error::open_output_file,
                                  handler_error);
  }
  goto error_cleanup;

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (file)
  {
    sf_close(file); // ignore error
  }
  if (created_temp_file)
  {
    if ("" != temporary_file_name)
    {
      unlink(temporary_file_name.c_str()); // ignore error
    }
    m_temporary_output_file = "";
  }
  return_value = 0;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// static member functions
//

void Async_audio_file_writer::mf_set_write_error_code(int write_error_code)
{
  int32_t have_write_error = __sync_fetch_and_or(&m_have_write_error, 0x1);
  if (!have_write_error)
  {
    m_write_error_code = write_error_code;
  }
}

void Async_audio_file_writer::mf_set_write_loop_error(AAFW_error* write_error)
{
  int32_t have_write_error = __sync_fetch_and_or(&m_have_write_error, 0x1);
  if (!have_write_error)
  {
    delete m_write_loop_error;
    m_write_loop_error = write_error;
  }
}

////////////////////////////
// worker thread function //
////////////////////////////
void* Async_audio_file_writer::mf_write_loop(void* arg)
{
  precondition(arg);

  Async_audio_file_writer* AAFW = reinterpret_cast<Async_audio_file_writer*>(arg);
  __sync_or_and_fetch(&AAFW->m_writer_thread_running, 1);

  Error_param error;
  bool first_write = true;

  start_AAFW_error_block();

  int posix_error_code;

  posix_error_code = pthread_mutex_lock(&AAFW->m_thread_mutex);
  on_AAFW_error(posix_error_code, new AAFW_error(error_location(),
                                                 AAFW_error::write_loop,
                                                 new Posix_error(posix_error_code,
                                                                 "pthread_mutex_lock",
                                                                 0)));

  // let thread creator know that the thread is now running
  pthread_cond_signal(&AAFW->m_thread_cond); // never returns error code

  posix_error_code = pthread_mutex_unlock(&AAFW->m_thread_mutex);
  on_AAFW_error(posix_error_code, new AAFW_error(error_location(),
                                                 AAFW_error::write_loop,
                                                 new Posix_error(posix_error_code,
                                                                 "pthread_mutex_unlock",
                                                                 0)));

  Long_int avail;
  bool success;

  while (true)
  {
    bool finished;

    // get the start marker
    success = AAFW->mf_wait_for_min_read_available(sizeof(uint32_t),
                                                   avail,
                                                   finished,
                                                   error);
    on_AAFW_error(!success, new AAFW_error(error_location(),
                                           AAFW_error::write_loop,
                                           error()));
    if (finished)
    {
      goto write_loop_finish;
    }
    uint32_t start_marker;
    AAFW->m_ring_buffer->read_data(reinterpret_cast<void*>(&start_marker), sizeof(uint32_t));
    if (ic_start_marker != start_marker)
    {
      stringstream diagnostic;
      diagnostic << "unexpected start marker 0x" << std::setfill('0') << std::setw(8) << std::hex << start_marker;
      diagnostic << " should be 0x" << std::setfill('0') << std::setw(8) << std::hex << ic_start_marker;
      on_AAFW_error(true, new AAFW_error(error_location(),
                                         AAFW_error::write_loop_corrupt_buffer,
                                         diagnostic.str()));
    }
    // get frame count
    success = AAFW->mf_wait_for_min_read_available(sizeof(int), avail, finished, error);
    on_AAFW_error(!success, new AAFW_error(error_location(),
                                           AAFW_error::write_loop,
                                           error()));
    if (finished)
    {
      goto write_loop_finish;
    }
    int frame_count;
    AAFW->m_ring_buffer->read_data(reinterpret_cast<void*>(&frame_count), sizeof(int));
    if (frame_count > AAFW->m_max_frames)
    {
      stringstream diagnostic;
      diagnostic << "frame count " << frame_count << " is greater than max frames " << AAFW->m_max_frames;
      on_AAFW_error(true, new AAFW_error(error_location(),
                                         AAFW_error::write_loop_corrupt_buffer,
                                         diagnostic.str()));
    }

    // get buffer data frames
    char *buffer = AAFW->m_output_buffer;
    Long_int bytes_to_read = frame_count * AAFW->m_sample_size * AAFW->m_recorder_config->sound_file->channels;
    while (bytes_to_read > 0)
    {
      success = AAFW->mf_wait_for_min_read_available(1, avail, finished, error);
      on_AAFW_error(!success, new AAFW_error(error_location(),
                                             AAFW_error::write_loop,
                                             error()));
      if (finished)
      {
        goto write_loop_finish;
      }
      if (AAFW->m_overflow_logging_enabled)
      {
        // Output percentage of buffer utilized
        Long_int buffer_size = AAFW->m_ring_buffer->buffer_size();
        double pct = 100.0 * static_cast<double>(avail) / static_cast<double>(buffer_size);
        if (pct > 50.0)
        {
          cerr << "Read buffer is " << pct << " % full: ";
          cerr << avail << " of " << buffer_size << endl;
        }
      }
      const Long_int compare_bytes_to_read = bytes_to_read;
      const Long_int read_count = min(avail, compare_bytes_to_read);
      AAFW->m_ring_buffer->read_data(buffer, read_count);

      buffer += read_count;
      bytes_to_read -= read_count;
    }

    // get the end marker
    success = AAFW->mf_wait_for_min_read_available(sizeof(uint32_t), avail, finished, error);
    on_AAFW_error(!success, new AAFW_error(error_location(),
                                           AAFW_error::write_loop,
                                           error()));
    if (finished)
    {
      goto write_loop_finish;
    }
    uint32_t end_marker;
    AAFW->m_ring_buffer->read_data(reinterpret_cast<void *>(&end_marker), sizeof(uint32_t));
    if (ic_end_marker != end_marker)
    {
      stringstream diagnostic;
      diagnostic << "unexpected end marker 0x" << std::setfill('0') << std::setw(8) << std::hex << end_marker;
      diagnostic << " should be 0x" << std::setfill('0') << std::setw(8) << std::hex << ic_end_marker;
      on_AAFW_error(true, new AAFW_error(error_location(),
                                         AAFW_error::write_loop_corrupt_buffer,
                                         diagnostic.str()));
    }

    if (first_write)
    {
      // truncate the sound file before the first write
      // in case we are writing over a file
      first_write = false;
      if (Sound_file_config::file_type_flac != AAFW->m_recorder_config->sound_file->file_type)
      {
        sf_count_t frames = 0;
        // TODO: FLAC subsystem does not like this command, is there a work-around to truncate it?
        int sf_ret = sf_command(AAFW->m_output_file, SFC_FILE_TRUNCATE, &frames, sizeof(frames));
        on_AAFW_error(sf_ret, new AAFW_error(error_location(),
                                             AAFW_error::write_loop,
                                             new Sndfile_error(sf_ret,
                                                               "sf_command",
                                                               AAFW->m_output_file_path.c_str())));
      }
    }

    const sf_count_t items_to_write = frame_count * AAFW->m_recorder_config->sound_file->channels;
    Stream_float* output_buffer = reinterpret_cast<Stream_float*>(AAFW->m_output_buffer);

    // in the sf_write_float call, libsndfile converts from the canonical float in the buffer
    // to file format specified when opened
    const sf_count_t items_written = sf_write_float(AAFW->m_output_file,
                                                    output_buffer,
                                                    items_to_write);
    if (items_written != items_to_write)
    {
      // amount of data written doesn't match the amount requested
      // figure out what the error is
      const int sound_file_error = sf_error(AAFW->m_output_file);
      if (SF_ERR_NO_ERROR == sound_file_error)
      {
        stringstream diagnostic;
        diagnostic << "file write failed, items written: " << items_written;
        diagnostic << " does not equal requested items to write: " << items_to_write;
        on_AAFW_error(true, new AAFW_error(error_location(),
                                           AAFW_error::write_loop,
                                           diagnostic.str()));
      }
      else
      {
        const string output_file = AAFW->m_output_file_path;
        if (SF_ERR_SYSTEM == sound_file_error)
        {
          const string parent_directory = File_manager::parent_directory_for_path(output_file);
          double dummy_fraction = 0.0;
          Long_int available_bytes = 0L;
          if (File_manager::file_system_available_for_path(parent_directory,
                                                           dummy_fraction,
                                                           available_bytes,
                                                           error) && (0 == available_bytes))
          {
            stringstream diagnostic;
            diagnostic << "The file system is full for file: " << output_file;
            on_AAFW_error(true, new AAFW_error(error_location(),
                                               AAFW_error::write_loop,
                                               diagnostic.str()));
          }
        }
        on_AAFW_error(true, new AAFW_error(error_location(),
                                           AAFW_error::write_loop,
                                           new Sndfile_error(sound_file_error,
                                                             "sf_write_float",
                                                             output_file.c_str())));
      }
    }
    sf_write_sync(AAFW->m_output_file); // force writing of all file cache buffers

    // Notes: if we implement a recoverable error in this function, we
    // have to adjust the frame count, maybe by subtracting the number of skipped
    // frames?

  }
  end_AAFW_error_block();

 error_handler:
  AAFW->mf_set_write_loop_error(handler_AAFW_error);
  goto write_loop_finish;

 write_loop_finish:

  __sync_and_and_fetch(&AAFW->m_writer_thread_running, 0);
  pthread_exit(0); // no return value (check_code_ignore)
}


//
// private member functions
//

bool Async_audio_file_writer::mf_wait_for_min_read_available(Long_int min_read_available,
                                                             Long_int& return_available,
                                                             bool& return_finished,
                                                             Error_param& return_error)
{
  precondition((min_read_available >= 0)
               && !return_error());

  bool return_value = false;

  struct timeval now;
  struct timespec timeout;
  bool have_lock = false;

  start_error_block();

  return_finished = false;

  Long_int avail = 0;

  // we want to loop here until we actually have
  // data available (i.e. not just signalled)
  //
  // this also gives us a chance to exit the
  // thread
  while (true)
  {
    avail = m_ring_buffer->read_available();
    if (avail < min_read_available)
    {
      // we want to wait until the ring buffer is empty to check for
      // thread exit
      if (__sync_fetch_and_add(&m_write_loop_finish, 0))
      { // thread finished
        return_finished = true;
        goto loop_exit;
      }
      // block until data is available
      int posix_error_code;

      posix_error_code = pthread_mutex_lock(&m_input_mutex);
      on_error(posix_error_code, new AAFW_error(error_location(),
                                                AAFW_error::write_loop,
                                                new Posix_error(posix_error_code,
                                                                "pthread_mutex_lock",
                                                                0)));
      have_lock = true;

      do
      {
        if (__sync_fetch_and_add(&m_write_loop_finish, 0))
        { // gives a chance to exit the wait loop if the thread is finishing up
          assert((0 == avail) && "ring buffer not empty when finishing up worker thread");
          break;
        }
        posix_error_code = gettimeofday(&now, 0);
        on_error(posix_error_code, new AAFW_error(error_location(),
                                                  AAFW_error::start,
                                                  new Posix_error(posix_error_code,
                                                                  "gettimeofday",
                                                                  0)));
        timeout.tv_sec = now.tv_sec + 1;
        timeout.tv_nsec = now.tv_usec * 1000;
        pthread_cond_timedwait(&m_input_cond, &m_input_mutex, &timeout);
      }
      while (ETIMEDOUT == posix_error_code);
      on_error(posix_error_code, new AAFW_error(error_location(),
                                                AAFW_error::write_loop,
                                                new Posix_error(posix_error_code,
                                                                "pthread_cond_timedwait",
                                                                0)));

      posix_error_code = pthread_mutex_unlock(&m_input_mutex);
      have_lock = false;
      on_error(posix_error_code, new AAFW_error(error_location(),
                                                AAFW_error::write_loop,
                                                new Posix_error(posix_error_code,
                                                                "pthread_mutex_unlock",
                                                                0)));
    }
    else
    {
      goto loop_exit;
    }
  }

 loop_exit:
  return_available = avail;
  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (have_lock)
  {
    pthread_mutex_unlock(&m_input_mutex); // ignore error code
  }
  return_value = false;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// constructor/destructor
//

Async_audio_file_writer::Async_audio_file_writer(bool enable_overflow_logging)
  : m_write_loop_finish(0),
    m_writer_thread_running(0),
    m_have_write_error(0),
    m_frames_written_count(0),
    m_recorder_config(new Sound_recorder_config),
    m_output_file(0),
    m_output_file_path(),
    m_output_buffer(0),
    m_ring_buffer(0),
    m_max_frames(0),
    m_write_chunk_size(0),
    m_write_buffer_size(0),
    m_sample_size(0),
    m_writer_thread(),
    m_input_mutex(),
    m_input_cond(),
    m_thread_mutex(),
    m_thread_cond(),
    m_overflow_logging_enabled(enable_overflow_logging),
    m_temporary_output_file(),
    m_write_loop_error(0),
    m_write_error_code(0),
    m_clear_frames_written_count(false)
{
  postcondition(mf_invariant(false));
}

Async_audio_file_writer::~Async_audio_file_writer()
{
  precondition(mf_invariant(false));

  if (running())
  {
    Error_param e(false);
    stop(e); // ignore return
  }

  delete m_write_loop_error;
  delete m_ring_buffer;
  delete [] m_output_buffer;
  delete m_recorder_config;
}

//
// Accessors
//

AAFW_error* Async_audio_file_writer::write_error() const
{
  precondition(mf_invariant());

  AAFW_error* return_value = 0;

  bool have_error = __sync_fetch_and_add(const_cast<int32_t *>(&m_have_write_error), 0x0);
  if (!have_error)
  {
    goto exit_point;
  }

  if (m_write_loop_error)
  {
    return_value = m_write_loop_error->clone();
  }
  else
  {
    return_value = new AAFW_error(__FILE__,
                                  "Async_audio_file_write::write",
                                  0,
                                  m_write_error_code);
  }

 exit_point:
  return return_value;
}

//
// Control
//

bool Async_audio_file_writer::start(const Sound_recorder_config& recorder_config, Error_param& return_error)
{
  precondition(!running()
               && !(recorder_config.file_overwrite
                    && (Sound_file_config::file_type_flac == recorder_config.sound_file->file_type))
               && !return_error()
               && mf_invariant());

  bool return_value = false;
  pthread_attr_t attr;
  bool file_open = false;
  bool have_input_mutex = false;
  bool have_input_cond = false;
  bool created_thread = false;
  bool have_attr = false;
  bool have_thread_mutex = false;
  bool have_thread_cond = false;
  bool thread_mutex_locked = false;

  start_error_block();

  m_output_file = 0;
  m_output_file_path = "";
  m_write_loop_finish = 0;
  m_have_write_error = 0;
  delete m_write_loop_error;
  m_write_loop_error = 0;
  m_write_error_code = AAFW_error::none;
  // update frames written count in write call
  m_clear_frames_written_count = true;

  *m_recorder_config = recorder_config;
  Error_param snd_file_error;
  string sound_file_path;
  SNDFILE* sound_file = mf_open_sound_file(sound_file_path, snd_file_error);
  on_error(!sound_file, new AAFW_error(error_location(),
                                       AAFW_error::start,
                                       snd_file_error()));
  m_output_file = sound_file;
  m_output_file_path = sound_file_path;
  file_open = true;

  // calculate ring buffer write sizes
  m_max_frames = m_recorder_config->frames_per_buffer;
  // input stream should always be 32 bit float
  m_sample_size = Pa_GetSampleSize(paFloat32); // Note: paFloat32 declaration uses an old-style cast
  m_write_buffer_size = m_max_frames * m_sample_size * m_recorder_config->sound_file->channels;
  delete [] m_output_buffer;
  m_output_buffer = new char[m_write_buffer_size];

  int overhead_size = sizeof(uint32_t) + sizeof(uint32_t) // start and end markers
    + sizeof(int); // frame count

  m_write_chunk_size = overhead_size + m_write_buffer_size ;

  // allocate the ring buffer
  Long_int byte_count = m_write_chunk_size * m_recorder_config->write_buffer_factor;

  delete m_ring_buffer;
  m_ring_buffer = new Ring_buffer(byte_count);

  pthread_mutex_init(&m_input_mutex, 0); // always returns 0
  have_input_mutex = true;
  pthread_cond_init(&m_input_cond, 0); // never returns error code
  have_input_cond = true;

  int posix_error_code;

  posix_error_code = pthread_attr_init(&attr);
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::start,
                                            new Posix_error(posix_error_code,
                                                            "pthread_attr_init",
                                                            0)));
  have_attr = true;

  // setup to check thread startup
  pthread_mutex_init(&m_thread_mutex, 0); // always returns 0
  have_thread_mutex = true;
  pthread_cond_init(&m_thread_cond, 0); // never returns error code
  have_thread_cond = true;

  posix_error_code = pthread_mutex_lock(&m_thread_mutex);
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::start,
                                            new Posix_error(posix_error_code,
                                                            "pthread_mutex_lock",
                                                            0)));
  thread_mutex_locked = true;

  // make thread joinable portably
  posix_error_code = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::start,
                                            new Posix_error(posix_error_code,
                                                            "pthread_attr_setdetachstate",
                                                            0)));

  posix_error_code = pthread_create(&m_writer_thread,
                                    &attr,
                                    mf_write_loop,
                                    this);
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::start,
                                            new Posix_error(posix_error_code,
                                                            "pthread_create",
                                                            0)));

  created_thread = true;

  have_attr = false;

  posix_error_code = pthread_attr_destroy(&attr); // ignore error code

  // wait for the thread to startup before proceeding

  struct timeval now;
  posix_error_code = gettimeofday(&now, 0);
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::start,
                                            new Posix_error(posix_error_code,
                                                            "gettimeofday",
                                                            0)));

  struct timespec timeout;
  timeout.tv_sec = now.tv_sec + 5;
  timeout.tv_nsec = now.tv_usec * 1000;

  posix_error_code = pthread_cond_timedwait(&m_thread_cond, &m_thread_mutex, &timeout);
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::start,
                                            new Posix_error(posix_error_code,
                                                            "pthread_cond_timedwait",
                                                            0)));

  posix_error_code = pthread_mutex_unlock(&m_thread_mutex);
  thread_mutex_locked = false;
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::start,
                                            new Posix_error(posix_error_code,
                                                            "pthread_mutex_unlock",
                                                            0)));

  return_value = true;

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (created_thread)
  {
    // let worker thread finish, if possible
    // (this method is "low tech" and might cause a delay in thread finishing since the worker thread might
    // have to timeout after waiting, but it is simple)
    __sync_fetch_and_or(&m_write_loop_finish, 1);
  }
  if (have_attr)
  {
    pthread_attr_destroy(&attr); // ignore error code
  }
  if (have_input_mutex)
  {
    // PORTING: might have to modify this call using an implementation of pthreads other than LinuxThreads,
    // PORTING: which currently does nothing except check for the mutex being locked
    pthread_mutex_destroy(&m_input_mutex); // ignore error code
  }
  if (have_input_cond)
  {
    // PORTING: might have to modify this call using an implementation of pthreads other than LinuxThreads,
    // PORTING: which currently does nothing except check for the condition having waiting threads
    pthread_cond_destroy(&m_input_cond); // ignore error code
  }
  if (file_open)
  {
    sf_close(m_output_file);
    m_output_file = 0;
    m_output_file_path = "";
  }
  if (thread_mutex_locked)
  {
    pthread_mutex_unlock(&m_thread_mutex);
  }
  if (have_thread_mutex)
  {
    pthread_mutex_destroy(&m_thread_mutex); // ignore error code
  }
  if (have_thread_cond)
  {
    pthread_cond_destroy(&m_thread_cond); // ignore error code
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(((return_value &&
                  // function success
                  (m_output_file
                   && running()))
                 || (!return_value
                     // function failure
                     && (!return_error.need_error() || return_error())
                     && (!m_output_file
                         && !running())))
                && mf_invariant());
  return return_value;
}

////////////////////
// callback write //
////////////////////

// This function should be minimal and efficient, and it should not
// have heap allocations, since it is to be called from the audio
// callback.

bool Async_audio_file_writer::write(int number_frames,
                                    const void* input_buffer,
                                    bool& return_have_overflow)
{
  precondition((running() || have_write_error())
               && (number_frames <= m_max_frames)
               && input_buffer
               && mf_invariant());

  bool return_value = false;

  start_error_block();
  static_cast<void>(handler_error); // avoid set but not used warning

  // see if there is already a write loop error or write error code
  bool have_write_error = __sync_fetch_and_add(&m_have_write_error, 0x0);
  on_error_with_label(have_write_error, write_error_handler, 0);

  const int frame_count = number_frames;

  Long_int avail = m_ring_buffer->write_available();
  Long_int buffer_byte_count = frame_count * m_recorder_config->sound_file->channels * m_sample_size;
  if (avail < buffer_byte_count)
  { // overflow!
    // just in case, signal writer loop that data is available
    // don't lock mutex, just signal (since we are in a time critical function)
    pthread_cond_signal(&m_input_cond); // never returns error code
    if (m_overflow_logging_enabled)
    {
      //  Log time and frame for overflow for debugging:
      time_t overflow_time = time(0);
      cerr << "Soft overflow at time " << asctime(localtime(&overflow_time)); // (check_code_ignore)
      cerr << " frame " << frames_written_count() << endl;
    }
    on_error_with_label(true, overflow_error_handler, 0);
  }

  // write start marker
  m_ring_buffer->write_data(reinterpret_cast<const void*>(&ic_start_marker), sizeof(uint32_t));
  // write frame count
  m_ring_buffer->write_data(reinterpret_cast<const void*>(&frame_count), sizeof(int));
  // write data frames from buffer
  m_ring_buffer->write_data(input_buffer, buffer_byte_count);
  // write end marker
  m_ring_buffer->write_data(reinterpret_cast<const void*>(&ic_end_marker), sizeof(uint32_t));

  if (m_clear_frames_written_count)
  {
    m_clear_frames_written_count = false;
    __sync_and_and_fetch(&m_frames_written_count, 0x0);
  }
  __sync_add_and_fetch(&m_frames_written_count, frame_count);

  // don't lock mutex, just signal (since we are in a time critical function)
  pthread_cond_signal(&m_input_cond); // never returns error code

  return_value = true;

  goto exit_point;

  end_error_block();

 overflow_error_handler:
  return_have_overflow = true;
  goto error_cleanup;

 write_error_handler:
  // the write error can be retrieved with write_error() function
  goto error_cleanup;

 error_cleanup:
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(mf_invariant());

  return return_value;
}

bool Async_audio_file_writer::pause(Error_param& return_error)
{
  precondition(running()
               && !return_error()
               && mf_invariant());

  static_cast<void>(return_error); // avoid unused warning

  pthread_cond_signal(&m_input_cond); // never returns error code

  postcondition(!return_error()
                && mf_invariant());

  return true;
}

bool Async_audio_file_writer::stop(Error_param& return_error)
{
  precondition((running() || have_write_error())
               && !return_error()
               && mf_invariant());

  bool return_value = false;

  bool have_lock = false;
  bool input_cond_destroyed = false;
  bool input_mutex_destroyed = false;
  bool thread_cond_destroyed = false;
  bool thread_mutex_destroyed = false;
  bool closed_output_file = false;

  start_error_block();

  // stop worker thread
  int posix_error_code;

  posix_error_code = pthread_mutex_lock(&m_input_mutex);
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::stop,
                                            new Posix_error(posix_error_code,
                                                            "pthread_mutex_lock",
                                                            0)));
  have_lock = true;

  __sync_or_and_fetch(&m_write_loop_finish, 1);
  // signal worker thread, in case it is waiting
  pthread_cond_signal(&m_input_cond); // never returns error code

  posix_error_code = pthread_mutex_unlock(&m_input_mutex);
  have_lock = false;
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::stop,
                                            new Posix_error(posix_error_code,
                                                            "pthread_mutex_unlock",
                                                            0)));

  // wait for worker thread to stop
  posix_error_code = pthread_join(m_writer_thread, 0);
  on_error(posix_error_code,  new AAFW_error(error_location(),
                                              AAFW_error::stop,
                                              new Posix_error(posix_error_code,
                                                              "pthread_join",
                                                              0)));

  posix_error_code = pthread_cond_destroy(&m_input_cond);
  input_cond_destroyed = true;
  on_error(posix_error_code, new AAFW_error(error_location(), AAFW_error::stop,
                                            new Posix_error(posix_error_code,
                                                            "pthread_cond_destroy",
                                                            0)));

  posix_error_code = pthread_mutex_destroy(&m_input_mutex);
  input_mutex_destroyed = true;
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::stop,
                                            new Posix_error(posix_error_code,
                                                            "pthread_mutex_destroy",
                                                            0)));

  posix_error_code = pthread_cond_destroy(&m_thread_cond);
  thread_cond_destroyed = true;
  on_error(posix_error_code, new AAFW_error(error_location(), AAFW_error::stop,
                                            new Posix_error(posix_error_code,
                                                            "pthread_cond_destroy",
                                                            0)));

  posix_error_code = pthread_mutex_destroy(&m_thread_mutex);
  thread_mutex_destroyed = true;
  on_error(posix_error_code, new AAFW_error(error_location(),
                                            AAFW_error::stop,
                                            new Posix_error(posix_error_code,
                                                            "pthread_mutex_destroy",
                                                            0)));

  int close_ret = sf_close(m_output_file);
  m_output_file = 0;
  closed_output_file = true;
  on_error(close_ret, new AAFW_error(error_location(),
                                     AAFW_error::stop,
                                     new Sndfile_error(close_ret,
                                                       "sf_close",
                                                       0)));

  return_value = true;

  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (have_lock)
  {
    pthread_mutex_unlock(&m_input_mutex); // ignore return
  }
  if (!input_cond_destroyed)
  {
    pthread_cond_destroy(&m_input_cond); // ignore return
  }
  if (!input_mutex_destroyed)
  {
    pthread_mutex_destroy(&m_input_mutex); // ignore return
  }
  if (!thread_cond_destroyed)
  {
    pthread_cond_destroy(&m_thread_cond); // ignore return
  }
  if (!thread_mutex_destroyed)
  {
    pthread_mutex_destroy(&m_thread_mutex); // ignore return
  }
  if (!closed_output_file)
  {
    sf_close(m_output_file); // ignore return
    m_output_file = 0;
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(!m_output_file
                && !running()
                && (return_value || !return_error.need_error() || return_error())
                && mf_invariant());

  return return_value;
}

//
// Protected member functions
//

// invariant check
bool Async_audio_file_writer::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (!m_recorder_config)
  {
    goto exit_point;
  }
  else if ((m_max_frames < 0)
           || (m_write_chunk_size < 0)
           || (m_write_buffer_size < 0)
           || (m_sample_size < 0))
  {
    goto exit_point;
  }
  else if (running()
           && (!m_output_file
               || !m_ring_buffer
               || (0 == m_max_frames)
               || (0 == m_write_chunk_size)
               || (0 == m_write_buffer_size)
               || (0 == m_sample_size)))
  {
    goto exit_point;
  }
  else if (!running()
           && !have_write_error()
           && m_output_file)
  {
    goto exit_point;
  }
  else if ((0 == m_write_loop_error)
           && (AAFW_error::none != m_write_error_code))
  {
    goto exit_point;
  }
  else if (__sync_fetch_and_add(const_cast<int32_t *>(&m_have_write_error), 0x0)
           && (!m_write_loop_error && (AAFW_error::none == m_write_error_code)))
  {
    goto exit_point;
  }
  else
  {
    return_value = true;
  }

 exit_point:
  return return_value;
}
