#!/usr/bin/perl -w
# $Id: kickproc.pl,v 1.8 2006/06/04 05:02:11 df Exp $
#
# Copyright (C) 2005,2006 Dmitry Fedorov <dm.fedorov@gmail.com>
#
# This file is part of kickproc.
#
# kickproc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# kickproc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA


=head1 NAME

 kickproc - performs actions on processes matched conditions


=head1 SYNOPSIS

 kickproc <conf-file>

=head1 DESCRIPTION

This utility usually runned as cron task.
On each call it builds list of processes and matches each of processes
against conditions list readed from configuration file.
If process matched conditions, specified action performed.
Action is shell command. Usually it is L<renice(8)> or L<kill(1)>.
For security reasons it is preferred to run kickproc from separate
non-root user and use L<super(1)> or L<sudo(8)> to perform actions.

Usually kickproc used by system admin to renice long running
user computing tasks and to kill suspended/deadlocked applications like
Midnight Commander, Acroread, Mozilla.


=head1 REQUIREMENTS

 any Unix flavor

 Perl >= 5.004

 Proc::ProcessTable
	module from any CPAN mirror:

 http://cpan.org/modules/by-module/Proc/Proc-ProcessTable-0.40.tar.gz


=head1 CONFIGURATION FILE FORMAT

I'm too lazy to describe it completely.
Look at $PREFIX/etc/kickproc.conf.example.

I<Global options:>

	global.minuid=136	# to filter out system processes
	global.maxuid=65533	#
	global.logfile=/var/log/kickproc.log

I<Per-process conditions:>

Each of lines consists free form conditions list in form I<var=value> and
'action:' field terminates line.
If process matches against any of conditions rest of line after 'action:'
executed as shell command. '$$' in action field replaced by
matched process pid.
'exec' field is perl regular expression.
Order of condition lines is significant: a process matched lines in top-down
order and scanning lines finished if condition matched.
Consequently, most common condition lines should be placed at bottom.


=head1 CAVEATS

Currently condition lines syntax does not allows adding new process fields
without adding new lines to internal table in this script.
It is only due to my laziness to write proper parser.


=head1 SEE ALSO

L<Proc::ProcessTable(3)>
L<Proc::ProcessTable::Process(3)>


=head1 AUTHOR

Dmitry Fedorov <dm.fedorov@gmail.com>


=head1 COPYRIGHT

Copyright (C) 2005,2006 Dmitry Fedorov <dm.fedorov@gmail.com>


=head1 LICENSE

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License,
or (at your option) any later version.

=head1 DISCLAIMER

The author disclaims any responsibility for any mangling of your system
etc, that this script may cause.

=cut


require 5.004;
use strict;
local $^W=1; # use warnings only since 5.006
use File::Spec qw(splitpath);

use Proc::ProcessTable;


my $progname = 'kickproc';

@ARGV == 1 or die "Usage: $progname conf-file\n";

my $cfgname = shift;
defined $cfgname or die "No conf-file parameter supplied\n";

my $t = new Proc::ProcessTable;



sub match($$)
{
    my $val = shift;
    my $lim = shift;

    $val = (split( /\s+/, $val ))[0];
    my ( $undef, $dir, $file ) = File::Spec->splitpath($val);
    $val=$file;

    my $rex = qr/$lim/i;
    my $rc = ($val =~ m/$rex/i);
    $rc = 0 if $rc eq '';

    return $rc;
}

sub gt($$)
{
    my $val = shift;
    my $lim = shift;

    return $val > $lim;
}

sub lt($$)
{
    my $val = shift;
    my $lim = shift;

    return $val < $lim;
}

sub gtM($$)	# ge() in MBytes
{
    my $val = shift;
    my $lim = shift;

    $lim *= (1024*1024);

    return $val > $lim;
}

# accepts: [2d][1h][30m][25[s]]
# returns: seconds
sub ctime($)
{
    my $val = shift;

    $val =~ m/(\d+d)?(\d+h)?(\d+m)?(\d+s?)?/;
    my $days    = $1; $days    =~ s/(\d+)d/$1/  if defined $days   ;
    my $hours   = $2; $hours   =~ s/(\d+)h/$1/  if defined $hours  ;
    my $minutes = $3; $minutes =~ s/(\d+)m/$1/  if defined $minutes;
    my $seconds = $4; $seconds =~ s/(\d+)s?/$1/ if defined $seconds;

    my $ret = 0;

    $ret += $days    * 86400 if defined $days;
    $ret += $hours   *  3600 if defined $hours;
    $ret += $minutes *    60 if defined $minutes;
    $ret += $seconds         if defined $seconds;

    return $ret;
}


sub atime($$)   # calendar time of process run in seconds
{
    my $val = shift;
    my $lim = shift;

    $val = time() - $val;

    my $limit = ctime($lim);

    return $val > $limit;
}

sub gtCpu($$)   # cpu time in milliseconds
{
    my $val = shift;
    my $lim = shift;

    $val /= 1000;

    my $limit = ctime($lim);

    return $val > $limit;
}

my $logname;
sub logfile($$)   # global option
{
    shift;
    $logname = shift;
    return 0;
}

my $minuid = 0;
sub minuid($$)   # global option
{
    shift;
    $minuid = shift;
    return 0;
}

my $maxuid = 65536;
sub maxuid($$)   # global option
{
    shift;
    $maxuid = shift;
    return 0;
}


my %keys =
    (
     "\Lexec"        => { 'sub' => \&match  , 'pname' => 'cmndline' },
     "\LMaxSize"     => { 'sub' => \&gt     , 'pname' => 'size'     },
     "\LMaxSizeM"    => { 'sub' => \&gtM    , 'pname' => 'size'     },
     "\LMax%mem"     => { 'sub' => \&gt     , 'pname' => 'pctmem'   },
     "\LMax%cpu"     => { 'sub' => \&gt     , 'pname' => 'pctcpu'   },
     "\LMinNice"     => { 'sub' => \&lt     , 'pname' => 'priority' },
     "\LMinPrio"     => { 'sub' => \&lt     , 'pname' => 'priority' },
     "\LMaxTime"     => { 'sub' => \&atime  , 'pname' => 'start'    },
     "\LMaxCpuTime"  => { 'sub' => \&gtCpu  , 'pname' => 'time'     },
     "global.minuid" => { 'sub' => \&minuid , 'pname' => 'uid'      },
     "global.maxuid" => { 'sub' => \&maxuid , 'pname' => 'uid'      },
     "global.logfile"=> { 'sub' => \&logfile, 'pname' => ''         },
    );

my %vals = ();

my @cfglines; #    ( { key => val, ... }, ... );

open(CFG, "< $cfgname") or die "Can't open $cfgname: $!\n";
my $line=0;

while(<CFG>)
{
    $line++;
    s/^([^#]*)#?.*$/$1/; # skip comments
    s/^\s*(.*)\s*$/$1/;  # strip white spaces
    next unless length;  # skip empty lines

    my @string = split /action:/;
    my @list   = split (/\s+/, $string [0]);
    shift (@string);
    my $action = join ("action:", @string);

    my %conditions;

    for my $l (@list)
    {
        my ($key, $val) = split('=', $l);
        #print "+++$key, $val\n";
        #$l =~ m/(.+)(>|=|<)(.+)/
        #    or die "Unrecognized expression $l at line $line\n";
        #my $key = $1; my $op = $2; my $val = $3;

        defined $key or die "Undefined keyword in condition $l at line $line\n";
        defined $val or die "Undefined value for key $key at line $line\n";
        #defined $op  or die "Undefined comparison for key $key at line $line\n";

        exists $keys{lc $key} or die "Unknown keyword: $key at line $line\n";

        exists $conditions{lc $key} and die "Duplicate key $key at line $line\n";

        $conditions{lc $key} = $val;
    }

    $conditions{'action'} = $action;
    push @cfglines, \%conditions;
}

close("CFG");


# global condtitions
for my $conditions (@cfglines)
{
    for my $c (keys %$conditions)
    {
        $c = lc($c);

        next unless $c =~ m/^global\./i;

        $keys{$c}{'sub'} ( undef, $conditions->{$c} );
    }
}

open (LOG, ">> $logname") or die "Can't open log file $logname: $!\n";
#print LOG "\n".`date`;


PROC:
for my $p ( @{$t->table} )
{
    next if $p->{'uid'} < $minuid;
    next if $p->{'uid'} > $maxuid;
    #print "pid: $p->{pid}, cmd: $p->{cmndline}, size: $p->{size}, rss: $p->{rss}\n";
    COND:
    for my $conditions (@cfglines)
    {
        my @whatsup;

	for my $c (keys %$conditions)
        {
            $c = lc($c);

            next if $c eq 'action';

	    @whatsup = (@whatsup, join ('=', $c, $conditions->{$c}));

            unless ( $keys{$c}{'sub'} (
                                       $p->{$keys{$c}{'pname'}},
                                       $conditions->{$c}
                                      )
                   )
            {
                next COND;
            }
        }

        my $action = $conditions->{'action'};
        $action =~ s/\$\$/$p->{pid}/g;

        my $user = (getpwuid($p->{uid}))[0];
        printf LOG
            "PID: %5d, User: %5s, exec: %-20s, matched: %-20s",
	    $p->{pid}, $user,
            (split( /\s+/, $p->{cmndline} ))[0],
            join (', ', @whatsup);
        print LOG "\taction: $action" if $action ne "";
        print LOG "\n";
        system($action);
        next PROC;
    }
}


__END__
uid         UID of process
gid         GID of process
euid        effective UID of process           (Solaris only)
egid        effective GID of process           (Solaris only)
pid         process ID
ppid        parent process ID
spid        sprod ID                           (IRIX only)
pgrp        process group
sess        session ID
cpuid       CPU ID of processor running on     (IRIX only)
priority    priority of process
ttynum      tty number of process
flags       flags of process
minflt      minor page faults                  (Linux only)
cminflt     child minor page faults            (Linux only)
majflt      major page faults                  (Linux only)
cmajflt     child major page faults            (Linux only)
utime       user mode time (1/100s of seconds) (Linux only)
stime       kernel mode time                   (Linux only)
cutime      child utime                        (Linux only)
cstime      child stime                        (Linux only)
time        user + system time
ctime       child user + system time
timensec    user + system nanoseconds part     (Solaris only)
ctimensec   child user + system nanoseconds    (Solaris only)
qtime       cumulative cpu time                (IRIX only)
size        virtual memory size (bytes)
rss         resident set size (bytes)
wchan       address of current system call
fname       file name
start       start time (seconds since the epoch)
pctcpu      percent cpu used since process started
state       state of process
pctmem      percent memory
cmndline    full command line of process
ttydev      path of process's tty
clname      scheduling class name              (IRIX only)
