/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2007-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "sallyeuchreai.h"
#include "rulebase.h"
#include "cardproperties.h"
#include "../../games/euchre/euchrerules.h"

SallyEuchreAI::SallyEuchreAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand): basicStrategies(rules)
{
    m_playSequence = playSequence;
    m_pRules = &rules;
    m_hand = hand;
}

SallyEuchreAI::~SallyEuchreAI()
{}

int SallyEuchreAI::turnUp(const Card &turnUpCard, int score, int opponentScore) const
{
    // If we have more than 2 high cards take it, but not if the score is 0-0
    if ((score > 0) && (opponentScore > 0))
    {
        if ((score == 4) && (opponentScore > 0)) // Prevent opponent's from winning by marching
            return EuchreRules::WANT_TURN_UP;
        if (basicStrategies.numberOfCardsOfSuit(m_hand, turnUpCard.suit()) >= 3)
            return EuchreRules::WANT_TURN_UP;
    }
    return EuchreRules::DOES_NOT_DECIDE;
}

int SallyEuchreAI::selectTrump(const Card &turnUpCard, int score, int opponentScore) const
{
    // If the score is not 0-0, then we'll propose a suit
    if ((score > 0) && (opponentScore > 0))
    {
        for (int suit = 0; suit < 4; ++suit)
            if (static_cast<Card::Suit>(suit) != turnUpCard.suit())
                if (basicStrategies.numberOfCardsOfSuit(m_hand, static_cast<Card::Suit>(suit)) >= 3)
                    return suit;
    }
    return Card::SUIT_ERR;
}

int SallyEuchreAI::playAlone(int score, int opponentScore) const
{
    if ((score > 0) && (opponentScore == 4)) // Try to win with a march
        return EuchreRules::PLAY_ALONE;
    if (basicStrategies.numberOfCardsHigher(m_hand, Card::QUEEN) >= 4)
        return EuchreRules::PLAY_ALONE;
    return EuchreRules::DOES_NOT_DECIDE;
}

int SallyEuchreAI::defendAlone() const
{
    if (basicStrategies.numberOfCardsHigher(m_hand, Card::QUEEN) >= 4)
        return EuchreRules::DEFEND_ALONE;
    return EuchreRules::DOES_NOT_DECIDE;
}

int SallyEuchreAI::wantTurnUp() const
{
    // The "m_playSequence" should actually be the card from the deck
    if (m_pRules->rankValue(m_playSequence.front()) > Card::QUEEN)
        return EuchreRules::WANT_TURN_UP_CARD;
    return EuchreRules::DOES_NOT_DECIDE;
}

CardSequence SallyEuchreAI::selectCards(const Card &partnersCard) const
{
    bool selectedCard = false;
    CardSequence selection, testSelection;
    CardProperties handProperties(m_hand);
    CardProperties playSequenceProperties(m_playSequence);
    Card::Suit trumpSuit = static_cast<const EuchreRules *>(m_pRules)->trumpSuit();
    Card::Suit leadSuit = m_playSequence.front().suit();
    Card::Suit leftBowerSuit = static_cast<const EuchreRules *>(m_pRules)->leftBowerSuit();
    Card leftBower = Card(Card::JACK, leftBowerSuit);
    Card highestCardPlayed;

    // Set the highest card played if a card has been played.
    if (! m_playSequence.isEmpty())
        highestCardPlayed = basicStrategies.highestCardOfSuit(m_playSequence, leadSuit);
    // If the lead suit is the Left Bower, then the lead suit is actually trump.
    if (m_playSequence.front() == leftBower)
        leadSuit = trumpSuit;

    // If leading, then lead with the highest card in our hand
    if (m_playSequence.isEmpty())
    {
        selection = basicStrategies.highestCard(m_hand);
        selectedCard = true;
    }

    // Make sure not to play over partner if partner leads, then play lowest
    if ((! selectedCard))
    {
        // Play the lowest legal card
        if ((! partnersCard.isEmpty()) && (highestCardPlayed == partnersCard))
        {
            selectedCard = true;
            selection = basicStrategies.lowestCardOfSuit(m_hand, leadSuit);
            if (selection.isEmpty())
            {
                // If trump was lead we have to check to see if we have the left-bower.
                if (m_hand.hasCard(leftBower))
                {
                    selection.clear();
                    selection.addCard(leftBower);
                }
                else
                    selection = basicStrategies.lowestCard(m_hand);
            }
        }
    }

    // If lead suit is trump try to beat the highest trump, else play the lowest
    if ((! selectedCard) && (leadSuit == trumpSuit))
    {
        testSelection = basicStrategies.highestCardOfSuit(m_hand, leadSuit); // Suit is trump

        // If our highest card it's not the right bower, see if we have the left bower
        if (m_hand.hasCard(leftBower)) // Do we have the left bower
            if (((! testSelection.isEmpty()) && (testSelection.front().rank() != Card::JACK)) || testSelection.isEmpty())
            {
                testSelection.removeCard(testSelection.front());
                testSelection.addCard(leftBower);
            }
        if ((! testSelection.isEmpty()) && (m_pRules->rankValue(testSelection.front()) > m_pRules->rankValue(highestCardPlayed))) // Make sure we'll beat the highest card
        {
            selection = testSelection;
            selectedCard = true;
        }
        else // Play our lowest trump card
        {
            testSelection = basicStrategies.lowestCardOfSuit(m_hand, leadSuit);
            if (! testSelection.isEmpty())
            {
                selection = testSelection;
                selectedCard = true;
            }
            else if (m_hand.hasCard(leftBower)) // Do we have the left bower
            {
                selection.addCard(leftBower);
                selectedCard = true;
            }
        }
    }

    // If trump hasn't been played, then play the highest card of the lead suit
    if ((! selectedCard) && playSequenceProperties.suits(trumpSuit).isEmpty())
    {
        testSelection = basicStrategies.highestCardOfSuit(m_hand, leadSuit);

        if (testSelection.isEmpty()) // If don't have a card of the lead suit, then play a trump
        {
            testSelection = basicStrategies.highestCardOfSuit(m_hand, trumpSuit);
            if (! testSelection.isEmpty())
            {
                selection = testSelection;
                selectedCard = true;
            }
        }
        else if (m_pRules->rankValue(testSelection.front()) > m_pRules->rankValue(highestCardPlayed)) // Make sure we'll beat the highest card
        {
            selection = testSelection;
            selectedCard = true;
        }
        else // Play our lowest card
        {
            selection = basicStrategies.lowestCardOfSuit(m_hand, leadSuit);
            selectedCard = true;
        }
    }
    else if (! selectedCard)// If trump has been played, then play a low card of the lead suit
    {
        testSelection = basicStrategies.lowestCardOfSuit(m_hand, leadSuit);
        // If don't have a card of the lead suit, then play a higher trump
        if (testSelection.isEmpty())
        {
            testSelection = basicStrategies.highestCardOfSuit(m_hand, trumpSuit);
            Card highestTrumpCard = basicStrategies.highestCardOfSuit(m_playSequence, trumpSuit);

            // If our highest card it's not the right bower, see if we have the left bower
            if (m_hand.hasCard(leftBower)) // Do we have the left bower
                if (((! testSelection.isEmpty()) && (testSelection.front().rank() != Card::JACK)) || testSelection.isEmpty())
                {
                    testSelection.removeCard(testSelection.front());
                    testSelection.addCard(leftBower);
                }
            if (! testSelection.isEmpty()) // Make sure we have a trump
            {
                // See if we have a higher trump
                if (m_pRules->rankValue(testSelection.front()) > m_pRules->rankValue(highestTrumpCard))
                {
                    selection = testSelection;
                    selectedCard = true;
                }
                else // Play the lowest non-trump card
                {
                    testSelection = basicStrategies.lowestNonTrumpCard(m_hand, trumpSuit);
                    if (! testSelection.isEmpty()) // Make sure we have a trump
                    {
                        selection = testSelection;
                        selectedCard = true;
                    }
                }
            }
        }
        else // Play the lowest card of the lead suit.
        {
            selection = testSelection;
            selectedCard = true;
        }
    }

    // Play the lowest card from a suit
    if (! selectedCard)
        selection = basicStrategies.lowestCard(m_hand);
    return selection;
}

CardSequence SallyEuchreAI::discard() const
{
    return basicStrategies.lowestNonTrumpCard(m_hand, static_cast<const EuchreRules *>(m_pRules)->trumpSuit());
}

